/* NickServ functions.
 *
 * Auspice Services is copyright (c) 2000-2001 In Mean
 *  http://www.auspice.org
 * Originally based on SirvNET Services(c) by Trevor Klingbeil.
 * This program is free but copyrighted software; see the file LICENSE for
 * details.
 */

#include "../inc/services.h"
#include "../inc/timeout.h"

static NickInfo *nicklists[256];	/* One for each initial character */

/* For local timeout use: */
#define TO_COLLIDE	0	/* Collide the user with this nick */
#define TO_RELEASE	1	/* Release a collided nick */
#define TO_COLLIDE_TL   2       /* Time left warning to collide */
#define TO_ENFORCE      3       /* Enforce a nick to guest* */

static void alpha_insert_nick(NickInfo *ni);
static NickInfo *makenick(const char *nick);
NickInfo * host(NickInfo * ni);
NickInfo * slave(const char *nick, int num);
int issibling (NickInfo * ni, const char *target);
int slavecount (const char *nick);
static void do_slaves(const char *source);
int userisnick (const char *nick);
long getflags (NickInfo * ni);
static void do_slavedrop (const char *source);

static int delnick(NickInfo *ni);
static void do_hold(const char *source);
static void collide(NickInfo *ni, int from_timeout);
static void release(NickInfo *ni, int from_timeout);
static void add_ns_timeout(NickInfo *ni, int type, time_t delay);
static void del_ns_timeout(NickInfo *ni, int type);

static void update_last_seen_time(Timeout *to);
static void do_set_sex(NickInfo * ni, char *param);
static void do_set_malay(NickInfo *ni, char *param);
static void do_set_auth(NickInfo *ni, char *param);
static void do_set_privmsg(NickInfo *ni, char *param);
static void do_ohelp(const char *source);
static void do_help(const char *source);
static void do_register(const char *source);
static void do_identify(const char *source);
static void do_delete(const char *source);
static void do_drop(const char *source);
static void do_set(const char *source);
static void do_set_password(NickInfo *ni, char *param);
static void do_set_url(NickInfo *ni, char *param);
static void do_set_email(NickInfo *ni, char *param);
static void do_set_uin (NickInfo *ni, char *param);
static void do_set_age (NickInfo * ni, char *param);
static void do_set_info (NickInfo * ni, char *param);
static void do_set_kill(NickInfo *ni, char *param);
static void do_set_secure(NickInfo *ni, char *param);
static void do_set_private(NickInfo *ni, char *param);
static void do_set_emailmemos(NickInfo *ni, char *param);
static void do_set_neverop(NickInfo *ni, char *param);
static void do_set_noop(NickInfo *ni, char *param);
static void do_set_nomemo(NickInfo *ni, char *param);
static void do_set_nohost(NickInfo *ni, char *param);
static void do_access(const char *source);
static void do_ajoin(const char *source);
static void do_comment(const char *source);
static void do_noteit(const char *source);
static void do_link(const char *source);
static void do_info(const char *source);
static void do_acc(const char *source);
static void do_flist(const char *source);
static void do_list(const char *source);
static void do_recover(const char *source);
static void do_release(const char *source);
static void do_ghost(const char *source);
static void do_getpass(const char *source);
#ifndef CYGWIN
static void do_sendpass(const char *source);
#endif
static void do_forbid(const char *source);
static void do_mark(const char *source);
static char *get_os_access(NickInfo *ni);
static void do_setpass(const char *source);
static void do_lock(const char *source);
static void do_set_hidemail(NickInfo *ni, char *param);
static void do_setflags(const char *source);
static void do_set_ircop(NickInfo *ni, char *param);
static void do_set_botmsg(NickInfo *ni, char *param);
static void do_setmlock(const char *source);
static void do_freeze(const char *source);
static void do_search(const char *source);
static void swhois(const char *source, NickInfo *ni);
void do_niautojoin(NickInfo *ni, const char *source);

static void do_set_mlock(NickInfo * ni, char *param);
static void do_set_autojoin(NickInfo *ni, char *param);
#ifdef MERGE
void load_nsmerge_dbase(void);
#endif
/*************************************************************************/

/* Display total number of registered nicks and info about each; or, if
 * a specific nick is given, display information about that nick (like
 * /msg NickServ INFO <nick>).  If count_only != 0, then only display the
 * number of registered nicks (the nick parameter is ignored).
 */

void listnicks(int count_only, const char *nick)
{
    long count = 0;
    NickInfo *ni;
    int i;

    if (count_only) {

	for (i = 33; i < 256; ++i) {
	    for (ni = nicklists[i]; ni; ni = ni->next)
		++count;
	}
	printf("%li nicknames registered.\n", count);

    } else if (nick) {

	struct tm tm;
	char buf[512];

	if (!(ni = findnick(nick))) {
	    printf("%s not registered.\n", nick);
	    return;
	} else if (ni->flags & NI_VERBOTEN) {
	    printf("%s is FORBIDden.\n", nick);
	    return;
	}
	printf("%s is %s\n", nick, ni->last_realname);
	printf("Last seen address: %s\n", ni->last_usermask);
	tm = *localtime(&ni->time_registered);

	printf("  Time registered: %s %2d %02d:%02d:%02d %d\n",
			month_name(tm.tm_mon+1), tm.tm_mday, tm.tm_hour,
			tm.tm_min, tm.tm_sec, tm.tm_year);
	tm = *localtime(&ni->last_seen);
	printf("   Last seen time: %s %2d %02d:%02d:%02d %d\n",
			month_name(tm.tm_mon+1), tm.tm_mday, tm.tm_hour,
			tm.tm_min, tm.tm_sec, tm.tm_year);
	if (ni->url)
	    printf("              URL: %s\n", ni->url);
        if (ni->email && (ni->flags & NI_HIDE_EMAIL))
            printf("   E-mail address: Hidden\n");
        else
	    printf("   E-mail address: %s\n", ni->email);
	if (ni->uin)
	    printf("              UIN: %s\n", ni->uin);
	*buf = 0;
	if (ni->flags & NI_KILLPROTECT)
	    strcat(buf, "Kill protection");
	if (ni->flags & NI_SECURE) {
	    if (*buf)
		strcat(buf, ", ");
	    strcat(buf, "Security");
	}
	if (ni->flags & NI_PRIVATE) {
	    if (*buf)
		strcat(buf, ", ");
	    strcat(buf, "Private");
	}
        if (ni->flags & NI_MARK) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "Marked");
        }
        if (ni->flags & NI_HOLD) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "Held");
        }
	if (!*buf)
	    strcpy(buf, "None");
	printf("          Options: %s\n", buf);

    } else {

	for (i = 33; i < 256; ++i) {
	    for (ni = nicklists[i]; ni; ni = ni->next) {
		printf("    %-20s  %s\n", ni->nick, ni->last_usermask);
		++count;
	    }
	}
	printf("%li nicknames registered.\n", count);

    }
}
/*************************************************************************/
#ifndef CYGWIN
void listmemos(const char *nick, const char *pass)
{
    NickInfo *ni;
    if (nick && pass) {
        if (!(ni = findnick(nick))) {
            printf("%s is not a registered nick.\n", nick);
            return;
        } else if (ni->flags & NI_VERBOTEN) {
            printf("%s is FORBIDden.\n", nick);
            return;
        }
	if (stricmp(ni->pass, pass) == 0) {
		do_webread(nick);
	} else {
		printf("Incorrect password\n");
  	}
   } else {
		printf("Nickname and password required.\n");
   }
}
#endif
/*************************************************************************/

NickInfo *retnick(int i)
{
    return nicklists[i];
}

/* Return information on memory use.  Assumes pointers are valid. */

void get_nickserv_stats(long *nrec, long *memuse)
{
    long count = 0, mem = 0;
    int i, j;
    NickInfo *ni;
    char **accptr;


    for (i = 0; i < 256; i++) {
	for (ni = nicklists[i]; ni; ni = ni->next) {
	    count++;
	    mem += sizeof(*ni);
	    if (ni->last_usermask)
		mem += strlen(ni->last_usermask)+1;
	    if (ni->last_realname)
		mem += strlen(ni->last_realname)+1;
            if (ni->url)
                mem += strlen(ni->url)+1;
            if (ni->email)
                mem += strlen(ni->email)+1;
            if (ni->uin)
                mem += strlen(ni->uin)+1;
            if (ni->forward)
                mem += strlen(ni->forward)+1;
            if (ni->mark)
                mem += strlen(ni->mark)+1;
            if (ni->hold)
                mem += strlen(ni->hold)+1;
            if (ni->forbid)
                mem += strlen(ni->forbid)+1;
            if (ni->age)
                mem += strlen(ni->age)+1;
            if (ni->info)
                mem += strlen(ni->info)+1;
            if (ni->sex)
                mem += strlen(ni->sex)+1;
            if (ni->mlock)
                mem += strlen(ni->mlock)+1;
            if (ni->last_quit)
                mem += strlen(ni->last_quit)+1;

	    mem += sizeof(char *) * ni->accesscount;
	    for (accptr=ni->access, j=0; j < ni->accesscount; accptr++, j++) {
		if (*accptr)
		    mem += strlen(*accptr)+1;
	    }
	    mem += sizeof(char *) * ni->ajoincount;
	    for (accptr=ni->autojoin, j=0; j < ni->ajoincount; accptr++, j++) {
		if (*accptr)
		    mem += strlen(*accptr)+1;
	    }

	    mem += sizeof(char *) * ni->comline;
	    for (accptr=ni->comment, j=0; j < ni->comline; accptr++, j++) {
		if (*accptr)
		    mem += strlen(*accptr)+1;
	    }

	    mem += sizeof(char *) * ni->noteline;
	    for (accptr=ni->note, j=0; j < ni->noteline; accptr++, j++) {
		if (*accptr)
		    mem += strlen(*accptr)+1;
	    }
	}
    }
    *nrec = count;
    *memuse = mem;
}
/*************************************************************************/
/* Main NickServ routine. */
void nickserv(const char *source, char *buf)
{
    const char *cmd = strtok(buf, " ");
    char *s;
    NickInfo *ni = findnick(source);

    if (!cmd) {
	return;

    } else if (stricmp(cmd, "\1PING") == 0) {
	if (!(s = strtok(NULL, "")))
	    s = "\1";
	notice(s_NickServ, source, "\1PING %s", s);

    } else if (stricmp (cmd, "\1VERSION\1") == 0) {
    
        if (!(s = strtok (NULL, "")))
            s = "\1";
        notice(s_NickServ, source, "%sVERSION Auspice IRC Services %s", s, s);

    } else if (stricmp(cmd, "HELP") == 0) {
	do_help(source);
    } else if (stricmp(cmd, "REGISTER") == 0) {
	do_register(source);
    } else if (stricmp(cmd, "AUTH") == 0) {
	do_auth(source);
    } else if (stricmp(cmd, "IDENTIFY") == 0) {
	do_identify(source);
    } else if (stricmp(cmd, "ID") == 0) {
        do_identify(source);
    } else if (stricmp(cmd, "LINK") == 0) {
	do_link(source);
#ifdef MERGE
    } else if (stricmp(cmd, "MERGE") == 0) {
        load_nsmerge_dbase();
#endif
    } else if (stricmp(cmd, "LINKS") == 0) {
	do_slaves(source);
    } else if (stricmp(cmd, "UNLINK") == 0) {
	do_slavedrop(source);
    } else if (stricmp(cmd, "DROP") == 0) {
	do_drop(source);
    } else if (stricmp(cmd, "SET") == 0) {
	do_set(source);
    } else if (stricmp(cmd, "GHOST") == 0) {
	do_ghost(source);
    } else if (stricmp(cmd, "ACCESS") == 0) {
	do_access(source);
    } else if (stricmp(cmd, "SEARCH") == 0) {
	do_search(source);
    } else if (stricmp(cmd, "AJOIN") == 0) {
	do_ajoin(source);
    } else if (stricmp(cmd, "COMMENT") == 0) {
	do_comment(source);
    } else if (stricmp(cmd, "NOTE") == 0) {
	do_noteit(source);
    } else if (stricmp(cmd, "INFO") == 0) {
	do_info(source);
    } else if (stricmp(cmd, "ACC") == 0) {
        do_acc(source);
    } else if (stricmp(cmd, "RECOVER") == 0) {
	do_recover(source);
    } else if (stricmp(cmd, "RELEASE") == 0) {
	do_release(source);
    } else if (stricmp(cmd, "SELFDEL") == 0) {
	if (!ni) {
		notice(s_NickServ, source, NS_ONLYREG);	
	} else if (!(ni->flags & (NI_RECOGNIZED | NI_IDENTIFIED))) {
	        notice(s_NickServ, source, NS_PLZID, s_NickServ);
	        notice(s_NickServ, source, NS_IDCOM, s_NickServ);
	} else {
		do_selfdel(source);
	}

    } else if (stricmp(cmd, "LISTCHANS") == 0) {

	if (!ni) {
		notice(s_NickServ, source, NS_ONLYREG);	
	} else if (!(ni->flags & (NI_RECOGNIZED | NI_IDENTIFIED))) {
	        notice(s_NickServ, source, NS_PLZID, s_NickServ);
	        notice(s_NickServ, source, NS_IDCOM, s_NickServ);
	} else {
		if (is_tokn(2)) {
			do_glist(source);
	         } else {
			if (rec_services_helpop(source)) {
				do_glist(source);
			} else {
				notice(s_NickServ, source, NS_DISABLED);
			} 
		}
	}

    } else if (stricmp(cmd, "AUTHCODE") == 0) {
	char *nick = strtok(NULL, " ");
	char *authcode = strtok(NULL, " ");

	if (!nick || !authcode) {
		notice(s_NickServ, source, "Syntax: /Msg NickServ AUTHCODE \2<Nick>\2 \2<Code>\2");
	} else {
		NickInfo *ani = findnick(nick);
	        User *u = finduser(nick);
		if (ani && (ani->flags & NI_WAUTH)) {
			if (stricmp(ani->last_realname, authcode)==0) {
			    if (is_oper(source) && !(ani->flags & NI_HOP))
			        notice(s_NickServ, source, NS_IRCOP);
				notice(s_NickServ, source, "Please type /Msg NickServ Identify %s", ani->pass);
				if (ani->last_realname);
					free(ani->last_realname);
			    	ani->last_realname = sstrdup(u->realname);
				ani->flags &= ~NI_WAUTH;
			} else {
				notice(s_NickServ, source, "\2incorrect code\2");
	        		return;
			}

		} else {
			notice(s_NickServ, source, "\2This nick is not a registered nick or already Authorized\2");
		}
	} /* End of !nick */

    } else if (!rec_services_helpop(source)) {
	notice(s_NickServ, source, NS_UNKNOWC, cmd, s_NickServ);
    } else if (stricmp(cmd, "OHELP") == 0) {
	do_ohelp(source);
    } else if (stricmp(cmd, "SETFLAG") == 0) {
         do_setflags(source);
    } else if (!is_services_oper(source)) {
	notice(s_NickServ, source, NS_UNKNOWC, cmd, s_NickServ);
#ifndef CYGWIN
    } else if (stricmp(cmd, "SENDPASS") == 0) {
        do_sendpass(source);
#endif
    } else if (stricmp(cmd, "FLIST") == 0) {
        do_flist(source);
    } else if (!is_services_admin(source)) {
	notice(s_NickServ, source, NS_UNKNOWC, cmd, s_NickServ);
    } else if (stricmp(cmd, "GETPASS") == 0) {
	do_getpass(source);
    } else if (stricmp(cmd, "MARK") == 0) {
	do_mark(source);
    } else if (stricmp(cmd, "FREEZE") == 0) {
	do_freeze(source);
    } else if (stricmp(cmd, "HOLD") == 0) {
	do_hold(source);
    } else if (!is_services_root(source)) {
         notice(s_NickServ,source,PMD);
    } else if (stricmp(cmd, "FORBID") == 0) {
         do_forbid(source);
    } else if (stricmp(cmd, "DELETE") == 0) {
        do_delete(source);
    } else if (stricmp(cmd, "SETPASS") == 0) {
        do_setpass(source);
    } else if (stricmp(cmd, "SETMLOCK") == 0) {
        do_setmlock(source);
    } else if (stricmp(cmd, "LIST") == 0) {
	do_list(source);
    } else if (!is_services_coder(source)) {
         notice(s_NickServ,source,PMD);
    } else if (stricmp(cmd, "LOCK") == 0) {
	do_lock(source);
    } else {
        User *u = finduser(source);
	notice(s_NickServ, source,
		NS_UNKNOWC,
		cmd, s_NickServ);

        slog("NS *C %s (%s@%s) [%s]",
            source, u->username, u->host, cmd);
    }

}

/*************************************************************************/

#ifdef MERGE
void load_nsmerge_dbase(void)
{
    FILE *f;
    int i,j;
    int ver;
    NickInfo *ni;

    if (!(f = open_db(s_NickServ, "nick2.db", "r", 1)))
        return;

    switch (ver = get_file_version(f, "nick2.db")) {
      case 6:
        for (i = 33; i < 256; ++i) {
            while (fgetc(f) == 1) {
                   ni = smalloc(sizeof(NickInfo));
                   if (1 != fread(ni, sizeof(NickInfo), 1, f))
                       fatal_perror("Read error on %s", "nick2.db");

                ni->flags &= ~(NI_IDENTIFIED | NI_RECOGNIZED);
                alpha_insert_nick(ni);
                if (ni->url)
                    ni->url = read_string(f, "nick2.db");
                if (ni->email)
                    ni->email = read_string(f, "nick2.db");
                if (ni->uin)
                    ni->uin = read_string(f, "nick2.db");
                if (ni->age)
                    ni->age = read_string(f, "nick2.db");
                if (ni->forward)
                    ni->forward = read_string(f, "nick2.db");
                if (ni->hold)
                    ni->hold = read_string(f, "nick2.db");
                if (ni->mark)
                    ni->mark = read_string(f, "nick2.db");
                if (ni->forbid)
                    ni->forbid = read_string(f, "nick2.db");
                if (ni->info)
                    ni->forbid = read_string(f, "nick2.db");
                if (ni->info)
                    ni->info = read_string(f, "nick2.db");
                if (ni->sex)
                    ni->sex = read_string(f, "nick2.db");
                if (ni->mlock)
                    ni->mlock = read_string(f, "nick2.db");
                else
                    ni->mlock = NULL;
                if (ni->last_quit)
                    ni->last_quit = read_string(f, "nick2.db");

                if (ni->last_usermask)
                        ni->last_usermask = read_string(f, "nick2.db");
                if (ni->last_usermask)
                        ni->last_realname = read_string(f, "nick2.db");

                if (ni->accesscount) {
                    char **access1;
                    access1 = smalloc(sizeof(char *) * ni->accesscount);
                    ni->access = access1;
                    for (j = 0; j < ni->accesscount; ++j, ++access1)
                        *access1 = read_string(f, "nick2.db");
                } else {
                          ni->access = NULL;
                    }

                if (ni->ajoincount) {
                    char **autojoin;
                    autojoin = smalloc(sizeof(char *) * ni->ajoincount);
                    ni->autojoin = autojoin;
                    for (j = 0; j < ni->ajoincount; ++j, ++autojoin)
                        *autojoin = read_string(f, "nick2.db");
                }

                if (ni->comline) {
                    char **nickcom;
                    nickcom = smalloc(sizeof(char *) * ni->comline);
                    ni->comment = nickcom;
                    for (j = 0; j < ni->comline; ++j, ++nickcom)
                        *nickcom = read_string(f, "nick2.db");
                }

                if (ni->noteline) {
                    char **nicknote;
                    nicknote = smalloc(sizeof(char *) * ni->noteline);
                    ni->note = nicknote;
                    for (j = 0; j < ni->noteline; ++j, ++nicknote)
                        *nicknote = read_string(f, "nick2.db");
                }
                ni->id_timestamp = 0;
            } /* while (fgetc(f) == 1) */
        } /* for (i) */
        break;
    } /* switch (version) */
    close_db(f, "nick2.db");
}
#endif

/* Load/save data files. */

void load_ns_dbase(void)
{
    FILE *f;
    int i,j;
    int ver;
    NickInfo *ni;

    if (!(f = open_db(s_NickServ, NICKSERV_DB, "r", 1)))
		return;

    switch (ver = get_file_version(f, NICKSERV_DB)) {
      case 6:
        for (i = 33; i < 256; ++i) {
            while (fgetc(f) == 1) {
                   ni = smalloc(sizeof(NickInfo));
                   if (1 != fread(ni, sizeof(NickInfo), 1, f))
                       fatal_perror("Read error on %s", NICKSERV_DB);

                ni->flags &= ~(NI_IDENTIFIED | NI_RECOGNIZED);

                alpha_insert_nick(ni);

                if (ni->url)
                    ni->url = read_string(f, NICKSERV_DB);

                if (ni->email)
                    ni->email = read_string(f, NICKSERV_DB);

                if (ni->uin)
                    ni->uin = read_string(f, NICKSERV_DB);

                if (ni->age)
                    ni->age = read_string(f, NICKSERV_DB);

			    if (ni->forward)
			 		ni->forward = read_string(f, NICKSERV_DB);

                if (ni->hold)
                    ni->hold = read_string(f, NICKSERV_DB);

                if (ni->mark)
                    ni->mark = read_string(f, NICKSERV_DB);

                if (ni->forbid)
                    ni->forbid = read_string(f, NICKSERV_DB);

                if (ni->info)
                    ni->info = read_string(f, NICKSERV_DB);

                if (ni->sex)
                    ni->sex = read_string(f, NICKSERV_DB);

                if (ni->mlock)
                    ni->mlock = read_string(f, NICKSERV_DB);
				else
		    		ni->mlock = NULL;

                if (ni->last_quit)
                    ni->last_quit = read_string(f, NICKSERV_DB);

				if (ni->last_usermask)
					ni->last_usermask = read_string(f, NICKSERV_DB);

				if (ni->last_realname)
					ni->last_realname = read_string(f, NICKSERV_DB);

                if (ni->accesscount) {
                    char **access1;
                    access1 = smalloc(sizeof(char *) * ni->accesscount);
                    ni->access = access1;
                    for (j = 0; j < ni->accesscount; ++j, ++access1)
                        *access1 = read_string(f, NICKSERV_DB);

		            } else {
						ni->access = NULL;
				}

                if (ni->ajoincount) {
                    char **autojoin;
                    autojoin = smalloc(sizeof(char *) * ni->ajoincount);
                    ni->autojoin = autojoin;
                    for (j = 0; j < ni->ajoincount; ++j, ++autojoin)
                        *autojoin = read_string(f, NICKSERV_DB);
                }

                if (ni->comline) {
                    char **nickcom;
                    nickcom = smalloc(sizeof(char *) * ni->comline);
                    ni->comment = nickcom;
                    for (j = 0; j < ni->comline; ++j, ++nickcom)
                        *nickcom = read_string(f, NICKSERV_DB);
                }

                if (ni->noteline) {
                    char **nicknote;
                    nicknote = smalloc(sizeof(char *) * ni->noteline);
                    ni->note = nicknote;
                    for (j = 0; j < ni->noteline; ++j, ++nicknote)
                        *nicknote = read_string(f, NICKSERV_DB);
                }
                ni->id_timestamp = 0;
            } /* while (fgetc(f) == 1) */
        } /* for (i) */
        break;
      case 5:
	{
	NickInfoV5 *old_ni;
        for (i = 33; i < 2000; ++i) {
            while (fgetc(f) == 1) {
                   ni = smalloc(sizeof(NickInfo));
                   old_ni = smalloc(sizeof(NickInfoV5));

                   if (1 != fread(old_ni, sizeof(NickInfoV5), 1, f))
                       fatal_perror("Read error on %s", NICKSERV_DB);
                strscpy(ni->nick, old_ni->nick, sizeof(ni->nick));
                strscpy(ni->pass, old_ni->pass, sizeof(ni->pass));

                ni->time_registered = old_ni->time_registered;
                ni->last_seen = old_ni->last_seen;
                ni->accesscount = old_ni->accesscount;
                ni->flags = old_ni->flags;
                ni->memomax = old_ni->memomax;
                ni->channelcount = old_ni->channelcount;
                ni->news = old_ni->news;
                ni->flags &= ~(NI_IDENTIFIED | NI_RECOGNIZED);
		ni->eflags = 0;
		ni->ajoincount = 0;
		ni->comline = 0;
		ni->noteline = 0;

                alpha_insert_nick(ni);

                if (old_ni->url) {
                    old_ni->url = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->url)>0)
			ni->url = old_ni->url;
		    else
			ni->url = NULL;
		} else
		    ni->url = NULL;

                if (old_ni->email) {
                    old_ni->email = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->email)>0)
			ni->email = old_ni->email;
		    else
			ni->email = NULL;
		} else
		    ni->email = NULL;

                if (old_ni->uin) {
                    old_ni->uin = read_string(f, NICKSERV_DB);
		    if(strlen(old_ni->uin)>0)
			ni->uin = old_ni->uin;
		    else
			ni->uin = NULL;
		} else
		    ni->uin = NULL;

                if (old_ni->age) {
                    old_ni->age = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->age)>0)
			ni->age = old_ni->age;
		    else
			ni->age = NULL;
		} else
		    ni->age = NULL;

		if (old_ni->forward) {
		    old_ni->forward = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->forward)>0)
			ni->forward = old_ni->forward;
		    else
			ni->forward = NULL;
		} else
		    ni->forward = NULL;

                if (old_ni->hold) {
                    old_ni->hold = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->hold)>0)
			ni->hold = old_ni->hold;
		    else
			ni->hold = NULL;
		} else
		    ni->hold = NULL;

                if (old_ni->mark) {
                    old_ni->mark = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->mark)>0)
			ni->mark = old_ni->mark;
		    else
			ni->mark = NULL;
		} else
		    ni->mark = NULL;

                if (old_ni->forbid) {
                    old_ni->forbid = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->forbid)>0)
			ni->forbid = old_ni->forbid;
		    else
			ni->forbid = NULL;
		} else
		    ni->forbid = NULL;

                if (old_ni->info) {
                    old_ni->info = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->info)>0)
			ni->info = old_ni->info;
		    else
			ni->info = NULL;	
		} else
			ni->info = NULL;
#ifdef OLDB3
  	        ni->mlock = NULL;
#else
                if (old_ni->mlock) {
                    old_ni->mlock = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->mlock)>0)
			ni->mlock = old_ni->mlock;
		    else
			ni->mlock = NULL;
		} else
		    ni->mlock = NULL;
#endif

                if (old_ni->sex) {
                    old_ni->sex = read_string(f, NICKSERV_DB);
		    if (strlen(old_ni->sex)>0)
			ni->sex = old_ni->sex;
		    else
			ni->sex = NULL;
		} else
		    ni->sex = NULL;

		ni->last_quit = NULL;

                old_ni->last_usermask = read_string(f, NICKSERV_DB);
		ni->last_usermask = old_ni->last_usermask;
                old_ni->last_realname = read_string(f, NICKSERV_DB);
		ni->last_realname = old_ni->last_realname;

		free(old_ni);

                if (ni->accesscount) {
                    char **access1;
                    access1 = smalloc(sizeof(char *) * ni->accesscount);
                    ni->access = access1;
                    for (j = 0; j < ni->accesscount; j++, access1++)
                        *access1 = read_string(f, NICKSERV_DB);
                }

                ni->autojoin = NULL;
                ni->note = NULL;
                ni->comment = NULL;

                ni->id_timestamp = 0;
            } /* while (fgetc(f) == 1) */
        } /* for (i) */
        break;
	}
      case 4:
      case 3:
      case 2:
      case 1:
      default:
	fatal("Unsupported version number (%d) on %s", i, NICKSERV_DB);
    } /* switch (version) */

    close_db(f, NICKSERV_DB);
}

void save_ns_dbase(void)
{
    FILE *f;
    int i,j;
    NickInfo *ni;
    char **access1, **autojoin, **nicknote, **nickcom;

    if (!(f = open_db(s_NickServ, NICKSERV_DB, "w", NS_VERSION)))
	return;


    for (i = 33; i < 256; ++i) {
	for (ni = nicklists[i]; ni; ni = ni->next) {
	    fputc(1, f);

	    if (1 != fwrite(ni, sizeof(NickInfo), 1, f))
  	    fatal_perror("Write error on %s", NICKSERV_DB);

	    if (ni->url) {
		write_string(ni->url, f, NICKSERV_DB);
	    }
	    if (ni->email) {
		write_string(ni->email, f, NICKSERV_DB);
	    }
            if (ni->uin) {
                write_string(ni->uin, f, NICKSERV_DB);
	    }
            if (ni->age) {
                write_string(ni->age, f, NICKSERV_DB); 
	    }
	    if (ni->forward) {
		write_string(ni->forward, f, NICKSERV_DB);
	    }
            if (ni->hold) {
                write_string(ni->hold, f, NICKSERV_DB);
	    }
            if (ni->mark) {
                write_string(ni->mark, f, NICKSERV_DB);
	    }
            if (ni->forbid) {
                write_string(ni->forbid, f, NICKSERV_DB);
	    }
            if (ni->info) {
                write_string(ni->info, f, NICKSERV_DB);
	    }
            if (ni->sex) {
                write_string(ni->sex, f, NICKSERV_DB);
	    }
            if (ni->mlock) {
                write_string(ni->mlock, f, NICKSERV_DB);
	    }
            if (ni->last_quit) {
                write_string(ni->last_quit, f, NICKSERV_DB);
	    }

            if (ni->last_usermask)
	    write_string(ni->last_usermask ? ni->last_usermask : "",
							f, NICKSERV_DB);
            if (ni->last_realname)
	    write_string(ni->last_realname ? ni->last_realname : "", f, NICKSERV_DB);

	    for (access1 = ni->access, j = 0; j < ni->accesscount; ++access1, ++j)
		{
		write_string(*access1, f, NICKSERV_DB);
		}

	    for (autojoin = ni->autojoin, j = 0; j < ni->ajoincount; ++autojoin, ++j)
		{
			write_string(*autojoin, f, NICKSERV_DB);
		}

	    for (nickcom = ni->comment, j = 0; j < ni->comline; ++nickcom, ++j)
		{
			write_string(*nickcom, f, NICKSERV_DB);
		}

	    for (nicknote = ni->note, j = 0; j < ni->noteline; ++nicknote, ++j)
		{
			write_string(*nicknote, f, NICKSERV_DB);
		}
	}
	fputc(0, f);
    }
    close_db(f, NICKSERV_DB);
}

/*************************************************************************/

/*************************************************************************/
/* Install any timeouts we need at startup. */

void ns_install_timeouts(void)
{
#if NS_LAST_SEEN_DELAY > 0
    add_timeout(NS_LAST_SEEN_DELAY, update_last_seen_time, 1);
#endif
}

/*************************************************************************/

/* Check whether a user is on the access list of the nick they're using.
 * If not, send warnings as appropriate.  If so (and not NI_SECURE), update
 * last seen info.  Return 1 if the user is valid and recognized, 0
 * otherwise (note that this means an NI_SECURE nick will always return 0
 * from here). */

int validate_user(User *u)
{
    NickInfo *ni, *hni;
    int on_access;

    if (!(ni = findnick (u->nick)) || !(hni = nickhost (ni)))
	return 0;

    if (ni->flags & NI_VERBOTEN) {
	ni->flags &= ~NI_RECOGNIZED;
        ni->flags &= ~NI_IDENTIFIED;
        send_cmd(s_NickServ, "SVSMODE %s -r 0", u->nick);
	notice(s_NickServ, u->nick, NS_NEEDID);
	notice(s_NickServ, u->nick, NS_NEEDID2);
        add_ns_timeout(ni, TO_COLLIDE_TL, 30);
	return 0;
    }

    if (ni->flags & NI_WAUTH)
	return 0;

    if (ni->flags & NI_FREEZE) {
        send_cmd(s_NickServ, "SVSMODE %s -r 0", u->nick);
	return 0;
    }

    on_access = is_on_access(u, ni);

    if (!(hni->flags & NI_SECURE) && (on_access || is_on_id_list(u->nick, hni->nick))) {
	ni->flags |= NI_RECOGNIZED;
	if (is_on_id_list(u->nick, hni->nick)) {
	    ni->flags |= NI_IDENTIFIED;
    	    ni->last_seen = hni->last_seen = CTime;    
	    if (hni->last_usermask)
	        free(hni->last_usermask);
	    hni->last_usermask = smalloc(strlen(u->username)+strlen(u->host)+2);
	    sprintf(hni->last_usermask, "%s@%s", u->username, u->host);
	    if (hni->last_realname)
 	        free(hni->last_realname);
	    hni->last_realname = sstrdup(u->realname);
        }
        send_cmd(s_NickServ, "SVSMODE %s +r 0", u->nick);
	return 1;
    }
    if (hni->flags & NI_SECURE) {
        send_cmd(s_NickServ, "SVSMODE %s -r 0", u->nick);
	notice(s_NickServ, u->nick, NS_NEEDID3);
	notice(s_NickServ, u->nick, NS_NEEDID4, s_NickServ);
	notice(s_NickServ, u->nick, NS_NEEDID5);
    } else {
        send_cmd(s_NickServ, "SVSMODE %s -r 0", u->nick);
	notice(s_NickServ, u->nick, NS_NEEDID6);
	notice(s_NickServ, u->nick, NS_NEEDID7, s_NickServ);
    }

    if ((hni->flags & NI_KILLPROTECT)
		&& !((hni->flags & NI_SECURE) && (on_access || is_on_id_list(u->nick, u->nick)))) {
	notice(s_NickServ, u->nick, NS_60SEC);
        add_ns_timeout(ni, TO_COLLIDE_TL, 30);
    }

    return 0;
}
/*************************************************************************/
static void do_slavedrop (const char *source)
{
    char *nick = strtok (NULL, " ");
    NickInfo *ni, *hni;
    User *u = finduser (source);

    if (!nick)
    {
        notice (s_NickServ, source, "Syntax: \2UNLINK Nick\2");
        notice (s_NickServ, source, ERR_MORE_INFO, s_NickServ, "UNLINK");
    }
    else if (stricmp (source, nick) == 0)
        notice (s_NickServ, source, ERR_NOT_ON_YOURSELF, "UNLINK");
    else if (!(ni = findnick (nick)) || !(hni = nickhost (ni)))
        notice (s_NickServ, source, NS_NOT_REGISTERED, nick);

    else if (stricmp (hni->nick, source) != 0)
    {
        if (issibling(findnick(source), nick))
            notice (s_NickServ, source, ERR_MUST_BE_HOST, "UNLINK");
        else
            notice (s_NickServ, source, ERR_ACCESS_DENIED);
    }
    else if (!u || !(hni->flags & NI_IDENTIFIED))
    {
        notice (s_NickServ, source, ERR_NEED_PASSWORD, "UNLINK");
        notice (s_NickServ, source, ERR_IDENT_NICK_FIRST, s_NickServ);
        if (!u)
            log ("[ERR] %s: UNLINK from nonexistent user %s", s_NickServ, source);
    }
    else
    {
        delnick (ni);
        log ("[DROP] %s: %s!%s@%s dropped nickname %s (slave)", s_NickServ,
                   source, u->username, u->host, nick);
        notice (s_NickServ, source, NS_DROPPED, nick, "", "", "");
    }
}

/*************************************************************************/

/* Cancel validation flags for a nick (i.e. when the user with that nick
 * signs off or changes nicks).  Also cancels any impending collide. */

void cancel_user(User *u)
{
    NickInfo *ni;
    ni = findnick(u->nick);
    if (ni) {
		ni->flags &= ~(NI_IDENTIFIED | NI_RECOGNIZED);
		del_ns_timeout(ni, TO_COLLIDE);
    }
}

/*************************************************************************/
int shouldexpire(NickInfo *ni) {

    const time_t expire_time = atoi(tokn[24])*24*60*60;
    const time_t expire_mail = atoi(tokn[23])*24*60*60;
    time_t now = CTime;
    FILE *fp;
    char cmdbuf[500];
    char cmdbuf2[500];
    char timebuf[64];
    struct tm tm;

	if (nickhost(ni)->flags & NI_VERBOTEN)
		return 0;

	if (nickhost(ni)->flags & NI_HOLD)
		return 0;

	if (nickhost(ni)->flags & NI_HOP)
		return 0;

	if (!ni->last_seen)
		return 1;

	if (((now - ni->last_seen) >= 60*60*24) && (ni->flags & NI_WAUTH))
		return 1;

	if (ni->flags & NI_WAUTH)
		return 0;

	if ((ni->flags & NI_SLAVE) && !(findnick(ni->last_usermask)))
		return 1;

	if (ni->flags & NI_SLAVE)
		return 0;

        if (stricmp(tokn[15], "ON") ==0) {
		    tm = *localtime(&now);
		    strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
		    timebuf[sizeof(timebuf)-1] = 0;
	}

	if (((now - nickhost(ni)->last_seen) >= (expire_time - expire_mail)) && !(nickhost(ni)->flags & NI_EXMAIL)) {

		slog("NS XE %s", ni->nick);
                do_break_log("NS_X", "NS XE %s", ni->nick);
		log("E-mail Near Expiring nickname %s", ni->nick);
	        ni->flags |= NI_EXMAIL;

	      if (is_tokn(15)) {
			fp = fopen("memo.txt", "a");
	        fprintf (fp, "Date: %s\n", timebuf);
	        fprintf (fp, "From: %s <%s>\n", s_NickServ, tokn[29]);
	        fprintf (fp, "To: %s <%s>\n", ni->nick, ni->email);
	        fprintf (fp, "Subject: %s Nickname Nearly Expire Reminder!!\n", s_NickServ);
	        fprintf (fp, "You received this email because you have registered\n");
	        fprintf (fp, "Your nickname (%s) on the %s\n\n",ni->nick, NETWORK_NAME);
	        fprintf (fp, "with password: %s\n\n", ni->pass);        
			fprintf (fp, "Your nick will expire in %s days if you don't identify.\n\n", tokn[23]);
	        fclose (fp);

			sprintf(cmdbuf, "cat memo.sig >> memo.txt");
			system(cmdbuf);
			*cmdbuf = 0;

			if (is_tokn(32)) {
              sprintf(cmdbuf2, "mail -s \"%s %s Memo from %s\" %s < memo.txt",
                  NETWORK_NAME, s_NickServ, ni->nick, ni->email);
			} else {
				sprintf (cmdbuf, "%s -t %s < %s", sendmail_path, ni->email, "memo.txt");
			}
				sprintf (cmdbuf2, "rm -f %s", "memo.txt");

				system(cmdbuf);
				system(cmdbuf2);
		  } /* If allow mail */


		return 0;
	}

	if (now - nickhost(ni)->last_seen >= expire_time)
		return 1;

	return 0;
}
/*************************************************************************/
/* Remove all nicks which have expired. */

void expire_nicks()
{
    NickInfo *ni, *next;
    int i;
    long count = 0;
    long xcount = 0;

    for (i = 33; i < 256; ++i) {
	for (ni = nicklists[i]; ni; ni = next) {
	    next = ni->next;
	    if (shouldexpire(ni)) {
		++xcount;
                slog("NS X %s", ni->nick);
                do_break_log("NS_X", "NS X %s", ni->nick);
		log("Expiring nickname %s", ni->nick);
		delnick(ni);
	    }
	}

    } /* Main For */

    for (i = 33; i < 256; ++i) {
        for (ni = nicklists[i]; ni; ni = ni->next)
            ++count;
    }

    if (DISPLAY_UPDATES == 1)
        wallops(SERVER_NAME, "Completed NickName Expire (%d/%d)",
        	xcount,count);
}


/*************************************************************************/

/* Return the NickInfo structure for the given nick, or NULL if the nick
 * isn't registered. */

NickInfo *findnick(const char *nick)
{
    NickInfo *ni;

    for (ni = nicklists[tolower(*nick)]; ni; ni = ni->next) {
	if (stricmp(ni->nick, nick) == 0)
	    return ni;
    }
    return NULL;
}

/*************************************************************************/
/*********************** NickServ private routines ***********************/
/*************************************************************************/

/* Is the given user's address on the given nick's access list?  Return 1
 * if so, 0 if not. */

int is_on_access(User *u, NickInfo *ni)
{
    int i;
    char *buf;
    NickInfo *hni = nickhost (ni);

    if (!ni || !hni) return 0;

    i = strlen(u->username);
    buf = smalloc(i + strlen(u->host) + 2);
    sprintf(buf, "%s@%s", u->username, u->host);
    strlower(buf+i+1);

    for (i = 0; i < hni->accesscount; ++i) {
	if (match_wild(hni->access[i], buf)) {
	    free(buf);
	    return 1;
	}
    }
    free(buf);
    return 0;
}

/*************************************************************************/

/* Insert a nick alphabetically into the database. */

static void alpha_insert_nick(NickInfo *ni)
{
    NickInfo *ni2, *ni3;
    char *nick = ni->nick;
#ifdef MERGE
    if (findnick(nick)) {
        delnick(findnick(nick));
        slog("DELETE %s for update", nick);
    }
#endif

    for (ni3 = NULL, ni2 = nicklists[tolower(*nick)];
			ni2 && stricmp(ni2->nick, nick) < 0;
			ni3 = ni2, ni2 = ni2->next)
	;
    ni->prev = ni3;
    ni->next = ni2;
    if (!ni3)
	nicklists[tolower(*nick)] = ni;
    else
	ni3->next = ni;
    if (ni2)
	ni2->prev = ni;
}

/*************************************************************************/

/* Add a nick to the database.  Returns a pointer to the new NickInfo
 * structure if the nick was successfully registered, NULL otherwise.
 * Assumes nick does not already exist. */

static NickInfo *makenick(const char *nick)
{
    NickInfo *ni;

    ni = scalloc(sizeof(NickInfo), 1);
    strscpy(ni->nick, nick, NICKMAX);
    alpha_insert_nick(ni);
    return ni;
}

/*************************************************************************/

/* Remove a nick from the NickServ database.  Return 1 on success, 0
 * otherwise.  Also deletes the nick from any channel access lists it is
 * on. 
 */

static int delnick(NickInfo *ni)
{
    User *u;
    int i;
    for (u = userlist; u; u = u->next) {
       for (i=0; u->id_nicks[i] && i < MAX_IDS; i++) {
          if (stricmp(u->id_nicks[i], ni->nick) == 0)
              free(u->id_nicks[i]);
       }
    }
#ifdef SENDLOG
    for (i=0; i < MAX_SENDLOGS; i++) {
       if (sendlogs[i]) {
          if (ni && stricmp(ni->nick, sendlogs[i]) == 0) {
              free(sendlogs[i]);
              sendlogs[i] = NULL;
          }
       }
    }
#endif

    if (ni->next)
	ni->next->prev = ni->prev;
    if (ni->prev)
	ni->prev->next = ni->next;
    else
	nicklists[tolower(*ni->nick)] = ni->next;
    if (ni->last_usermask)
	free(ni->last_usermask);
    if (ni->last_realname)
	free(ni->last_realname);
    if (ni->access) {
	for (i = 0; i < ni->accesscount; ++i)
	    free(ni->access[i]);
	free(ni->access);
    }
    if (ni->autojoin) {
	for (i = 0; i < ni->ajoincount; i++) {
	    if (ni->autojoin[i])
	    free(ni->autojoin[i]);
	}
	free(ni->autojoin);
    }
    if (ni->comline) {
	for (i = 0; i < ni->comline; i++) {
	    if (ni->comment[i])
	    free(ni->comment[i]);
	}
	free(ni->comment);
    }
    if (ni->noteline) {
	for (i = 0; i < ni->noteline; i++) {
	    if (ni->note[i])
	    free(ni->note[i]);
	}
	free(ni->note);
    }

    if (ni->url)
      free(ni->url);
    if (ni->email)
      free(ni->email);
    if (ni->forward)
      free(ni->forward);
    if (ni->hold)
      free(ni->hold);
    if (ni->mark);
      free(ni->mark);
    if (ni->forbid)
      free(ni->forbid);
    if (ni->uin)
      free(ni->uin);
    if (ni->age)
      free(ni->age);
    if (ni->info)
      free(ni->info);
    if (ni->sex)
      free(ni->sex);
    if (ni->mlock)
      free(ni->mlock);
    if (ni->last_quit)
      free(ni->last_quit);

    clear_auth(ni->nick);
    cs_remove_nick(ni->nick);
    clear_memos(ni->nick);
    os_remove_nick(ni->nick);
#ifdef DEFHOSTSERV
    del_hostline(ni->nick);
#endif
    free(ni);
    return 1;
}

/*************************************************************************/
void do_niautojoin(NickInfo *ni, const char *source)
{
#ifdef SVSJOIN
    char **autojoin;
    char chans [512]; 
    User *u;
    Channel *c;
    int i;
    *chans = 0;

    if (!ni->ajoincount)
	return;

    u = finduser(ni->nick);

    if (!u)
	return;

    for (autojoin = ni->autojoin, i = 0; i < ni->ajoincount; autojoin++, i++) {
	c = findchan(*autojoin);
	if (!c) {
	        strcat(chans, *autojoin);
	        strcat(chans,",");
	        if (strlen(chans) > 200) {
		    send_cmd(s_NickServ, "%s %s %s", SVSJOIN, source,chans);
		    *chans =0;
		}
	} else {
		if (hasmode("k", c->modes) || hasmode("i", c->modes) 
		|| hasmode("O", c->modes) || hasmode("A", c->modes) || isinban(c, u)) {
		    notice(s_NickServ, source, "Can not autojoin \2%s\2, you are banned", c->name);
		} else {
		        strcat(chans, *autojoin);
		        strcat(chans,",");
		        if (strlen(chans) > 200) {
			    send_cmd(s_NickServ, "%s %s %s", SVSJOIN, source,chans);
			    *chans=0;
			}
		}
	}
    }
    send_cmd(s_NickServ, "%s %s %s", SVSJOIN, source,chans);
    *chans=0;

#endif
}

/*************************************************************************/
/* Collide a nick. */

static void collide(NickInfo *ni, int from_timeout)
{
    char newnick[NICKMAX];
    int success;
    User *user = finduser(ni->nick);

    if (!from_timeout)
		del_ns_timeout(ni, TO_COLLIDE);

	srand(time(NULL));

	for (success = 0, strscpy (newnick, ni->nick, NICKMAX);
		  strlen (newnick) < NICKMAX && success != 1;)
	{
		*newnick = 0;
		sprintf (newnick, "Guest%d",
                1+(int) (70000.0*rand()/(RAND_MAX+10000.0)));
		if (!(finduser (newnick)))
				 success = 1;
        }

   send_cmd(SERVER_NAME, "433 %s %s :Nickname is registered to someone else", user->nick, user->nick);
   send_cmd(NULL, "SVSNICK %s %s :%ld", ni->nick, newnick, time(NULL));
   CNEWNICK(ni->nick, "enforcer", services_host, "Services Enforcer", "+", 0);

   if (user)
       strscpy(user->nick, newnick, NICKMAX);

   ni->flags |= NI_KILL_HELD;

   add_ns_timeout(ni, TO_RELEASE, RELEASE_TIMEOUT);

   if (is_tokn(44)) {
	    notice(s_NickServ, newnick, "For more information on what this means, go to:");
	    notice(s_NickServ, newnick, "%s", tokn[14]);
   }

}
/*************************************************************************/

/* Release hold on a nick. */

static void release(NickInfo *ni, int from_timeout)
{
    if (!from_timeout)
	del_ns_timeout(ni, TO_RELEASE);
    send_cmd(ni->nick, "QUIT :Leaving");
    ni->flags &= ~NI_KILL_HELD;
}

/*************************************************************************/
/*************************************************************************/

static struct my_timeout {
    struct my_timeout *next, *prev;
    NickInfo *ni;
    Timeout *to;
    int type;
} *my_timeouts;

/*************************************************************************/
/* Remove a collide/release timeout from our private list. */

static void rem_ns_timeout(NickInfo *ni, int type)
{
    struct my_timeout *t, *t2;

    t = my_timeouts;
    while (t) {
	if (t->ni == ni && t->type == type) {
	    t2 = t->next;
	    if (t->next)
		t->next->prev = t->prev;
	    if (t->prev)
		t->prev->next = t->next;
	    else
		my_timeouts = t->next;
	    free(t);
	    t = t2;
	} else {
	    t = t->next;
	}
    }
}
/*************************************************************************/
/* Collide a nick on timeout. */

static void timeout_collide(Timeout *t)
{
    NickInfo *ni = t->data;
    User *u;

    rem_ns_timeout(ni, TO_COLLIDE);
    /* If they identified or don't exist anymore, don't kill them. */
    if (ni->flags & NI_IDENTIFIED
		|| !(u = finduser(ni->nick))
		|| u->my_signon > t->settime)
	return;
    /* The RELEASE timeout will always add to the beginning of the
     * list, so we won't see it.  Which is fine because it can't be
     * triggered yet anyway. */
    collide(ni, 1);
}

static void timeout_collide_tl(Timeout *t)
{
   NickInfo *ni = t->data;
   User *u;

   rem_ns_timeout(ni, TO_COLLIDE_TL);
   if (ni->flags & NI_IDENTIFIED
		|| !(u = finduser(ni->nick))
		|| u->my_signon > t->settime)
	return;
   add_ns_timeout(ni, TO_COLLIDE, 30);
   notice(s_NickServ, ni->nick, NS_30SEC);
}

/*************************************************************************/

/* Release a nick on timeout. */

static void timeout_release(Timeout *t)
{
    NickInfo *ni = t->data;

    rem_ns_timeout(ni, TO_RELEASE);
    release(ni, 1);
}

/*************************************************************************/
/* Add a collide/release timeout. */

static void add_ns_timeout(NickInfo *ni, int type, time_t delay)
{
    Timeout *to;

    struct my_timeout *t;
    void (*timeout_routine)(Timeout *);

    if (type == TO_COLLIDE_TL)
        timeout_routine = timeout_collide_tl;
    else if (type == TO_COLLIDE)
	timeout_routine = timeout_collide;
    else if (type == TO_RELEASE)
	timeout_routine = timeout_release;
    else {
	log("NickServ: unknown timeout type %d!  ni=%p (%s), delay=%ld",
		type, ni, ni->nick, delay);
	return;
    }

    to = add_timeout(delay, timeout_routine, 0);
    to->data = ni;
    t = smalloc(sizeof(*t));
    t->next = my_timeouts;
    my_timeouts = t;
    t->prev = NULL;
    t->ni = ni;
    t->to = to;
    t->type = type;
}

/*************************************************************************/

/* Delete a collide/release timeout. */

static void del_ns_timeout(NickInfo *ni, int type)
{

    struct my_timeout *t, *t2;

    t = my_timeouts;
    while (t) {
	if (t->ni == ni && t->type == type) {
	    t2 = t->next;
	    if (t->next)
		t->next->prev = t->prev;
	    if (t->prev)
		t->prev->next = t->next;
	    else
		my_timeouts = t->next;
	    del_timeout(t->to);
	    free(t);
	    t = t2;
	} else {
	    t = t->next;
	}
    }  
}

/*************************************************************************/
/*************************************************************************/

/* Update last-seen time for all users currently online. */

static void update_last_seen_time(Timeout *to_unused)
{
    User *u;
    NickInfo *ni;

    /* This is O(n^2) at the moment, because the user list isn't known to
     * be sorted.
     */
    for (u = first_user(); u; u = u->next) {
	ni = findnick(u->nick);
	if (ni)
            if (ni && (ni->flags & (NI_IDENTIFIED | NI_RECOGNIZED)))
        	 ni->last_seen = CTime;
    }
}

/*************************************************************************/
/*********************** NickServ command routines ***********************/
/*************************************************************************/

/* Return a help message. */

static void do_ohelp(const char *source)
{
    const char *cmd = strtok(NULL, "");

    char buf[BUFSIZE];
    snprintf(buf, sizeof(buf), "%s%s",s_NickServ, cmd ? " oper " : " oper");
    strscpy(buf+strlen(buf), cmd ? cmd : "", sizeof(buf)-strlen(buf));
    helpserv(s_NickServ, source, buf);
    if (!cmd)
        slog("NS - %s requested ohelp: (index)", source);

    if (cmd)
        slog("NS - %s requested ohelp: %s", source, cmd);
}

static void do_help(const char *source)
{

    const char *cmd = strtok(NULL, "");
    char buf[BUFSIZE];
    snprintf(buf, sizeof(buf), "%s%s", s_NickServ, cmd ? " " : "");
    strscpy(buf+strlen(buf), cmd ? cmd : "", sizeof(buf)-strlen(buf));
    helpserv(s_NickServ, source, buf);
    if (!cmd || (cmd && stricmp(cmd, "REGISTER") == 0))  
       notice(s_NickServ, source,
             "Nicknames will expire after \2%d days\2 of inactivity",
              atoi(tokn[24]));

}

/*************************************************************************/

/* Register a nick. */
#ifdef WEBSERVICE
void do_nsregister(const char *source) {
	do_register(source);
}
#endif
static void do_register(const char *source)
{
    NickInfo *ni;
    User *u;
    char *pass = strtok(NULL, " ");
    char *email = strtok(NULL, " ");
    int i;
    long int guestbuf = 0;
    double x = RAND_MAX + 1.0;
    int y;
    FILE *fp;
    char *apoint;
    char cmdbuf[500];
    char cmdbuf2[500];
    char newcode[64];

    if (readonly) {
	notice(s_NickServ, source,
		"Sorry, nickname registration is temporarily disabled.");
    }
    ni = findnick(source);
    if (!pass || !email) {
        notice(s_NickServ, source, "Syntax: \2REGISTER\2 <password> <email>"); 
	notice(s_NickServ, source,
			"\2/msg %s HELP REGISTER\2 for more information.",
			s_NickServ);

    } else if (strlen(pass) >= PASSMAX) {
        notice(s_NickServ, source, "Password too long."); 
    } else if (!(u = finduser(source))) {
	log("%s: Can't register nick %s: nick not online", s_NickServ, source);
	notice(s_NickServ, source, NS_REGFAIL);
    } else if (CTime < u->lastnickreg + NS_REG_DELAY) {
	notice(s_NickServ, source,
	    "Please wait %d seconds before using the REGISTER command again.",
	    NS_REG_DELAY);
    } else if (ni) {
	if (ni->flags & NI_VERBOTEN) {
	    log("%s: %s@%s tried to register FORBIDden nick %s", s_NickServ,
			u->username, u->host, source);
	    notice(s_NickServ, source,
			"Nickname \2%s\2 may not be registered.", source);
	} else {
	    notice(s_NickServ, source,
			"Nickname \2%s\2 is already registered!", source);
	}
    } else if (stricmp(source, pass) == 0 || strlen(pass) < 5) {
	notice(s_NickServ, source,
		"Please try again with a more obscure password.  Passwords "
		"should be at least five characters long, should not be "
		"something easily guessed (e.g. your real name or your "
		"nick), and cannot contain the space or tab characters.  "
		"\2/msg %s HELP REGISTER\2 for more information.", s_NickServ);
    } else if (match_wild_nocase("guest*", source)) {
         notice(s_NickServ, source,
                "You may not register nicks starting with \"Guest\". "
                "\2/msg %s HELP REGISTER\2 for more information.", s_NickServ);
    } else if (strlen(pass) > PASSMAX) {
         notice(s_NickServ, source,
             "Sorry, nick passwords may only be up to %d characters long",
                 PASSMAX);
     } else if (!strchr(email, '@')) {
            notice(s_NickServ, source,
               "ERROR: E-Mail Address must contain an '@'");
     } else if (!strchr(email, '.')) {
            notice(s_NickServ, source,
               "ERROR: Look like you give invalid email.");
     } else if (strchr(email, '\\')) {
           notice(s_NickServ, source,
               "ERROR: invalid email characters.");
    } else {
        ni = makenick(source);
	if (ni) {
	    strscpy(ni->pass, pass, PASSMAX);
	    if (!is_tokn(1)) {
		    ni->flags = NI_IDENTIFIED | NI_RECOGNIZED;
	            if (!is_on_id_list(source, source)) {
        	        for (i=0; u->id_nicks[i] && i < MAX_IDS; i++)
                	    ;
	                if (i < MAX_IDS)
        	             u->id_nicks[i] = sstrdup(source);
	            }
            }
	    nsreg += 1;
	    ni->flags |= NI_MEMO_SIGNON | NI_MEMO_RECEIVE;
	    if (is_oper(source))
	   	ni->flags |= NI_HOP;
	    if (NS_PRIVATE > 0)
	   	ni->flags |= NI_PRIVATE;
	    if (NS_KILL > 0)
		ni->flags |= NI_KILLPROTECT;
	    if (NS_SECURE > 0)
		ni->flags |= NI_SECURE;
	    if (u->iswebtv)
		ni->flags |= NI_PRIVMSG;
	    ni->last_usermask = smalloc(strlen(u->username)+strlen(u->host)+2);
	    sprintf(ni->last_usermask, "%s@%s", u->username, u->host);
	    ni->last_realname = sstrdup(u->realname);
	    ni->time_registered = ni->last_seen = CTime;
	    ni->email = sstrdup (email);
            ni->news = 0;
            ni->uin = NULL;
            ni->age = NULL;
            ni->info = NULL;
            ni->sex = NULL;
            ni->mlock = NULL;
            ni->ajoincount = 0;
            ni->comline = 0;
            ni->noteline = 0;
	    if(is_tokn(1)) {
		    y = 1 + rand() * (99999 / x);
		    guestbuf = y;
		    *newcode = 0;
		    snprintf (newcode, sizeof (newcode), "auspice%d", guestbuf);
		    if (ni->last_realname)
			free(ni->last_realname);
		    ni->last_realname = sstrdup(newcode);
	    }
	    ni->autojoin = NULL;
	    ni->note = NULL;
	    ni->comment = NULL;
	    ni->accesscount = 1;
	    ni->access = smalloc(sizeof(char *));
	    ni->access[0] = create_mask(u);
            ni->memomax = DEF_MAX_MEMOS;
	    clear_memos(ni->nick);
	    log("%s: `%s' registered by %s@%s", s_NickServ,
		    source, u->username, u->host);
#ifdef CYGWIN
	    winlog("%s: `%s' registered by %s@%s", s_NickServ,
		    source, u->username, u->host);
#endif
            slog("NS R %s (%s@%s)", source, u->username, u->host);
            do_break_log("NS_R", "NS R %s (%s@%s)",
                 source, u->username, u->host);

	    if (is_tokn(1) && (fp = fopen("memo.txt", "a"))) {
        	        fprintf (fp, "From: %s <%s>\n", s_NickServ, tokn[29]);
	                fprintf (fp, "To: %s <%s>\n", ni->nick, ni->email);
        	        fprintf (fp, "Subject: Authorize code!!\n");
	                fprintf (fp, "You received this email because you have registered a nickname\n");
	                fprintf (fp, "in %s IRC Network.\n", NETWORK_NAME);
	                fprintf (fp, "Your nickname (%s) authorize code  is %s\n\n",ni->nick, ni->last_realname);
	                fprintf (fp, "Type /msg NickServ Authcode %s %s\n\n ",ni->nick, ni->last_realname);
			fprintf (fp, "If you leave this nick Unauthorize for one day this nick will expire.\n");
		        fclose (fp);

		      sprintf(cmdbuf, "cat memo.sig >> memo.txt");
		      system(cmdbuf);
		      *cmdbuf = 0;

			if (is_tokn(32)) {
		              sprintf(cmdbuf2, "mail -s \"%s %s Memo from %s\" %s < memo.txt",
		                  NETWORK_NAME, s_NickServ, ni->nick, ni->email);
			} else {
				sprintf (cmdbuf, "%s -t %s < memo.txt", sendmail_path, ni->email);
			}

			sprintf (cmdbuf2, "rm -f %s", "memo.txt");
		      system(cmdbuf);
	              system(cmdbuf2);
		      *cmdbuf = 0;
		      *cmdbuf2 = 0;
		} else {
			free(tokn[1]);
			tokn[1] = sstrdup("OFF");
		}
        	    notice(s_NickServ, source, "Register completed.");
        	    notice(s_NickServ, source, NS_REG, source, ni->access[0]);
	            notice(s_NickServ, source, NS_REG2, pass);
	            notice(s_NickServ, source, MISC_OFHELP, tokn[17]);
			if (is_tokn(44)) {
				notice(s_NickServ, source, NS_OWN, tokn[12]);
			}
		    if (is_oper(source) && !(ni->flags & NI_HOP))
		        notice(s_NickServ, source, NS_IRCOP, s_NickServ);
		    if (is_tokn(1)) {
				notice(s_NickServ, source, NS_REGA);
				notice(s_NickServ, source, NS_REGA2);
				ni->flags |= NI_WAUTH;
		    }
		    if (is_tokn(4))
			    wallops(s_NickServ, "Nick \2%s\2 is registered to %s",u->nick, ni->access[0]);
#if REG_SAVES > 0
            if (regcnt++ >= REG_SAVES) {
                wallops(SERVER_NAME,
                     "Automatic DataBase save on %d new registrations",
                          regcnt);
                save_data = 1;
            }
#endif

	    u->lastnickreg = CTime;

	    if (!is_tokn(1))
	            send_cmd(s_NickServ, "SVSMODE %s +r 0", source);

	    if (!is_tokn(1) && is_tokn(33)) {
		    apoint = create_mask(u);
		    do_sendnreg(source, apoint, pass, email);
		    free(apoint);
	    }
	
            show_next_db(source, s_NickServ);

	} else {
	    log("%s: makenick(%s) failed", s_NickServ, source);
	    notice(s_NickServ, source, "Sorry, couldn't register your nickname.");

	}

	}	
}

/*************************************************************************/
#ifdef WEBSERVICE
void do_nsidentify(const char *source)
{
	do_identify(source);
}
#endif
static void do_identify(const char *source)
{
    char *nick = strtok(NULL, " ");
    char *pass = strtok(NULL, " ");
    NickInfo *ni, *hni;
    User *u;
    int res;
    int i, x=0;
    int is_force;
    
    if (!nick) {
        notice(s_NickServ, source, "Syntax: \2IDENTIFY\2 [nick] <password>");
		notice(s_NickServ, source,
		"\2/msg %s HELP IDENTIFY\2 for more information.",
		s_NickServ);
		return;
    }
    if (!pass) {
		pass = sstrdup(nick);
		nick = sstrdup(source);
		x = 1;
    }

    ni = findnick(nick); hni = nickhost(ni);

    is_force = (rec_services_root(source) && !stricmp(pass, "force"));
    
    if (!ni || !hni) {

		notice(s_NickServ, source, "Your nick (%s) isn't registered.",nick);

    } else if (!(u = finduser(source))) {

		log("%s: IDENTIFY from nonexistent nick %s", s_NickServ, source);
		notice(s_NickServ, source, NS_IDFAIL);

    } else if (hni->flags & NI_WAUTH) {

		notice(s_NickServ, source, "Your nick (%s) isn't activate yet.",nick);
		notice(s_NickServ, source, "Type /msg NickServ AUTHCODE %s <code>",nick);
		notice(s_NickServ, source, "The Authore code have been sent to your e-mail after you have registered.");

    } else if (u->flags & U_ABUSIVE) {

		log("%s: Failed identify from abusive user %s", s_NickServ, source);
		notice(s_NickServ, source, "Sorry, You have the abusive flag set");
		nsidf += 1;
		if (EXTRA_SNOOP) {
			slog("NS *I %s (%s!%s@%s) [Abusive]",
				nick, source, u->username, u->host);
			do_break_log("NS_I", "NS *I %s (%s!%s@%s)",
				nick, source, u->username, u->host);
		}

    } else if (!is_force && !(res = check_password(pass, hni->pass))) {

	log("%s: Failed IDENTIFY for %s (%s!%s@%s)",
		s_NickServ, nick, source, u->username, u->host);

	if (EXTRA_SNOOP) {
		slog("NS *I %s (%s!%s@%s)",
			nick, source, u->username, u->host);
		do_break_log("NS_I", "NS *I %s (%s!%s@%s)",
			nick, source, u->username, u->host);
	}

	notice(s_NickServ, source, toknmsg[1], BAD_PW_LIMIT - u->invalid_pw_count - 1);

	if ((BAD_PW_LIMIT - u->invalid_pw_count - 1) == 1)
		notice(s_NickServ, source, toknmsg[2],tokn[17]);

	bad_password(u, nick);
	nsidf += 1;

    } else if (res == -1) {

	notice(s_NickServ, source, NS_IDFAIL);
	nsidf += 1;

    } else if (ni->flags & NI_FREEZE) {

        notice(s_NickServ, source, "This nick name can not Identify as it frozed.");

    } else if ((ni->flags & NI_IDENTIFIED) && x) {
		/* memory will free at the end :) */
        notice(s_NickServ, source, toknmsg[14]);

		if (!(hni->flags & NI_NEVEROP) && is_tokn(41))

			do_doop(source, ni);

		if (hni->eflags & NIX_AUTOJOIN)

			do_niautojoin(hni, source);

		if (hni->mlock)
		        send_cmd(s_NickServ,"SVSMODE %s %s 0",source, hni->mlock);

	} else {
		if (EXTRA_SNOOP) {
				slog("NS I %s (%s!%s@%s) %s",
					nick, source, u->username, u->host,
					is_force ? "[Forced]" : "");
				do_break_log("NS_I", "NS I %s (%s!%s@%s) %s",
					nick, source, u->username, u->host,
						is_force ? "[Forced]" : "");
		}
		nsid += 1;

#ifdef DEFHOSTSERV
		if (is_hostline(nick))
			do_hostline(nick, u);
#endif

		if (!is_force) {
			ni->flags |= NI_IDENTIFIED;
		}
			hni->id_timestamp = u->signon;

		if (!is_on_id_list(u->nick, hni->nick)) {
			for (i=0; u->id_nicks[i] && i < MAX_IDS; i++)
				;
			if (i < MAX_IDS)
				 u->id_nicks[i] = sstrdup(hni->nick);
		}


#ifdef CYGWIN
	    winlog("%s: %s!%s@%s ID %s", s_NickServ,
		    source, u->username, u->host, nick);
#endif

		if (!(hni->flags & NI_SECURE)) {

			hni->last_seen = CTime;

			if (hni->last_usermask)
				free(hni->last_usermask);

			hni->last_usermask = smalloc(strlen(u->username)+strlen(u->host)+2);
			sprintf(hni->last_usermask, "%s@%s", u->username, u->host);
			
			if (hni->last_realname)
				free(hni->last_realname);
			
			hni->last_realname = sstrdup(u->realname);

			log("%s: %s!%s@%s identified for nick %s %s", s_NickServ,
			
				source, u->username, u->host, source,
                        is_force ? "[Forced]" : "");
		}

		if (is_tokn(47)) {
			notice(s_NickServ,source, "Password accepted for nick %s", ni->nick);
		} else {
			send_cmd(server_name, "304 %s :Password accepted for nick %s", source, ni->nick);
		}

		i = is_adminlevel(nick);

		if (i>0 && i<4) {
#ifdef UNREAL
		    swhois(source, ni);
                    send_cmd(s_NickServ,"SVSMODE %s +aqS 0", source);
#endif
#ifdef ULTIMATE
                    send_cmd(s_NickServ,"SVSMODE %s +nPR 0", source);
#endif
#ifdef LIQUID
                    send_cmd(s_NickServ,"SVSMODE %s +atZ 0", source);
#endif
		    if (is_tokn(35))
			m_smartinfo(source);
		} else if (i>0 && i>3) {
#ifdef ULTIMATE
        	        send_cmd(s_NickServ,"SVSMODE %s +a 0",source);
#else
	                send_cmd(s_NickServ,"SVSMODE %s +h 0",source);
#endif
#ifdef UNREAL
			swhois(source, ni);
#endif
		} 

		if (!(hni->flags & NI_NEVEROP) && is_tokn(41))
			do_doop(source, ni);

		if (hni->mlock)
		        send_cmd(s_NickServ,"SVSMODE %s %s 0",source, hni->mlock);

		if (x)
			send_cmd(s_NickServ, "SVSMODE %s +r 0", source);

		if (is_oper(source) && !(hni->flags & NI_HOP) && x)
			    notice(s_NickServ, source, toknmsg[12]);

		if (hni->flags & NI_EXMAIL) {
			   hni->flags &= ~NI_EXMAIL;
			   notice(s_NickServ, source, "Welcome Back!! Long time no see, how are you?.");
		}

		if (u && (hni->flags & NI_PRIVMSG)) {
			u->iswebtv = 1;
		} else {
			u->iswebtv = 0;
		}


		if (hni->eflags & NIX_AUTOJOIN)
			do_niautojoin(hni, source);

		if (!(ni->flags & NI_RECOGNIZED) && x) {
			check_memos(source);
			check_auth(source);
		}

	}

	if (x) {
		free(nick);
		free(pass);
	}

}

/*************************************************************************/

static void do_delete(const char *source)
{
    const char *nick = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);

    if (readonly) {
		notice(s_NickServ, source,
                "Sorry, nickname deletion is temporarily disabled. (Read Only)");
		return;
    }
    if (!nick) {
       notice(s_NickServ, source, "\2Delete\2: You MUST specify a nick");
       return;
    }

    if (!(ni = findnick(nick))) {
        if (nick)
           notice(s_NickServ, source, NS_NOTREG, nick);
        return;
    }
    if (!stricmp(nick, SERVICES_MASTER)) {
        notice(s_NickServ, source,
            "You are not permitted to delete a Master's account");
        return;
    }

    delnick(ni);
    slog("NS +D %s (%s!%s@%s)", nick, source, u->username, u->host);
    do_break_log("NS_G", "NS +D %s (%s!%s@%s)",
       nick, source, u->username, u->host);
    log("%s: %s!%s@%s deleted nickname %s", s_NickServ,
            source, u->username, u->host, nick);
        notice(s_NickServ, source, NS_DELETE, nick);
        wallops(s_NickServ, "%s deleted nickname \2%s\2", source, nick);
}

/*************************************************************************/
NickInfo * nickhost (NickInfo * ni)
{
    NickInfo *ni2;

    if (!ni)
        return NULL;

    if (ni->flags & NI_SLAVE) {
	if (ni->last_usermask) {
		ni2 = findnick(ni->last_usermask);
		if (!ni2)
			return ni;
		else
		        return ni2;
	} else {
		return ni;
	}
    } else
        return ni;
}
/*************************************************************************/
int userisnick (const char *nick)
{
    NickInfo *ni = findnick (nick);
    if (!finduser (nick) || !ni)
        return 0;
    if ((getflags (ni) & NI_SECURE) ? (ni->flags & NI_IDENTIFIED)
        : (ni->flags & (NI_RECOGNIZED | NI_IDENTIFIED)))
        return 1;
    return 0;
}
/*************************************************************************/
/* Find flags (assume host's flags if there is one) */

long getflags (NickInfo * ni)
{
    NickInfo *hni = nickhost (ni);
    long flags;

    if (hni)
    {
        flags = hni->flags;
        flags &= ~NI_RECOGNIZED;
        flags &= ~NI_IDENTIFIED;
        flags &= ~NI_KILL_HELD;
        if (ni->flags & NI_RECOGNIZED)
            flags |= NI_RECOGNIZED;
        if (ni->flags & NI_IDENTIFIED)
            flags |= NI_IDENTIFIED;
        if (ni->flags & NI_KILL_HELD)
            flags |= NI_KILL_HELD;
        if (ni->flags & NI_SLAVE)
            flags |= NI_SLAVE;
        return flags;
    }
    return 0;
}

/*************************************************************************/
NickInfo * slave(const char *nick, int num)
{
    NickInfo *ni;
    int i, cnt = 0;

    for (i = 33; i < 256; ++i)
        for (ni = nicklists[i]; ni; ni = ni->next)
        {
	    if (ni->last_usermask) {
            if (stricmp (ni->last_usermask, nick) == 0)
                cnt++;
            if (cnt == num)
                return ni;
	    }
        }
    return NULL;
}
/*************************************************************************/
int issibling (NickInfo * ni, const char *target)
{
    if (ni)
    {
        NickInfo *hni = nickhost (ni);
        int i;
        if (stricmp (ni->nick, target) == 0)
            return 1;
        if (stricmp (hni->nick, target) == 0)
            return 1;
        for (i = slavecount (hni->nick); i; i--)
            if (stricmp (target, slave (hni->nick, i)->nick) == 0)
                return 1;
    }
    return 0;
}
/*************************************************************************/
int slavecount (const char *nick)
{
    NickInfo *ni;
    int i, cnt = 0;

    for (i = 33; i < 256; ++i)
        for (ni = nicklists[i]; ni; ni = ni->next)
	  if (ni->last_usermask)
            if (stricmp (ni->last_usermask, nick) == 0)
                cnt++;
    return cnt;
}
/*************************************************************************/
static void do_slaves (const char *source)
{
    char *nick = strtok (NULL, " ");
    NickInfo *ni;
    User *u = finduser (source);

    if (!is_oper (source) && nick)
    {
        notice (s_NickServ, source, "Syntax: \2LINKS\2%s", is_oper(source) ? " [Nick]" : "");
        notice (s_NickServ, source, ERR_MORE_INFO, s_NickServ, "LINKS");
    }
    else if (!(ni = findnick (nick ? nick : source)))
        if (nick)
            notice (s_NickServ, source, NS_NOT_REGISTERED, nick);
        else
            notice (s_NickServ, source, NS_YOU_NOT_REGISTERED, source);

    else if (!u)
        notice (s_NickServ, source, ERR_YOU_DONT_EXIST);

    else if (!nick && !userisnick(source))
    {
        notice (s_NickServ, source, NS_NOT_YOURS, source);
        notice (s_NickServ, source, ERR_IDENT_NICK_FIRST, s_NickServ);
        if (!u)
            log ("[ERR] %s: LINKS from nonexistent user %s", s_NickServ, source);
    }
    else
    {
        NickInfo *hni = nickhost(ni);
        int i = slavecount (hni->nick);
        char buf[NICKMAX * i];
        char buf2[256];

        if (i < 1)
            if (nick)
                notice (s_NickServ, source, NS_NO_SLAVES, nick);
            else
                notice (s_NickServ, source, NS_YOU_NO_SLAVES);
        else
        {
            *buf = 0;
            *buf2 = 0;
            sprintf (buf, "\2%s\2: ", hni->nick);
            for (; i > 0; i--) {
                sprintf (buf2, "%s ", slave (hni->nick, i)->nick);
                strcat (buf, buf2);
                if (i==1)
                    notice (s_NickServ, source, "%s", buf);
            }
        }
    }
}

/*************************************************************************/
static void do_drop(const char *source)
{
    const char *nick = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);

    if (readonly) {
	notice(s_NickServ, source,
		"Sorry, nickname de-registration is temporarily disabled.");
	return;
    }

    if (!nick) {
       notice(s_NickServ, source, "Syntax: \2DROP\2 <nick>");
       return;
    }

    if (!(ni = findnick(nick))) {
	    notice(s_NickServ, source, "Your nick isn't registered.");
    } else if (ni && (ni->eflags & NIX_LOCK)) {
        notice (s_NickServ, source, "This nick is locked from drop.");
    } else if (!is_on_id_list(source, nick) && !is_services_root(source)) {
        slog("NS *D %s (%s!%s@%s) [User Hasn't Identified]", source, source, u->username, u->host);
        do_break_log("NS_O", "NS *D %s (%s!%s@%s) [User Hasn't Identified]", source, source, u->username, u->host);
	notice(s_NickServ, source,
		"Password authentication required for that command.");
	notice(s_NickServ, source,
		"Retry after typing \2/msg %s IDENTIFY [\37nick\37] \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: DROP from nonexistent user %s", s_NickServ, source);
    } else if (!u) {
	log("%s: DROP %s from nonexistent oper %s", s_NickServ, nick, source);
	notice(s_NickServ, source, "Can't find your user record!");
    } else {
        if (!stricmp(nick, SERVICES_MASTER)) {
            notice(s_NickServ, source,
               "You are not permitted to delete a Master's account");
            return;
        }
        send_cmd(s_NickServ, "SVSMODE %s -r 0", u->nick);
	delnick(ni);
	nsdrop += 1;
        slog("NS D %s (%s!%s@%s)", nick, source, u->username, u->host);
        do_break_log("NS_O", "NS D %s (%s!%s@%s)",
             nick, source, u->username, u->host);
	log("%s: %s!%s@%s dropped nickname %s", s_NickServ,
		source, u->username, u->host, nick ? nick : source);
        notice(s_NickServ, source, NS_DROP);
    }
}

/*************************************************************************/

static void do_set(const char *source)
{
    char *cmd    = strtok(NULL, " ");
    char *param  = strtok(NULL, "");
    NickInfo *ni = findnick(source);
    User *u;

    if (readonly) {
	notice(s_NickServ, source,
		"Sorry, nickname option setting is temporarily disabled.");
	return;
    }
    nsset += 1;
    if (!param) {

	notice(s_NickServ, source,
                "Syntax: \2SET\2 <command> [<parameters>]");
	notice(s_NickServ, source,
		"\2/msg %s HELP SET\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(source))) {

	notice(s_NickServ, source, NS_NOTREGED);

    } else if (!(u = finduser(source)) || !(ni->flags & NI_IDENTIFIED)) {

	notice(s_NickServ, source,
		"Password authentication required for that command.");
	notice(s_NickServ, source,
		"Retry after typing \2/msg %s IDENTIFY \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: SET from nonexistent user %s", s_NickServ, source);

    } else if (stricmp(cmd, "PASSWORD") == 0) {

	do_set_password(ni, param);

    } else if (stricmp(cmd, "PASSWD") == 0) {

	do_set_password(ni, param);

    } else if (stricmp(cmd, "URL") == 0) {

	do_set_url(ni, param);

    } else if (stricmp(cmd, "EMAIL") == 0) {

	do_set_email(ni, param);

    } else if (stricmp(cmd, "UIN") == 0) {

	do_set_uin (ni, param);

    } else if (stricmp(cmd, "MLOCK") == 0) {
	
	do_set_mlock (ni, param);

    } else if (stricmp(cmd, "AUTOJOIN") == 0) {

	do_set_autojoin(ni, param);


    } else if (stricmp(cmd, "AGE") == 0) {

	do_set_age (ni, param);

    } else if (stricmp(cmd, "INFO") == 0) {

	do_set_info (ni, param);

    } else if (stricmp(cmd, "SEX") == 0) {

	do_set_sex (ni, param);

    } else if (stricmp(cmd, "KILL") == 0) {

	do_set_kill(ni, param);

    } else if (stricmp(cmd, "SECURE") == 0) {

	do_set_secure(ni, param);

    } else if (stricmp(cmd, "PRIVATE") == 0) {

	do_set_private(ni, param);

    } else if (stricmp(cmd, "EMAILMEMOS") == 0) {
	do_set_emailmemos(ni, param);

    } else if (stricmp(cmd, "NEVEROP") == 0) {
        do_set_neverop(ni, param);

    } else if (stricmp(cmd, "NOOP") == 0) {
        do_set_noop(ni, param);

    } else if (stricmp(cmd, "AUTH") == 0) {
        do_set_auth(ni, param);

    } else if (stricmp(cmd, "MALAY") == 0) {
        do_set_malay(ni, param);

    } else if (stricmp(cmd, "PRIVMSG") == 0) {
        do_set_privmsg(ni, param);

    } else if (stricmp(cmd, "NOMEMO") == 0) {
        do_set_nomemo(ni, param);

    } else if (stricmp(cmd, "NOHOST") == 0) {
        do_set_nohost(ni, param);

    } else if (stricmp(cmd, "HIDEMAIL") == 0) {
	do_set_hidemail(ni, param);
    } else if (stricmp(cmd, "BOTMSG") == 0) {
	do_set_botmsg(ni, param);
    } else if (stricmp(cmd, "IRCOP") == 0) {
	if (is_oper(source))
		do_set_ircop(ni, param);

    } else {
	notice(s_NickServ, source,
			"Unknown SET option \2%s\2.", strupper(cmd));
    }
}

/*************************************************************************/

static void do_set_password(NickInfo *ni, char *param)
{
    NickInfo *hni = nickhost(ni);
    const char *source = ni->nick;
    if (strlen(param) >= PASSMAX) {
	    notice(s_NickServ, ni->nick, "Password too long.");
    } else if (stricmp(source, param) == 0 || strlen(param) < 5) {
	notice(s_NickServ, source,
		"Please try again with a more obscure password.  Passwords "
		"should be at least five characters long, should not be "
		"something easily guessed (e.g. your real name or your "
		"nick), and cannot contain the space or tab characters.  "
		"\2/msg %s HELP SET PASSWD\2 for more information.", s_NickServ);
    } else {
	    strscpy(hni->pass, param, PASSMAX);
	    notice(s_NickServ, ni->nick, "Password changed to \2%s\2.", param);
	    show_next_db(source, s_NickServ);
    }
}

/*************************************************************************/

static void do_set_url(NickInfo *ni, char *param)
{
    NickInfo *hni = nickhost(ni);
    const char *source = ni->nick;

    if (stricmp(param,"none")) {
    if (hni->url)
	free(hni->url);
        hni->url = sstrdup(param);
        notice(s_NickServ, ni->nick,
		"URL for %s changed to \2%s\2.", ni->nick, param);
    } else {
       if (!hni->url) {
            notice(s_NickServ, ni->nick,
                "You have no url set to disable!");
            return;
       }
       if (hni->url)
	free(hni->url);
       ni->url = NULL;
       notice(s_NickServ, source, "URL disabled for nick \2%s\2", ni->nick);
    }
    show_next_db(source, s_NickServ);
}

/*************************************************************************/

static void do_set_email(NickInfo *ni, char *param)
{
    NickInfo *hni = nickhost(ni);
    const char *source = ni->nick;
    char *param1 = strtok(param, " ");

    if ((stricmp(param1,"none")) && (stricmp(param1,"hide"))) {

       if (!strchr(param1, '@')) {
            notice(s_NickServ, source,
               "ERROR: E-Mail Address must contain an '@'");
            return;
       }
       if (strchr(param1, '\\') || strchr(param1, ';')
	   || strchr(param1,'|') || strchr(param1, '&')) {
           notice(s_NickServ, source,
               "ERROR: invalid email characters.");
           return;
       }

       if (hni->email)
           free(hni->email);

       hni->email = sstrdup(param1);
       notice(s_NickServ, ni->nick,
		"E-mail address for %s changed to \2%s\2.", ni->nick, param1);
       slog("NS E %s [%s]", ni->nick, param1);

    } else if (!stricmp(param1,"hide")) {
         if (!hni->email) {
            notice(s_NickServ, ni->nick,
                "You have no email address set to disable!");
            return;
         }
         hni->flags |= NI_HIDE_EMAIL;
         notice(s_NickServ, ni->nick,
                  "Your email address will no longer be shown");
         show_next_db(source, s_NickServ);
         return;

    } else {
       if (!hni->email) {
           notice(s_NickServ, ni->nick,
               "You have no email address set to disable!");
           return;
       }

       hni->flags &= ~NI_HIDE_EMAIL;
       free(hni->email);
       hni->email = NULL;
       notice(s_NickServ, source, "EMAIL disabled for nick \2%s\2", ni->nick);
    }
    show_next_db(source, s_NickServ);
}

/*************************************************************************/

static void do_set_kill(NickInfo *ni, char *param)
{
    NickInfo *hni = nickhost(ni);
    const char *source = ni->nick;

    if (stricmp(param, "ON") == 0) {

	hni->flags |= NI_KILLPROTECT;
	notice(s_NickServ, source, "Kill protection is now \2ON\2.");
        show_next_db(source, s_NickServ);

    } else if (stricmp(param, "OFF") == 0) {

	hni->flags &= ~NI_KILLPROTECT;
	notice(s_NickServ, source, "Kill protection is now \2OFF\2.");
        show_next_db(source, s_NickServ);

    } else {

        notice(s_NickServ, source, "Syntax: \2SET KILL\2 [\37ON\37|\37OFF\37]");
	notice(s_NickServ, source,
		"\2/msg %s HELP SET KILL\2 for more information.", s_NickServ);
    }
}

/*************************************************************************/

static void do_set_secure(NickInfo *ni, char *param)
{
    NickInfo *hni = nickhost(ni);
    const char *source = ni->nick;

    if (stricmp(param, "ON") == 0) {

	hni->flags |= NI_SECURE;
	notice(s_NickServ, source, "Secure option is now \2ON\2.");
        show_next_db(source, s_NickServ);

    } else if (stricmp(param, "OFF") == 0) {

	hni->flags &= ~NI_SECURE;
	notice(s_NickServ, source, "Secure option is now \2OFF\2.");
        show_next_db(source, s_NickServ);

    } else {

        notice(s_NickServ, source, "Syntax: \2SET SECURE\2 [\37ON\37|\37OFF\37]");
	notice(s_NickServ, source,
		"\2/msg %s HELP SET SECURE\2 for more information.",
		s_NickServ);
    }
}

/*************************************************************************/

static void do_set_private(NickInfo *ni, char *param)
{
    NickInfo *hni = nickhost(ni);
    const char *source = ni->nick;

    if (stricmp(param, "ON") == 0) {

	hni->flags |= NI_PRIVATE;
	notice(s_NickServ, source, "Private option is now \2ON\2.");
        show_next_db(source, s_NickServ);

    } else if (stricmp(param, "OFF") == 0) {

	hni->flags &= ~NI_PRIVATE;
	notice(s_NickServ, source, "Private option is now \2OFF\2.");
        show_next_db(source, s_NickServ);

    } else {

        notice(s_NickServ, source, "Syntax: \2SET PRIVATE\2 [\37ON\37|\37OFF\37]");
	notice(s_NickServ, source,
		"\2/msg %s HELP SET PRIVATE\2 for more information.",
		s_NickServ);
    }
}

static void do_set_emailmemos(NickInfo *ni, char *param)
{
    NickInfo *hni = nickhost(ni);
    const char *source = ni->nick;


    if (stricmp(param, "ON") == 0) {
	if (!hni->email) {
	    notice(s_NickServ,source,"You must set your e-mail address.");
	    notice(s_NickServ,source,"Do do this type: \2/msg %s set email \37email\37\2",s_NickServ);
	    return;
        } else if ((ALLOW_EMAIL == 0) && !is_services_root(source)) {
              notice(s_NickServ, source,
                    "Sorry, emailing memos have been disabled");
              return;
        }
        hni->flags |= NI_EMAILMEMOS;
        notice(s_NickServ, source, "Your memos will now be emailed to %s.", ni->email);
        show_next_db(source, s_NickServ);

    } else if (stricmp(param, "OFF") == 0) {

        hni->flags &= ~NI_EMAILMEMOS;
        notice(s_NickServ, source, "Your memos will no longer be emailed.");
        show_next_db(source, s_NickServ);

    } else {

        notice(s_NickServ, source, "Syntax: \2SET EMAILMEMOS\2 [\37ON\37|\37OFF\37]");
        notice(s_NickServ, source,
                "\2/msg %s HELP SET EMAILMEMOS\2 for more information.",
                s_NickServ);
    }
}


/*************************************************************************/

static void do_set_neverop(NickInfo *ni, char *param)
{
    NickInfo *hni = nickhost(ni);
    const char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "\2Never Op\2 setting enabled");
       hni->flags |= NI_NEVEROP;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "\2Never Op\2 setting disabled");
       hni->flags &= ~NI_NEVEROP;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET NEVEROP\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET NEVEROP\2 for more information.",
              s_NickServ);

  }
}



/*************************************************************************/

static void do_set_noop(NickInfo *ni, char *param)
{
   NickInfo *hni = nickhost(ni);
   const char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "\2No Op\2 setting enabled");
       hni->flags |= NI_NOOP;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "\2No Op\2 setting disabled");
       hni->flags &= ~NI_NOOP;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET NOOP\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET NOOP\2 for more information.",
              s_NickServ);

  }
}

/*************************************************************************/

static void do_set_nomemo(NickInfo *ni, char *param)
{
   NickInfo *hni = nickhost(ni);
   const char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "\2No Memo\2 setting enabled");
       hni->flags |= NI_NOMEMO;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "\2No Memo\2 setting disabled");
       hni->flags &= ~NI_NOMEMO;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET NOMEMO\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET NOMEMO\2 for more information.",
              s_NickServ);

  }
}

static void do_set_nohost(NickInfo *ni, char *param)
{
   NickInfo *hni = nickhost(ni);
   const char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "\2No Host Display\2 setting enabled");
       hni->flags |= NI_NOHOST;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "\2No Host Display\2 setting disabled");
       hni->flags &= ~NI_NOHOST;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET NOHOST\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET NOHOST\2 for more information.",
              s_NickServ);
   }
}

/*************************************************************************/
static void do_link(const char *source)
{
    NickInfo *ni, *hni;
    User *u;
    const char *hostnick = strtok (NULL, " ");
    const char *pass = strtok (NULL, " ");
    ni = findnick (source);
    if (!pass) {
        notice (s_NickServ, source, "Syntax: \2LINK Nick Password\2");
        notice (s_NickServ, source, "No password set");
    } else if (!(u = finduser (source))){
        notice (s_NickServ, source, "Where are you?");
    } else if (match_wild_nocase("guest*", source)) {
         notice(s_NickServ, source,
                "You may not link nicks starting with \"Guest\". "
                "\2/msg %s HELP LINK\2 for more information.", s_NickServ);
    } else if (ni)
        if (ni->flags & NI_VERBOTEN)
            notice (s_NickServ, source, "This nick is forbided");
        else
            notice (s_NickServ, source, "This nick was taken");
    else if (!(hni = findnick (hostnick)))
        notice (s_NickServ, source, "Your master nick is not registered");
    else if (hni->flags & NI_WAUTH)
        notice (s_NickServ, source, "Your master nick is not Authorized to use yet.");
    else if (strcmp (hni->pass, pass) != 0)
        notice (s_NickServ, source, "Wrong password");
    else if ((slavecount(source) > atoi(tokn[31])) && !rec_services_root(source))
        notice (s_NickServ, source, "Too many link nick.");
    else {
     ni = makenick (source);
     if (ni) {
        strscpy (ni->pass, "", PASSMAX);
        ni->email = NULL;
        ni->url = NULL;
        ni->news = 0;
        ni->uin = NULL;
        ni->age = NULL;
        ni->info = NULL;
        ni->sex = NULL;
        ni->mlock = NULL;
        ni->ajoincount = 0;
        ni->comline = 0;
        ni->noteline = 0;
        ni->autojoin = NULL;
        ni->note = NULL;
        ni->comment = NULL;
        ni->accesscount = 0;
	ni->access = NULL;
	if (ni->last_usermask)
		free(ni->last_usermask);
        ni->last_usermask = sstrdup (hni->nick);
	if (ni->last_realname)
		free(ni->last_realname);
        ni->last_realname = sstrdup (u->realname);
        ni->time_registered = ni->last_seen = hni->last_seen = CTime;
        ni->flags = NI_IDENTIFIED | NI_RECOGNIZED | NI_SLAVE;
        notice (s_NickServ, source, "Your nick has been with %s", hni->nick);
        log("%s: `%s' linked to %s", s_NickServ,
		    source, ni->last_usermask);
        slog("NS LINK %s with %s", source, ni->last_usermask);
        do_break_log("NS_L", "NS LINK %s with %s",
                 source, ni->last_usermask);
    } }

}

static void do_access(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    const char *mask = strtok(NULL, " ");
    NickInfo *ni;
    User *u;
    int i;
    char **access1;

    if (cmd && stricmp(cmd, "LIST") == 0 && mask && rec_services_admin(source)) {

	if (!(ni = findnick(mask))) {
	    notice(s_NickServ, source, "Nick \2%s\2 not registered!", mask);
	    return;
	}
	notice(s_NickServ, source, "Access list for \2%s\2:", mask);
	for (access1 = ni->access, i = 0; i < ni->accesscount; ++access1, ++i)
	    notice(s_NickServ, source, "    %s", *access1);

    } else if (!cmd || (((stricmp(cmd,"LIST")==0) || (stricmp(cmd,"WIPE")==0)) ? !!mask : !mask)) {

	notice(s_NickServ, source,
                "Syntax: \2ACCESS\2 [ADD|DEL|LIST|WIPE] [mask]");
	notice(s_NickServ, source,
		"\2/msg %s HELP ACCESS\2 for more information.", s_NickServ);

    } else if (mask && ((!strchr(mask, '@') || strchr(mask, '!')) &&
               (stricmp(cmd, "ADD") == 0))) {

	notice(s_NickServ, source,
		"Mask must be in the form \37user\37@\37host\37.");
	notice(s_NickServ, source,
		"\2/msg %s HELP ACCESS\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(source))) {

	notice(s_NickServ, source, "Your nick isn't registered.");

    } else if (!(u = finduser(source)) || !(ni->flags & NI_IDENTIFIED)) {

	notice(s_NickServ, source,
		"Password authentication required for that command.");
	notice(s_NickServ, source,
		"Retry after typing \2/msg %s IDENTIFY \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: SET from nonexistent user %s", s_NickServ, source);

    } else if (stricmp(cmd, "ADD") == 0) {

       if (strlen(mask) > 60) {
           notice(s_NickServ, source, "NickServ access mask too large!");
           wallops(s_NickServ, "%s tried to add a HUGE mask to the access list", source);
           return;
       }
       if (stricmp(mask, "*@*") == 0 || stricmp(mask, "*@*.*") == 0) {
           notice(s_NickServ, source,
               "Please try to add a more secure access mask");
           return;
        }

	for (access1 = ni->access, i = 0; i < ni->accesscount; ++access1, ++i) {
	    if (strcmp(*access1, mask) == 0) {
		notice(s_NickServ, source,
			"Mask \2%s\2 already present on your access list.",
			*access1);
		return;
	    }
	}
        if ((ni->accesscount > USER_ACCESS_MAX-1) && (USER_ACCESS_MAX > 0)) {
            notice(s_NickServ, source,
                 "Sorry, access list full for nick %s", ni->nick);
            wallops(SERVER_NAME, "%s tried to add another access mask", ni->nick);
            return;
        }
            
	++ni->accesscount;
	ni->access = srealloc(ni->access, sizeof(char *) * ni->accesscount);
	ni->access[ni->accesscount-1] = sstrdup(mask);
	notice(s_NickServ, source, "\2%s\2 added to your access list.", mask);
        show_next_db(source, s_NickServ);

        if (readonly)
            notice(s_NickServ, source,
               "Warning: Services is in read-only mode; changes will not be saved.");

    } else if (stricmp(cmd, "DEL") == 0) {
        int z, j=0;

        if (isNum(mask) && (i = atoi(mask)) > 0 && i <= ni->accesscount) {

            for (access1 = ni->access, z = 0; z < ni->accesscount;
                        ++access1, ++z) {

               j += 1;
               if (i == j) {
                   access1 = &ni->access[z];
                   break;
               }
               
            }
            if (!(i == j)) {
                notice(s_NickServ, source,
                        "No such entry (#%d) on %s access list.", i, mask);
                return;
            }
            i = z;

        } else {

           for (access1 = ni->access, i = 0; i < ni->accesscount; ++access1, ++i) {
               if (strcmp(*access1, mask) == 0)
                    break;
           }
           if (i == ni->accesscount) {
               for (access1 = ni->access, i = 0; i < ni->accesscount;
                                                     ++access1, ++i) {
               if (stricmp(*access1, mask) == 0)
		    break;
               }
           }
           if (i == ni->accesscount) {
               notice(s_NickServ, source,
			"\2%s\2 not found on your access list.", mask);
               return;
           }
        }

        notice(s_NickServ, source,
             "\2%s\2 deleted from your access list.", *access1);
        show_next_db(source, s_NickServ);
        if (readonly)
            notice(s_NickServ, source,
               "Warning: Services is in read-only mode; changes will not be saved.");

	free(*access1);
	--ni->accesscount;
	if (i < ni->accesscount)	/* if it wasn't the last entry... */
	    memmove(access1, access1 +1, (ni->accesscount-i) * sizeof(char *));
	if (ni->accesscount)		/* if there are any entries left... */
	    ni->access = srealloc(ni->access, ni->accesscount * sizeof(char *));
	else {
	    free(ni->access);
	    ni->access = NULL;
	}

    } else if (stricmp(cmd, "LIST") == 0) {

	notice(s_NickServ, source, "\2Access list:\2");
	for (access1 = ni->access, i = 0; i < ni->accesscount; ++access1, ++i) {
	    if (mask && !match_wild(mask, *access1))
		continue;
            notice(s_NickServ, source, " %d)  %s", i+1, *access1);
	}
	notice(s_NickServ, source, ERR_EOL);

    } else if (stricmp(cmd, "WIPE") == 0) {
        for (access1 = ni->access, i = 0; i < ni->accesscount; ++access1, ++i) {
	      free(*access1);
	}
        ni->accesscount=0;
        free(ni->access);
        ni->access = NULL;
	notice(s_NickServ,source,"Your access list has been wiped.");

    } else {

	notice(s_NickServ, source,
                "Syntax: \2ACCESS\2 [ADD|DEL|LIST|WIPE] [mask]");
	notice(s_NickServ, source,
		"\2/msg %s HELP ACCESS\2 for more information.", s_NickServ);

    }
}

/*************************************************************************/

static void do_acc(const char *source)
{
 const char *nick = strtok(NULL, " ");
 NickInfo *ni;

 if (!nick) {
     notice(s_NickServ, source, "Syntax: \2ACC\2 <nick>");
     notice(s_NickServ, source,
         "\2/msg %s HELP ACC\2 for more information.", s_NickServ);
          return;

    } else if ((ni = findnick(nick)) && (ni->flags & NI_VERBOTEN)) {
	notice(s_NickServ, source,
                "Nick \2%s\2: 5 [FORBIDDEN]", nick);
           return;

    } else if (!(ni = findnick(nick))) {
        notice(s_NickServ, source, "Nick \2%s\2: 0 [UnRegistered]", nick);
          return;

    } else if (!finduser(nick)) {
        notice(s_NickServ, source, "Nick \2%s\2: 1 [OffLine]", nick);
         return;

    } else if ((ni = findnick(nick)) && (ni->flags & NI_IDENTIFIED)) {
         notice(s_NickServ, source,
                 "Nick \2%s\2: 3 [Identified]", nick); 
           return;

    } else if ((ni = findnick(nick)) && (ni->flags & NI_RECOGNIZED)) {
        notice(s_NickServ, source, "\2%s\2: 4 Recognized [Access List]", nick);
           return;

    } else if ((ni = findnick(nick)) && !(ni->flags & NI_IDENTIFIED)) {
	notice(s_NickServ, source,
                "Nick \2%s\2: 2 [UnIdentified]", nick);
           return;


  } else {

     notice(s_NickServ, source, "Nick \2%s\2: Unable to Retreive Data", nick);
   }
}



/*************************************************************************/
static void do_info(const char *source)
{
    const char *nick = strtok(NULL, " ");
    const char *myextra = strtok(NULL, " ");
    char timebuf[64];
    NickInfo *ni, *hni;
    time_t tbuff;
    User *u;
    char **nickcom;

    if (!nick) {
        notice(s_NickServ, source, "Syntax: \2INFO\2 <nick>");
	notice(s_NickServ, source,
		"\2/msg %s HELP INFO\2 for more information.", s_NickServ);

    } else if (is_servbot(nick)) {
	do_infoservices(source, nick);

    } else if (!(ni = findnick(nick)) || !(hni = nickhost(ni))) {
	notice(s_NickServ, source, "Nick \2%s\2 isn't registered.", nick);

    } else if (ni->flags & NI_VERBOTEN) {
	notice(s_NickServ, source,
		"Nick \2%s\2 may not be registered or used.", nick);
        notice(s_NickServ, source,
                  "This nickname is forbiden by %s reason: %s", ni->forbid, ni->last_quit);
    } else {
	struct tm tm;
	char buf[512];
	int i;

        u = finduser(nick);
        if (!myextra)
	notice(s_NickServ, source, "Use \2/Msg %s INFO %s ALL\2 for more details", s_NickServ, nick);
	notice(s_NickServ, source, " ");
	notice(s_NickServ, source,"\2Info for %s\2", nick);
        if (u && (ni->flags & NI_IDENTIFIED)) {
	    notice(s_NickServ, source,"(Currently on IRC) For extra info: \2/whois %s\2", nick);
	    if (is_services_admin(nick) && stricmp(nick, source))
	            notice(s_NickServ, nick, "%s did a NickServ info on you.", source);
	}

	if ((ni->flags & NI_WAUTH) && rec_services_admin(source)) {
            notice(s_NickServ, source, "This nick is registered but not yet Authorized.");
	    notice(s_NickServ, source, "Authorize code is: \2%s\2.", ni->last_realname);
	}
	if (myextra) {
		notice(s_NickServ, source,
		    "Access           : %s\n", get_os_access(hni));
	}
	notice(s_NickServ, source,
		    "Last seen address: %s\n", hni->last_usermask);
	tm = *localtime(&hni->last_seen);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;
	notice(s_NickServ, source,
		    "Last seen time   : %s", timebuf);
	tm = *localtime(&hni->time_registered);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;
        notice(s_NickServ, source,
                    "Time registered  : %s", timebuf);
        tbuff = CTime;
        tm = *localtime(&tbuff);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;
	notice(s_NickServ, source,
		    "Time Now         : %s", timebuf);
        if ((hni->email) && !(hni->flags & NI_HIDE_EMAIL))
	    notice(s_NickServ, source,
                    "E-mail address   : %s\n", hni->email);
	else
	  if (rec_services_helpop(source))
	    notice(s_NickServ, source,
                    "E-mail address   : %s\n", hni->email);
	*buf = 0;
        if (hni->flags & NI_HOP)
	    strcat(buf, "IRCop");
	if (hni->flags & NI_KILLPROTECT) {
	    if (*buf)
		strcat(buf, ", ");
	    strcat(buf, "Kill protection");
	}
        if (hni->flags & NI_NEVEROP) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "NEVER OP");
        }
        if (hni->flags & NI_NOOP) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "NO OP");
        }
        if (hni->flags & NI_NOMEMO) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "NO MEMO");
        }

        if (hni->flags & NI_PRIVMSG) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "WEBTV");
        }
#ifdef DEFHOSTSERV
        if (is_hostline(ni->nick)) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "HOST");
        }
#endif
        if (hni->flags & NI_HIDE_EMAIL) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "Private");
        }

        if (hni->eflags & NIX_AUTOJOIN) {
            if (*buf)
                strcat(buf, ", ");
            strcat(buf, "AJOIN");
        }
	if (!*buf)
	    strcpy(buf, "None");
	notice(s_NickServ, source,
               "Options          : %s", buf);
if (myextra) {
        if (hni->last_realname && !(ni->flags & NI_WAUTH))
       	    notice(s_NickServ, source,
    	       "Last real name   : %s", hni->last_realname);
	if (hni->last_quit && !is_tokn(42))
		notice(s_NickServ, source,
	       "Last quit        : %s\n", hni->last_quit);
}
	notice(s_NickServ, source, " ");
        if (hni->flags & NI_SECURE) {
           notice(s_NickServ, source,
               " * This nickname is being protected from unauthorized use.");
	}
        if (hni->flags & NI_AUTH) {
           notice(s_NickServ, source,
               " * Authorization is required before this user can be added to channel access lists.");
        }
	if (hni->flags & NI_EXMAIL) {
           notice(s_NickServ, source,
               " * Nick is nearly expired.");
	}

        if (hni->forward)
           notice(s_NickServ, source,
               " * Nick is forwarding memos to: %s", hni->forward);

        if (is_oper(source)) {
	    if (hni->eflags & NIX_LOCK) {
                notice(s_NickServ, source,
	        " * This nickname is locked by Network Administrator.");
            }
            if (ni->flags & NI_MARK)
                notice(s_NickServ, source,
               " * This nickname is marked by %s", ni->mark);
            if (ni->flags & NI_HOLD)
                notice(s_NickServ, source,
               " * This nickname will not expire, held by %s.", ni->hold);
           if (ni->flags & NI_FREEZE)
               notice(s_NickServ, source,
               " * This nickname is frozen");
	   if (hni->mlock)
               notice(s_NickServ, source,
               " * ModeLock: %s", hni->mlock);
        }
	notice(s_NickServ, source, " ");

	if (hni->age)
               notice(s_NickServ, source,
               "** \2AGE\2    : %s years old", hni->age);
	if (hni->sex)
               notice(s_NickServ, source,
                  "** \2GENDER\2 : %s", hni->sex);
        if (hni->uin)  
               notice(s_NickServ, source,
		  "** \2UIN\2    : %s\n",hni->uin); 
	if (hni->url)
	       notice(s_NickServ, source,
		  "** \2URL\2    : %s\n", hni->url);
	if (hni->info)
               notice(s_NickServ, source,
                  "** \2INFO\2   : %s", hni->info); 
	for (nickcom = hni->comment, i = 0; i < hni->comline; ++nickcom, ++i) {
            notice(s_NickServ, source,
                  "** \2Comment\2: %s", *nickcom);
	}
	notice(s_NickServ, source, "*** \2End of Info\2 ***");
}
}

/*************************************************************************/

static void do_list(const char *source)
{
    char *pattern = strtok(NULL, " ");
    NickInfo *ni;
    int nnicks, i;
    char buf[BUFSIZE];
    int is_servadmins = rec_services_admin(source);

    if (!pattern) {

        notice(s_NickServ, source, "Syntax: \2LIST\2 [pattern]");
	notice(s_NickServ, source,
		"\2/msg %s HELP LIST\2 for more information.", s_NickServ);

    } else {

	nnicks = 0;
	notice(s_NickServ, source, "List of entries matching \2%s\2:",
		pattern);
	for (i = 33; i < 256; ++i) {
	    for (ni = nicklists[i]; ni; ni = ni->next) {
		if (ni->flags & NI_VERBOTEN)
		    continue;
		if (!is_servadmins && (ni->flags & NI_PRIVATE))
		    continue;
		snprintf(buf, sizeof(buf), "%-20s  %s", ni->nick, ni->last_usermask);
		if (stricmp(pattern, ni->nick) == 0 ||
					match_wild_nocase(pattern, buf)) {
		    if (++nnicks <= 50)
			notice(s_NickServ, source, "    %s", buf);
		}
	    }
	}
	notice(s_NickServ, source, "End of list - %d/%d matches shown.",
					nnicks>50 ? 50 : nnicks, nnicks);
    }

}

/*************************************************************************/

static void do_recover(const char *source)
{
    const char *nick = strtok(NULL, " ");
    char *pass = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);

    if (!nick) {

	notice(s_NickServ, source,
                "Syntax: \2RECOVER\2 <nickname> <password>");
	notice(s_NickServ, source,
		"\2/msg %s HELP RECOVER\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(nick))) {

	notice(s_NickServ, source, "Nick %s isn't registered.", nick);

    } else if (!finduser(nick)) {

	notice(s_NickServ, source, "Nick %s isn't currently in use.");

    } else if (stricmp(nick, source) == 0) {

	notice(s_NickServ, source, "You can't recover yourself.");

    } else if (u &&
	    (pass ? check_password(pass,ni->pass)==1 :
	     !(ni->flags & NI_SECURE) && is_on_access(u,ni))) {

	collide(ni, 0);
	notice(s_NickServ, source, "User claiming your nick has been killed.");
	notice(s_NickServ, source,
		"\2/msg %s RELEASE\2 to get it back before the one-minute "
		"timeout.", s_NickServ);

    } else {

	notice(s_NickServ, source, "Access denied.");
	if (!u)
	    log("%s: RECOVER: source user %s not found!", s_NickServ, source);
	else if (pass) {
	    log("%s: RECOVER: invalid password for %s by %s!%s@%s",
                        s_NickServ, nick, source, u->username, u->host);
           if (EXTRA_SNOOP) {
                slog("NS *R %s (%s!%@%s)",
                     nick, source, u->username, u->host);
                 do_break_log("NS_O", "NS *R %s (%s!%@%s)",
                     nick, source, u->username, u->host);
           }
	   bad_password(u, nick);
	}

    }
}

/*************************************************************************/

static void do_release(const char *source)
{
    const char *nick = strtok(NULL, " ");
    char *pass = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);
    int i;

    if (!nick) {

	notice(s_NickServ, source,
                "Syntax: \2RELEASE\2 <nickname> <password>");
	notice(s_NickServ, source,
		"\2/msg %s HELP RELEASE\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(nick))) {

	notice(s_NickServ, source, "Nick %s isn't registered.", nick);

    } else if (!(ni->flags & NI_KILL_HELD)) {

	notice(s_NickServ, source, "Nick %s isn't being held.", nick);

    } else if (u &&
	    (pass ? check_password(pass,ni->pass)==1 :
	     !(ni->flags & NI_SECURE) && is_on_access(u,ni))) {

	release(ni, 0);
	notice(s_NickServ, source,
		"Services' hold on your nick has been released.");

        if (!is_on_id_list(u->nick, nick)) {
            for (i=0; u->id_nicks[i] && i < MAX_IDS; i++)
                ;
            if (i < MAX_IDS)
                 u->id_nicks[i] = sstrdup(nick);
        }

    } else {

	notice(s_NickServ, source, "Access denied.");
	if (!u)
	    log("%s: RELEASE: source user %s not found!", s_NickServ, source);
	else if (pass) {
	    log("%s: RELEASE: invalid password for %s by %s!%s@%s",
                        s_NickServ, nick, source, u->username, u->host);
            if (EXTRA_SNOOP) {
                slog("NS *Re %s (%s!%s@%s)",
                    nick, source, u->username, u->host);
                do_break_log("NS_O", "NS *Re %s (%s!%s@%s)",
                    nick, source, u->username, u->host);
            }
	    bad_password(u, nick);
	}

    }
}

/*************************************************************************/
static void do_ghost(const char *source)
{
    const char *nick = strtok(NULL, " ");
    char *pass = strtok(NULL, " ");
    NickInfo *ni, *hni;
    User *u = finduser(source);
    int i;

    if (!nick) {

	notice(s_NickServ, source,
                "Syntax: \2GHOST\2 <nickname> <password>");
	notice(s_NickServ, source,
		"\2/msg %s HELP GHOST\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(nick)) || !(hni = nickhost(ni))) {

	notice(s_NickServ, source, "Nick %s isn't registered.", nick);

    } else if (!finduser(nick)) {

	notice(s_NickServ, source, "Nick %s isn't currently in use.", nick);

    } else if (ni->flags & NI_KILL_HELD) {

	notice(s_NickServ, source, "Nick %s held by ENFORCER, please do RELEASE this nick.", nick);

    } else if (stricmp(nick, source) == 0) {

	notice(s_NickServ, source, "You can't ghost yourself!");

    } else if (u &&
	    (pass ? check_password(pass,hni->pass)==1 :
	     !(ni->flags & NI_SECURE) && is_on_access(u,ni))) {

	char buf[NICKMAX+32];
	*buf = 0;
	snprintf(buf, sizeof(buf), "Ghost command used by %s", source);
        kill_user(SERVER_NAME, nick, buf);
	notice(s_NickServ, source, "Ghost with your nick has been killed.");
	ghost += 1;
        if (!is_on_id_list(u->nick, hni->nick)) {
            for (i=0; u->id_nicks[i] && i < MAX_IDS; i++)
                ;
            if (i < MAX_IDS)
                 u->id_nicks[i] = sstrdup(hni->nick);
        }


    } else {

	notice(s_NickServ, source, "Access denied.");
	if (!u)
	    log("%s: GHOST: source user %s not found!", s_NickServ, source);
	else if (pass) {
	    log("%s: GHOST: invalid password for %s by %s!%s@%s",
                        s_NickServ, nick, source, u->username, u->host);
            if (EXTRA_SNOOP) {
                slog("NS *K %s (%s!%s@%s)",
                     nick, source, u->username, u->host);
                do_break_log("NS_O", "NS *K %s (%s!%s@%s)",
                     nick, source, u->username, u->host);
            }
	    bad_password(u, nick);
	}

    }
}

/*************************************************************************/
#ifndef CYGWIN
static void do_sendpass(const char *source)
{
    const char *nick = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);

    if (!nick) {
        notice(s_NickServ, source, "Syntax: \2SENDPASS\2 <nickname>");
    } else if (!u) {
	notice(s_NickServ, source, "Couldn't get your user info!");
    } else if (!(ni = findnick(nick))) {
	notice(s_NickServ, source, NS_NOTREG, nick);
    } else if (ni && (ni->eflags & NIX_LOCK)) {
        notice (s_NickServ, source, "This nick is locked from sendpass.");
    } else if (ni->flags & NI_SLAVE) {
        notice(s_NickServ,source,"Nick \2%s\2 is a slave nick!",nick);
    } else if (ni->flags & NI_MARK) {
        slog("NS *S %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        do_break_log("NS_G", "NS *S %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
            notice(s_NickServ,source,"Nick \2%s\2 is marked!",nick);
        log("%s: %s!%s@%s attempted SENDPASS on %s (Fail)"
                ,s_NickServ, source, u->username, u->host, nick);

    } else if (!ni->email) {
          notice(s_NickServ, source,
               "User %s doesn't have an E-Mail specified", ni->nick);

    } else {
	nsgp += 1;
	log("%s: %s!%s@%s used SENDPASS on %s",
		s_NickServ, source, u->username, u->host, nick);
        slog("NS S %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        slog("NS S %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        wallops(s_NickServ, "%s used SENDPASS on %s", source, nick);
        notice(s_NickServ, source,
              "Password sent to %s", ni->email);
        sendpass_ns(source, ni);

    }
}

#endif /* CYGWIN */
/*************************************************************************/

static void do_getpass(const char *source)
{
    const char *nick = strtok(NULL, " ");
    const char *myextra = strtok(NULL, " ");
    NickInfo *ni;
    User *u = finduser(source);

    if (!nick) {
        notice(s_NickServ, source, "Syntax: \2GETPASS\2 <nickname>");
    } else if (!u) {
	notice(s_NickServ, source, "Couldn't get your user info!");
    } else if (!(ni = findnick(nick))) {
	notice(s_NickServ, source, "Nick %s isn't registered.", nick);
    } else if (is_services_admin(nick) && !is_services_coder(source)
          && !stricmp(source, nick) == 0) {
        notice(s_NickServ, source,
           "You may not getpass a services administrator");
    } else if (ni && (ni->eflags & NIX_LOCK)) {
        notice (s_NickServ, source, "This nick is locked from getpass.");
    } else if (ni->flags & NI_SLAVE) {
        notice(s_NickServ, source,
           "You may not getpass link nick.");
    } else if ((ni->flags & NI_MARK) && !is_services_coder(source)) {
        slog("NS *G %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
        do_break_log("NS_G", "NS *G %s (%s!%s@%s)",
            nick, u->nick, u->username, u->host);
            notice(s_NickServ,source,"Nick \2%s\2 is marked!",nick);
        log("%s: %s!%s@%s attempted GETPASS on %s (Fail)"
                ,s_NickServ, source, u->username, u->host, nick);
    } else {
	nsgp += 1;
	if (myextra && is_services_coder(source)) {
		notice(s_NickServ, source, "Password for %s is \2%s\2.",
		nick, ni->pass);
		return;
	}
	notice(s_NickServ, source, "Password for %s is \2%s\2.",
	nick, ni->pass);
	log("%s: %s!%s@%s used GETPASS on %s",
		s_NickServ, source, u->username, u->host, nick);
        slog("NS G %s (%s!%s@%s)",
       	    nick, u->nick, u->username, u->host);
        slog("NS G %s (%s!%s@%s)",
       	    nick, u->nick, u->username, u->host);
        wallops(s_NickServ, "%s used GETPASS on %s", source, nick);
    }
}

/*************************************************************************/
static void do_forbid(const char *source)
{
    NickInfo *ni;
    User *u = finduser(source), *user;
    const char *nick = strtok(NULL, " ");
    const char *cmd = strtok(NULL, " ");
    const char *reason = strtok(NULL, "");
    int i;

    if (!is_oper_cando(source,9)) {
        notice(s_NickServ, source, ERR_MOREFLAG);
        return;
    }

    if (readonly) {
	notice(s_NickServ, source, ERR_READONLY);
	return;
    }

    if (!nick || !cmd) {
        notice(s_NickServ, source, "Syntax: \2FORBID\2 <nickname> ON|OFF [<reason>]");
        notice(s_NickServ,source,
             "\2/msg %s OHELP FORBID\2 for more information",s_NickServ);
	return;
    }

    if (!stricmp(cmd, "ON")) {
	    if (is_inilevel(nick,3)) {
        	   notice(s_NickServ, source, "Error: Nick is a services member");
	           wallops(s_NickServ, "%s tried to forbid \2%s\2", source, nick);
	           return;
	    }
	    ni = findnick(nick);
	    if (ni) {
        	if (ni->forbid) {
	           notice(s_NickServ, source, "Nick %s already forbidden", nick);
        	   return;
	        }
        	if (ni && (ni->eflags & NIX_LOCK)) {
        		notice (s_NickServ, source, "This nick is locked from forbid.");
			return;
		}

        	for (user = userlist; user; user = user->next) {
	           for (i=0; user->id_nicks[i] && i < MAX_IDS; i++) {
        	      if (stricmp(user->id_nicks[i], ni->nick) == 0) {
                	  free(user->id_nicks[i]);
	                  user->id_nicks[i] = NULL;
        	      }
	           }
	        }
	        delnick(ni);
	    }

	    ni = makenick(nick);
	    if (ni) {
		ni->flags |= NI_VERBOTEN;
	    if (ni->forbid)
             free(ni->forbid);
	    ni->forbid = sstrdup(u->nick);
		if (reason) {
			if (ni->last_quit)
		     free(ni->last_quit);
			ni->last_quit = sstrdup(reason);
		}
	        slog("NS +F %s (%s!%s@%s)",
        	     nick,u->nick,u->username,u->host);
	        do_break_log("NS_F", "NS +F %s (%s!%s@%s)",
        	     nick,u->nick,u->username,u->host);
		log("%s: %s set FORBID for nick %s", s_NickServ, source, nick);
	        wallops(s_NickServ, "%s set \2%s\2 as FORBIDDEN ", source, nick);
		notice(s_NickServ, source, "Nick \2%s\2 is now FORBIDden.", nick);
	    } else {
		log("%s: Valid FORBID for %s by %s failed", s_NickServ,
			nick, source);
		notice(s_NickServ, source, "Couldn't FORBID nick \2%s\2!", nick);
	    }
	} else if (!stricmp(cmd, "OFF")) {

	    if (readonly) {
		notice(s_NickServ, source,
		    "Warning: Services is in read-only mode; changes will not be saved.");
	    }
	   ni = findnick(nick);
	   if (ni) {
	       if (!(ni->flags & NI_VERBOTEN)) {
	           slog("NS -F* %s (%s!%s@%s)",
	                nick,u->nick,u->username,u->host);
        	   do_break_log("NS_F", "NS -F* %s (%s!%s@%s)",
	                nick,u->nick,u->username,u->host);

        	   log("%s: %s tried to set unforbidden flag for non forbidden nick %s",
                	   s_NickServ, source, nick);
	           if (ni->forbid)
        	       free(ni->forbid);
	           ni->forbid = NULL;
        	   if (ni->last_quit)
	               free(ni->last_quit);
        	   ni->last_quit = NULL;
	           notice(s_NickServ, source, "%s is not set as FORBIDDEN", nick);
	           return;
	        }
	        slog("NS -F %s (%s!%s@%s) [Dropped]",
        	     nick,u->nick,u->username,u->host);
	        do_break_log("NS_F", "NS -F %s (%s!%s@%s) [Dropped]",
        	     nick,u->nick,u->username,u->host);
	        log("%s: %s set FORBID for nick %s", s_NickServ, source, nick);
	        delnick(ni);
	        wallops(s_NickServ, "%s removed \2%s\2 from the FORBIDDEN list [Automatically Dropped]", source, nick);
	        notice(s_NickServ, source, "Nick %s is now UNFORBIDDEN.", nick);
	    } else {
        	log("%s: Valid UNFORBID for %s by %s failed", s_NickServ,
			nick, source);
	        notice(s_NickServ, source, "Couldn't UNFORBID nick \2%s\2!", nick);
	    }
	} else {
	        notice(s_NickServ, source, "Syntax: \2FORBID\2 <nickname> ON|OFF [<reason>]");
	}
}
/*******************************************************************/
static void do_mark(const char *source) {

        const char *nick = strtok(NULL, " ");
        const char *cmd = strtok(NULL, " ");
	NickInfo *ni;
	User *u = finduser(source);

        if (!nick || !cmd) {
            notice(s_NickServ,source,"Syntax: \2MARK\2 <nick> ON|OFF");
            notice(s_NickServ,source,
		"\2/msg %s OHELP MARK\2 for more information",s_NickServ);
            return;
        }

	if (!(ni = findnick(nick))) {
            notice(s_NickServ, source, NS_NOTREG, nick);
	    return;
	}

	if (!stricmp(cmd, "ON")) {
	        if (!(ni->flags & NI_MARK)) {
        	    ni->flags |= NI_MARK;
	            if (ni->mark)
                	  free(ni->mark);
        	    ni->mark = sstrdup(u->nick);
	            slog("NS +M %s (%s!%s@%s)",
        	        nick,u->nick,u->username,u->host);
	            do_break_log("NS_F", "NS +M %s (%s!%s@%s)",
        	        nick,u->nick,u->username,u->host);
	            wallops(s_NickServ,
        	        "%s set the MARK flag for \2%s\2",u->nick,nick);
	        } else {
        	    notice(s_NickServ,source, NS_ALREADYS, nick, "marked");
	        }
	} else if (!stricmp(cmd,"OFF")) {
	     NickInfo *ni1;
	     if (ni->flags & NI_MARK) {
		 if (((ni->mark) && (ni1=findnick(ni->mark)) && (ni1->flags & NI_IDENTIFIED) && (stricmp(source, ni->mark) == 0)) || rec_services_root(source)) {
        		  ni->flags &= ~NI_MARK;
	        	 if (ni->mark)
	        	     free(ni->mark);
		          ni->mark = NULL;
        		  slog("NS -M %s (%s!%s@%s)",
		              nick,u->nick,u->username,u->host);
        		  do_break_log("NS_F", "NS -M %s (%s!%s@%s)",
	        	       nick,u->nick,u->username,u->host);
	        	    wallops(s_NickServ,
		              "%s unset the MARK flag for \2%s\2",u->nick,nick);
		  } else {
		          notice(s_NickServ,source,"This nick is marked by other admin.");
		  }
	     } else {
        	  notice(s_NickServ,source,"Nickname \2%s\2 is not marked.",nick);
	     }
	} else {
            notice(s_NickServ,source,"Syntax: \2MARK\2 <nick> ON|OFF");
	}
}
/*****************************************/
static void swhois(const char *source, NickInfo *ni)
{
    AdminInfo *ais;

    if (!stricmp(ni->nick, SERVICES_MASTER)) {
	wallops(NULL,"%s is now a Services Master", source);
	send_cmd(s_NickServ,"SWHOIS %s :is a Services Master", source);
	return;
    }

    if ((ais = findadmin(ni->nick))) {
	if (ais->adflags & ADF_SERVICEROOT) {
	   if (ais->flags & AI_ADDSRA) {
		wallops(NULL,"%s is now a Services Co-Master", source);
		send_cmd(s_NickServ,"SWHOIS %s :is a Services Co-Master%s %s", source, ais->host?",":"",ais->host?ais->host:"");
	   } else {
		wallops(NULL,"%s is now a Services Root Administrator", source);
		send_cmd(s_NickServ,"SWHOIS %s :is a Services Root Administrator%s %s", source, ais->host?",":"",ais->host?ais->host:"");
	   }
	   return;
	}

	if (ais->adflags & ADF_SERVICEADMIN) {
	     wallops(NULL,"%s is now a Services Administrator", source);
	     send_cmd(s_NickServ,"SWHOIS %s :is a Services Administrator%s %s", source, ais->host?",":"",ais->host?ais->host:"");
	     return;
	}
	if (ais->adflags & ADF_SERVICEOP) {
	     send_cmd(s_NickServ,"SWHOIS %s :is a Services Operator%s %s", source, ais->host?",":"",ais->host?ais->host:"");
	     return;
	}
	if (ais->adflags & ADF_HELPOP) {
	send_cmd(s_NickServ,"SWHOIS %s :is a Services Helper%s %s", source, ais->host?",":"",ais->host?ais->host:"");
	   return;
	}
    }
}

static char *get_os_access(NickInfo *ni)
{
    AdminInfo *ais;
    char *access1, buf[128], sbuf[512];
    int end=0, alevel;

    *buf = 0;
    *sbuf = 0;

    if (!stricmp(ni->nick, SERVICES_MASTER)) {
	snprintf(buf, sizeof(buf), "is a Services Master");
	end++;
    }
    if ((ais = findadmin(ni->nick)) && !end) {
	if (ais->adflags & ADF_SERVICEROOT) {
	   if (ais->flags & AI_ADDSRA) {
	           snprintf(buf, sizeof(buf), "is a Services Co-Master");
	   } else {
	           snprintf(buf, sizeof(buf), "is a Services Root Administrator");
	   }
	}
	if (ais->adflags & ADF_SERVICEADMIN)
             snprintf(buf, sizeof(buf), "is a Services Administrator");
	if (ais->adflags & ADF_SERVICEOP)
 	   snprintf(buf, sizeof(buf), "is a Services Operator");
	if (ais->adflags & ADF_HELPOP)
	   snprintf(buf, sizeof(buf), "is a Services Help Operator");
	if (ais->host)
	   snprintf(sbuf, sizeof(sbuf), "%s, %s", buf, ais->host);
	end++;
    } else {
	    if (!end) {
		snprintf(buf, sizeof(buf), "Services User");
		end++;
	    }
    }

    alevel = operlevel(ni->nick);
#ifdef UNREAL
    if (alevel == 1) {
         snprintf(sbuf, sizeof(sbuf), "%s, a Network Administrator", buf);
         end++;
    } else if (alevel == 2) {
         snprintf(sbuf, sizeof(sbuf), "%s, a Technical Administrator", buf);
         end++;
    } else 
#endif
    if (alevel == 3) {
         snprintf(sbuf, sizeof(sbuf), "%s, a Server Administrator", buf);
         end++;
    } else if (alevel == 4) {
         snprintf(sbuf, sizeof(sbuf), "%s, an IRC Operator", buf);
         end++;
    } else {
         snprintf(sbuf, sizeof(sbuf), "%s", buf);
    }

    if (!end)
       snprintf(sbuf, sizeof(sbuf), "User");

    access1 = sbuf;
    return access1;
}

static void do_flist(const char *source)
{
    char *cmd = strtok(NULL, " ");
    NickInfo *ni;
    int i;
    int totalaccess = 0;

    if (!rec_services_oper(source)) {
	notice(s_NickServ, source,
		"Unrecognized command \2%s\2.  Type \"/msg %s HELP\" for help.",
		cmd, s_NickServ);
        return;
    }

    if (!cmd) {
        notice(s_NickServ,source,"Syntax: \2FLIST\2 [HELD|MARKED|FORBIDDEN]");
        notice(s_NickServ,source,
            "\2/msg %s OHELP FLIST\2 for more information",s_NickServ);

    } else if (stricmp(cmd, "HELD") == 0) {
        notice(s_NickServ,source,"\2HELD\2 nickname listing:");
        for  (i = 0; i < 256; ++i) {
          for (ni = nicklists[i]; ni; ni = ni->next) {
              if (ni->hold) {
                notice(s_NickServ,source,"%-20s  %s",
                   ni->nick,ni->last_usermask);
                notice(s_NickServ,source,
                     "Set By: %s",ni->hold);
		  ++totalaccess;
              }
          }
        }
        notice(s_NickServ,source, NS_NFEOL, totalaccess);
    } else if (stricmp(cmd, "MARKED") == 0) {
        notice(s_NickServ,source,"\2MARKED\2 nickname listing:");
        for (i = 0; i < 256; ++i) {
            for (ni = nicklists[i]; ni; ni = ni->next) {
                if (ni->mark) {
                  notice(s_NickServ,source,"%-20s  %s",
                        ni->nick,ni->last_usermask);
                notice(s_NickServ,source,
                     "Set By: %s",ni->mark);
		  ++totalaccess;
                }
            }
        }
        notice(s_NickServ,source, NS_NFEOL, totalaccess);
    } else if (stricmp(cmd, "FORBIDDEN") == 0) {
        notice(s_NickServ,source,"\2FORBIDDEN\2 nickname listing:");
        for (i = 0; i < 256; ++i) {
            for (ni = nicklists[i]; ni; ni = ni->next) {
                if (ni->forbid) {
                  notice(s_NickServ,source,"%-20s  %s",
                        ni->nick,ni->last_usermask);
                notice(s_NickServ,source,
                     "Set By: %s",ni->forbid);
		  ++totalaccess;
                }
            }
        }
        notice(s_NickServ,source, NS_NFEOL, totalaccess);
    } else if (stricmp(cmd, "HOP") == 0) {
        notice(s_NickServ,source,"\2HOP\2 nickname listing:");
        for (i = 0; i < 256; ++i) {
            for (ni = nicklists[i]; ni; ni = ni->next) {
                if (ni->flags & NI_HOP) {
                  notice(s_NickServ,source,"%-20s",
                        ni->nick);
		  ++totalaccess;
                }
            }
        }
        notice(s_NickServ,source, NS_NFEOL, totalaccess);
    } else if (stricmp(cmd, "SOP") == 0) {
        notice(s_NickServ,source,"\2SOP\2 nickname listing:");
        for (i = 0; i < 256; ++i) {
            for (ni = nicklists[i]; ni; ni = ni->next) {
                if (ni->flags & NI_SOP) {
                  notice(s_NickServ,source,"%-20s",
                        ni->nick);
		  ++totalaccess;
                }
            }
        }
        notice(s_NickServ,source, NS_NFEOL, totalaccess);
    }
}

/*************************************************************************/

static void do_hold(const char *source)
{
        const char *nick = strtok(NULL, " ");
        const char *cmd = strtok(NULL, " ");
	NickInfo *ni;
	NickInfo *ni1;
        User *u = finduser(source);

	if (!is_oper_cando(source, 3)) {
		notice(s_NickServ, source, ERR_MOREFLAG);
		return;
	}

        if (!nick || !cmd) {
            notice(s_NickServ,source,"Syntax: \2HOLD\2 <nick> ON|OFF");
            notice(s_NickServ,source,
                ERR_OHELP,s_NickServ);
            return;
        }

        if (!(ni = findnick(nick))) {
            notice(s_NickServ, source, NS_NOTREG,nick);
	    return;
	}
	if (!stricmp(cmd,"ON")) {
	        if (!(ni->flags & NI_HOLD)) {
        	    ni->flags |= NI_HOLD;
	            if (ni->hold)
        	          free(ni->hold);
	            ni->hold = sstrdup(u->nick);
	            slog("NS +H %s (%s!%s@%s)",
	                nick,u->nick,u->username,u->host);
	            do_break_log("NS_F", "NS +H %s (%s!%s@%s)",
	                nick,u->nick,u->username,u->host);
	            wallops(s_NickServ,
	                "%s set the HELD flag for \2%s\2",u->nick,nick);
	        } else {
        	    notice(s_NickServ,source,NS_ALREADYS,nick, "held");
	        }
	} else if (!stricmp(cmd, "OFF")) {
	     if (ni->flags & NI_HOLD) {
			if (((ni->hold) && (ni1=findnick(ni->hold)) && (ni1->flags & NI_IDENTIFIED) && (stricmp(source, ni->hold) == 0)) || is_services_root(source)) {
		         ni->flags &= ~NI_HOLD;
		         if (ni->hold)
		             free(ni->hold);
		          ni->hold = NULL;
		          slog("NS -H %s (%s!%s@%s)",
			         nick,u->nick,u->username,u->host);
		          do_break_log("NS_F", "NS -H %s (%s!%s@%s)",
		               nick,u->nick,u->username,u->host);
		          wallops(s_NickServ,
		              "%s unset the HELD flag for \2%s\2",u->nick,nick);
			} else {
			          notice(s_NickServ,source,"This nickname is held by other admin.");
			}
	     } else {
        	  notice(s_NickServ,source,"Nickname \2%s\2 is not held.",nick);
	     }
	} else {
            notice(s_NickServ,source,"Syntax: \2HOLD\2 <nick> ON|OFF");
	}
}

/*************************************************************************/

static void do_freeze(const char *source)
{
     const char *nick = strtok(NULL, " ");
     const char *cmd = strtok(NULL, " ");
     User *u = finduser(source);
     NickInfo *ni;
     if (!nick || !cmd) {
        notice(s_NickServ,source,"Syntax: \2FREEZE\2 <nick> ON|OFF");
        notice(s_NickServ,source,
		"\2/msg %s OHELP FREEZE\2 for more information",s_NickServ);
        return;
     }
     if (!(ni = findnick(nick))) {
		notice(s_NickServ, source, NS_NOTREG, nick);
		return;
     }
     if (!stricmp(cmd,"ON")) {
	    if (is_inilevel(nick,3)) {
        	   notice(s_NickServ, source, "Error: Nick is a services member");
	           wallops(s_NickServ, "%s tried to freeze \2%s\2", source, nick);
	           return;
	    }
		if (!(ni->flags & NI_FREEZE)) {
        	    ni->flags |= NI_FREEZE;
	            slog("NS +FR %s (%s!%s@%s)",
	                nick,u->nick,u->username,u->host);
        	    do_break_log("NS_F", "NS +FR %s (%s!%s@%s)",
	                nick,u->nick,u->username,u->host);
	            wallops(s_NickServ,
        	        "%s set the FREEZE flag for \2%s\2",u->nick,nick);
	        } else {
        	    notice(s_NickServ,source, NS_ALREADYS, nick, "freeze");
	        }
     } else if (!stricmp(cmd,"OFF")) {
     		if (ni->flags & NI_FREEZE) {
	          ni->flags &= ~NI_FREEZE;
        	  slog("NS -FR %s (%s!%s@%s)",
	              nick,u->nick,u->username,u->host);
        	  do_break_log("NS_F", "NS -FR %s (%s!%s@%s)",
	               nick,u->nick,u->username,u->host);
        	    wallops(s_NickServ,
	              "%s unset the FREEZE flag for \2%s\2",u->nick,nick);
	     } else {
        	  notice(s_NickServ,source,"Nickname \2%s\2 is not frozen.",nick);
	     }
	} else {
	        notice(s_NickServ,source,"Syntax: \2FREEZE\2 <nick> ON|OFF");
	}
}
/*************************************************************************/

static void do_setpass(const char *source)
{
    char *nick = strtok (NULL, " ");
    char *newpass = strtok (NULL, " ");
    NickInfo *ni;
    User *u = finduser (source);

    if (!nick || !newpass)
        notice (s_NickServ, source, "Syntax: \2SETPASS Nick Password\2");
    else if (!u)
        notice (s_NickServ, source, "Where are you?");

    else if (!(ni = findnick (nick)))
        notice (s_NickServ, source, "nick %s is not registered. ", nick);
    else if (ni && (ni->eflags & NIX_LOCK))
        notice (s_NickServ, source, "This nick is locked from getpass.");
    else if (ni && (ni->flags & NI_MARK) && (!is_services_admin(source)))
        notice (s_NickServ, source, "Can not SETPASS on marked nick. ", nick);
    else
    {
       strscpy (ni->pass, newpass, PASSMAX);
       show_next_db(source, s_NickServ);
       notice(s_NickServ, source, "Password for nick %s changed to \2%s\2.", nick, newpass);
    }
}
/**************************************************************************/
static void do_lock(const char *source)
{
    char *nick = strtok (NULL, " ");
    char *cmd = strtok (NULL, " ");
    NickInfo *ni, *hni;
    
    if (!nick || !cmd)
        notice (s_NickServ, source, "Syntax: \2LOCK Nick ON|OFF\2");
    else if (!(ni = findnick (nick)))
        notice (s_NickServ, source, "nick %s is not registered. ", nick);
    else if (!is_services_coder(source))
        notice (s_NickServ, source, "Only for services Master.");
    else
    {
       hni = nickhost(ni);
       if (!stricmp(cmd,"ON")) {
	       hni->eflags |= NIX_LOCK;
	       notice(s_NickServ, source, "Lock nick %s", ni->nick);
	       show_next_db(source, s_NickServ);
       } else if (!stricmp(cmd,"OFF")) {
	       hni->eflags &= ~NIX_LOCK;
	       notice(s_NickServ, source, "Unlock nick %s", ni->nick);
	       show_next_db(source, s_NickServ);
       } else {
	       notice (s_NickServ, source, "Syntax: \2LOCK Nick ON|OFF\2");
       }
    }
}
/**************************************************************************/
static void do_setmlock(const char *source)
{
    char *nick = strtok (NULL, " ");
    char *modelocks = strtok (NULL, "");
    NickInfo *ni;
    User *u = finduser (source);

    if (!nick || !modelocks)
        notice (s_NickServ, source, "Syntax: \2SETMLOCK umodes\2");
    else if (!u)
        notice (s_NickServ, source, "You not exist in services");
    else if (!(ni = findnick (nick)))
        notice (s_NickServ, source, "nick %s is not registered. ", nick);
    else if ((ni->flags & NI_MARK) && (!is_services_coder(source)))
        notice (s_NickServ, source, "Can not SETMLOCK on marked nick. ");
    else
    {
	   if (ni->mlock) free(ni->mlock);
	   ni->mlock = sstrdup (modelocks);
       show_next_db(source, s_NickServ);
       notice(s_NickServ, source, "Mlock for nick %s changed to \2%s\2.", nick, modelocks);
    }
}
/**************************************************************************/

static void do_set_hidemail(NickInfo *ni, char *param)
{
    NickInfo *hni;
    char *source = ni->nick;
    hni = nickhost(ni);

    if (stricmp(param,"ON") == 0) {

	 if (hni->flags & NI_HIDE_EMAIL) {
            notice(s_NickServ, ni->nick,
                "Your email already HIDEN");
            return;
	 }

         if (!hni->email) {
            notice(s_NickServ, ni->nick,
                "You have no email to HIDE");
            return;
         }
         hni->flags |= NI_HIDE_EMAIL;
         notice(s_NickServ, ni->nick,
                  "Your email address will no longer be shown");
         show_next_db(source, s_NickServ);

    } else if (stricmp(param,"OFF") == 0) {

       if (!hni->email) {
           notice(s_NickServ, ni->nick,
               "You have no email address set to show!");
           return;
       }
	if (!(hni->flags & NI_HIDE_EMAIL)) {
            notice(s_NickServ, ni->nick,
                "Your email already show");
            return;
	}

       hni->flags &= ~NI_HIDE_EMAIL;
       notice(s_NickServ, source, "Your EMAIL is no longer hide");
       show_next_db(source, s_NickServ);

     } else  {
           notice(s_NickServ, source, "Syntax: SET HIDEMAIL ON/OFF");
     }
}
/************************************************************************/
static void do_set_auth(NickInfo *ni, char *param)
{
   NickInfo *hni = nickhost(ni);
   char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "You need to authorized before add to any channel access");
       hni->flags |= NI_AUTH;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "Disable authorized.");
       hni->flags &= ~NI_AUTH;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET AUTH\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET AUTH\2 for more information.",
              s_NickServ);

  }
}
/***************************************************************************/
static void do_set_botmsg(NickInfo *ni, char *param)
{
   NickInfo *hni = nickhost(ni);
   char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "BotMsg Enable");
       hni->flags |= NI_BOTMSG;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "BotMsg Disable.");
       hni->flags &= ~NI_BOTMSG;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET BOTMSG\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET BOTMSG\2 for more information.",
              s_NickServ);

  }
}
/***************************************************************************/
static void do_set_privmsg(NickInfo *ni, char *param)
{
   NickInfo *hni = nickhost(ni);
   User *u = finduser(ni->nick);
   char *source = ni->nick;
   if (!u)
	return;
   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "You may recieve notice in private message");
       hni->flags |= NI_PRIVMSG;
       u->iswebtv=1;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "You may recieve services message in notice.");
       u->iswebtv=0;
       hni->flags &= ~NI_PRIVMSG;
       show_next_db(source, s_NickServ);
       return;

   } else {
       notice(s_NickServ, source, "Syntax: \2SET PRIVMSG\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET PRIVMSG\2 for more information.",
              s_NickServ);

  }
}
/**************************************************************************/
static void do_set_malay(NickInfo *ni, char *param)
{
   NickInfo *hni = nickhost(ni);
   char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "Change help language to Malay");
       hni->flags |= NI_MALAY;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "Change help language to English.");
       hni->flags &= ~NI_MALAY;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET MALAY\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET MALAY\2 for more information.",
              s_NickServ);

  }
}
/******************************************************************/
static void do_setflags(const char *source)
{
    char *nick = strtok (NULL, " ");
    char *nflag = strtok (NULL, " ");
    char *noption = strtok (NULL, " ");
    NickInfo *ni;
    User *u = finduser (source);

    if (!nick || !nflag || !noption)
        notice (s_NickServ, source, "Syntax: \2SETFLAG Nick flag ON/OFF\2");
    else if (!u)
        notice (s_NickServ, source, "Where are you?");

    else if (!(ni = findnick (nick)))
        notice (s_NickServ, source, "nick %s is not registered. ", nick);

    else if ((ni->flags & NI_MARK) && (!is_services_admin(source)))
        notice (s_NickServ, source, "Can not SETFLAG on marked nick. ", nick);

    else if (!(stricmp(noption, "ON")==0) && !(stricmp(noption, "OFF")==0))
        notice (s_NickServ, source, "Option must be ON or OFF");

    else
    {
        if (stricmp(nflag, "AUTH")==0) {
		if(stricmp(noption, "ON")==0)
		       ni->flags |= NI_AUTH;
		else 
		       ni->flags &= ~NI_AUTH;
	       notice(s_NickServ, source, "set flag %s for nick %s \2%s\2.", nflag, nick, noption);

	} else if (stricmp(nflag, "HOP")==0) {
		if (!is_services_admin(source)) {
		        notice(s_NickServ, source, "You need to be Services Admin to set this flag.");
			return;
		}

		if(stricmp(noption, "ON")==0) {
		       ni->flags |= NI_HOP;
		} else 
		       ni->flags &= ~NI_HOP;
	       notice(s_NickServ, source, "set flag %s for nick %s \2%s\2.", nflag, nick, noption);
	
	} else if (stricmp(nflag, "SOP")==0) {

		if (!is_services_admin(source)) {
		        notice(s_NickServ, source, "You need to be Services Admin to set this flag.");
			return;
		}

		if(stricmp(noption, "ON")==0) {
		       ni->flags |= NI_SOP;
		} else 
		       ni->flags &= ~NI_SOP;
	       notice(s_NickServ, source, "set flag %s for nick %s \2%s\2.", nflag, nick, noption);

	} else if (stricmp(nflag, "MALAY")==0) {
		if(stricmp(noption, "ON")==0)
		       ni->flags |= NI_MALAY;
		else 
		       ni->flags &= ~NI_MALAY;
	       notice(s_NickServ, source, "set flag %s for nick %s \2%s\2.", nflag, nick, noption);

	} else if (stricmp(nflag, "PRIVMSG")==0) {
		if(stricmp(noption, "ON")==0)
		       ni->flags |= NI_PRIVMSG;
		else 
		       ni->flags &= ~NI_PRIVMSG;
	       notice(s_NickServ, source, "set flag %s for nick %s \2%s\2.", nflag, nick, noption);

	} else if (stricmp(nflag, "EMAILMEMOS")==0) {

		if (!is_services_admin(source)) {
		        notice(s_NickServ, source, "You need to be Services Admin to set this flag.");
			return;
		}

		if(stricmp(noption, "ON")==0)
		       ni->flags |= NI_EMAILMEMOS;
		else 
		       ni->flags &= ~NI_EMAILMEMOS;
	       notice(s_NickServ, source, "set flag %s for nick %s \2%s\2.", nflag, nick, noption);

	} else if (stricmp(nflag, "NOOP")==0) {
		if(stricmp(noption, "ON")==0)
		       ni->flags |= NI_NOOP;
		else 
		       ni->flags &= ~NI_NOOP;
	       notice(s_NickServ, source, "set flag %s for nick %s \2%s\2.", nflag, nick, noption);

	} else if (stricmp(nflag, "HIDEMAIL")==0) {
		if(stricmp(noption, "ON")==0)
		       ni->flags |= NI_HIDE_EMAIL;
		else 
		       ni->flags &= ~NI_HIDE_EMAIL;
	       notice(s_NickServ, source, "set flag %s for nick %s \2%s\2.", nflag, nick, noption);

	} else if (stricmp(nflag, "KILL")==0) {
		if(stricmp(noption, "ON")==0)
		       ni->flags |= NI_KILLPROTECT;
		else 
		       ni->flags &= ~NI_KILLPROTECT;
	       notice(s_NickServ, source, "set flag %s for nick %s \2%s\2.", nflag, nick, noption);
	}	
    }
}
/***********************************************************************/
static void do_set_uin (NickInfo * ni, char *param)
{
       NickInfo *hni = nickhost(ni);

       if (stricmp (param, "NONE") == 0) {
		  if (hni->uin) free(hni->uin);
          hni->uin = NULL;
          notice (s_NickServ, ni->nick, "Your ICQ UIN was removed.");
       } else if (isNum(param) && (strlen(param) > 5) && (strlen(param) < 11)) {
		  if (hni->uin) free(hni->uin);
		   hni->uin = sstrdup (param);
          notice (s_NickServ, ni->nick, "Change your ICQ UIN to: \2%s\2", param);
       } else {
          notice(s_NickServ, ni->nick, "Syntax: \2SET UIN\2 [\37uin\37|\37NONE\37]");
          notice(s_NickServ, ni->nick, "\2/msg %s HELP SET UIN\2 for more information.", s_NickServ);
       }
}

/**********************************************************************/
static void do_set_age (NickInfo * ni, char *param)
{
	NickInfo *hni = nickhost(ni);
	if (!hni)
		return;

        if (stricmp (param, "NONE") == 0) {
			if (hni->age)
				free(hni->age);
			hni->age = NULL;
			notice (s_NickServ, ni->nick, "Your Age was removed.");
        } else if ((strlen(param) < 3) && isNum(param))  {
			if (hni->age) free(hni->age);
			hni->age = sstrdup (param);
			notice (s_NickServ, ni->nick, "Change your AGE to: \2%s\2",param);
        } else {
          notice(s_NickServ, ni->nick, "Syntax: \2SET AGE\2 [\37age\37|\37NONE\37]");
          notice(s_NickServ, ni->nick, "\2/msg %s HELP SET AGE\2 for more information.", s_NickServ);
        }
}

/**********************************************************************/
static void do_set_info (NickInfo * ni, char *param)
{
    NickInfo *hni = nickhost(ni);

    if (!hni)
	return;
    if (stricmp (param, "NONE") == 0)
    {
	if (hni->info)
		free(hni->info);
        hni->info = NULL;
        notice (s_NickServ, ni->nick, "Set your info to NULL.");
    }
    else
    {
	if (strlen(param) > 225) {
        notice (s_NickServ, ni->nick, "Your info can not set more than 225 characters.");
	} else {
	if (hni->info)
		free(hni->info);
        hni->info = sstrdup (param);
        notice (s_NickServ, ni->nick, "Change your info to :%s", param);
	}
    }
}
/**********************************************************************/
static void do_set_sex(NickInfo * ni, char *param)
{
      NickInfo *hni = nickhost(ni);

      if (!hni)
	return;

      if (stricmp(param, "M") == 0)
      {
	 if (hni->sex)
		free(hni->sex);
         hni->sex = sstrdup("Male");
         notice (s_NickServ, ni->nick, "Changed your GENDER to: \2Male\2");
      } else if (stricmp(param, "F") == 0) {
	 if (hni->sex)
		free(hni->sex);
         hni->sex = sstrdup("Female");
         notice (s_NickServ, ni->nick, "Changed your GENDER to: \2Female\2");
      } else if (stricmp(param, "NONE") == 0) {
	 if (hni->sex)
		free(hni->sex);
         hni->sex = NULL;
         notice (s_NickServ, ni->nick, "Your Gender information was cleared.");
      } else {
          notice (s_NickServ, ni->nick, "Syntax: \2SET SEX\2 [\37M\37|\37F\37|\37NONE\37]");
          notice (s_NickServ, ni->nick,  "\2/msg %s HELP SET SEX\2 for more information.", s_NickServ);
      }
}
/**********************************************************************/
static void do_set_ircop(NickInfo *ni, char *param)
{
   char *source = ni->nick;

   if (stricmp(param, "ON") == 0) {
       notice(s_NickServ, source, "Your IRCop flag is now on, your nick won't expire.");
       ni->flags |= NI_HOP;
       show_next_db(source, s_NickServ);
       return;

   } else if (stricmp(param, "OFF") == 0) {
       notice(s_NickServ, source, "Your IRCop flag is now off.");
       ni->flags &= ~NI_HOP;
       show_next_db(source, s_NickServ);
       return;

   } else {

       notice(s_NickServ, source, "Syntax: \2SET IRCOP\2 [\37ON\37|\37OFF\37]");
       notice(s_NickServ, source,
              "\2/msg %s HELP SET IRCOP\2 for more information.",
              s_NickServ);

  }
}
/**********************************************************************/
static void do_set_mlock(NickInfo * ni, char *param)
{
    char c, nmlock_on[MODEMAX], nmlock_off[MODEMAX], modes[MODEMAX];
    NickInfo *hni = nickhost(ni);

    int add = -1;
    strscpy(nmlock_on, "", sizeof(nmlock_on));
    strscpy(nmlock_off, "", sizeof(nmlock_off));
    strscpy(modes, "", sizeof(modes));

    if (!param) {
        notice (s_NickServ, ni->nick, "Syntax: \2SET MLOCK <umodes> or <NONE>\2");
        notice (s_NickServ, ni->nick, "The user mode you can set are: \2%s\2", ALLOWUMODE);
    } else if (stricmp(param, "NONE")==0) {
	    hni->mlock = NULL;
	    notice (s_NickServ, ni->nick, "Your don't have user modelock set");
    } else if (!(strchr(param, '+')) && !(strchr(param, '-'))) {
        notice (s_NickServ, ni->nick, "Syntax: \2SET MLOCK <umodes> or <NONE>\2");
        notice (s_NickServ, ni->nick, "The user mode you can set are: \2%s\2", ALLOWUMODE);
    } else {
	    while (*param)
	    {
	        if (*param != '+' && *param != '-' && add < 0)
	        {
        	    ++param;
	            continue;
	        }
	        switch ((c = *param++))
	        {
		        case '+':
		            add = 1;
		            break;
		        case '-':
		            add = 0;
		            break;
			default:
		        if (hasmode(myctoa(c), ALLOWUMODE)) {
			if (add) {
			        if(!hasmode(myctoa(c), nmlock_on))
					strcat (nmlock_on, myctoa(c));
			} else {
		        if(!hasmode(myctoa(c), nmlock_off))
				strcat (nmlock_off, myctoa(c));
			}
			}
		}
	    }
	    if (strlen(nmlock_on)) {	
			strcat (modes, "+");
			strcat (modes, nmlock_on);
	    }
	    if (strlen(nmlock_off)) {	
			strcat (modes, "-");
			strcat (modes, nmlock_off);
	    }
	    if (strlen(nmlock_on) || strlen(nmlock_off)) {
			if (hni->mlock) free(hni->mlock);
			hni->mlock = sstrdup(modes);
		    notice (s_NickServ, ni->nick, "Your user modelock set to %s", modes);
	    } else {
			if (hni->mlock) free(hni->mlock);
		    hni->mlock = NULL;
		    notice (s_NickServ, ni->nick, "Your don't have user modelock set");
	    }
    }
}


/****************************************************************/
static void do_set_autojoin(NickInfo *ni, char *param)
{
    NickInfo *hni = nickhost(ni);

    if (stricmp(param, "ON") == 0) {
        hni->eflags |= NIX_AUTOJOIN;
        notice(s_NickServ, ni->nick, toknmsg[20]);
    } else if (stricmp(param, "OFF") == 0) {
        hni->eflags &= ~NIX_AUTOJOIN;
        notice(s_NickServ, ni->nick, toknmsg[21]);
    } else {
        notice(s_NickServ, ni->nick, "Syntax: SET AUTOJOIN {ON | OFF}");
    }
}
/*****************************************************/
static void do_ajoin(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    const char *chan = strtok(NULL, " ");
    NickInfo *ni, *hni;
    User *u;
    int i;
    char **autojoin;

    if (cmd && stricmp(cmd, "LIST") == 0 && chan && is_services_admin(source)) {

	if (!(ni = findnick(chan))) {
	    notice(s_NickServ, source, "Nick \2%s\2 not registered!", chan);
	    return;
	}
	notice(s_NickServ, source, "Channel Autojoin list for \2%s\2:", chan);
	for (autojoin = ni->autojoin, i = 0; i < ni->ajoincount; ++autojoin, ++i)
	    notice(s_NickServ, source, "    %s", *autojoin);

    } else if (!(ni = findnick(source)) || !(hni=nickhost(ni))) {

	notice(s_NickServ, source, "Your nick isn't registered.");

    } else if (!cmd || (((stricmp(cmd,"LIST")==0) || (stricmp(cmd,"WIPE")==0)) ? !!chan : !chan)) {

	notice(s_NickServ, source,
                "Syntax: \2AJOIN\2 [ADD|DEL|LIST|WIPE] #[channel]");
	notice(s_NickServ, source,
		"\2/msg %s HELP AJOIN\2 for more information.", s_NickServ);

    } else if (chan && (stricmp(cmd, "ADD") == 0) && (*chan != '#')) {

	notice(s_NickServ, source,
		"You must add a #channel.");
	notice(s_NickServ, source,
		"\2/msg %s HELP AJOIN\2 for more information.", s_NickServ);

    } else if (!(u = finduser(source)) || !(ni->flags & NI_IDENTIFIED)) {

	notice(s_NickServ, source,
		"Password authentication required for that command.");
	notice(s_NickServ, source,
		"Retry after typing \2/msg %s IDENTIFY \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: SET from nonexistent user %s", s_NickServ, source);

    } else if (stricmp(cmd, "ADD") == 0) {

       if (strlen(chan) > 30) {
           notice(s_NickServ, source, "NickServ channel name too large!");
           wallops(s_NickServ, "%s tried to add a HUGE channel name to the channel autojoin list", source);
           return;
       }

	for (autojoin = hni->autojoin, i = 0; i < hni->ajoincount; autojoin++, i++) {
	    if (strcmp(*autojoin, chan) == 0) {
		notice(s_NickServ, source,
			"Channel \2%s\2 already present on your autojoin list.",
			*autojoin);
		return;
	    }
	}
        if ((hni->ajoincount > 20) && !is_services_admin(source)){
            notice(s_NickServ, source,
                 "Sorry, autojoin list full for nick %s", hni->nick);
            wallops(SERVER_NAME, "%s tried to add another autojoin channel", ni->nick);
            return;
        }
            
	hni->ajoincount++;
	hni->autojoin = srealloc(hni->autojoin, sizeof(char *) * hni->ajoincount);
	hni->autojoin[hni->ajoincount-1] = sstrdup(chan);
	notice(s_NickServ, source, "\2%s\2 added to your autojoin list.", chan);
        show_next_db(source, s_NickServ);

        if (readonly)
            notice(s_NickServ, source,
               "Warning: Services is in read-only mode; changes will not be saved.");

	if (!(hni->eflags & NIX_AUTOJOIN)) {
	    notice(s_NickServ, source, toknmsg[23], s_NickServ);
	}

    } else if (stricmp(cmd, "DEL") == 0) {
        int z, j=0;

        if (isNum(chan) &&  (i = atoi(chan)) > 0 && i <= hni->ajoincount) {

            for (autojoin = hni->autojoin, z = 0; z < hni->ajoincount;
                        ++autojoin, ++z) {

               j += 1;
               if (i == j) {
                   autojoin = &hni->autojoin[z];
                   break;
               }
               
            }
            if (!(i == j)) {
                notice(s_NickServ, source,
                        "No such entry (#%d) on %s autojoin list.", i, chan);
                return;
            }
            i = z;

        } else {

           for (autojoin = hni->autojoin, i = 0; i < hni->ajoincount; ++autojoin, ++i) {
               if (strcmp(*autojoin, chan) == 0)
                    break;
           }
           if (i == hni->ajoincount) {
               for (autojoin = hni->autojoin, i = 0; i < hni->ajoincount;
                                                     ++autojoin, ++i) {
               if (stricmp(*autojoin, chan) == 0)
		    break;
               }
           }
           if (i == hni->ajoincount) {
               notice(s_NickServ, source,
			"\2%s\2 not found on your autojoin list.", chan);
               return;
           }
        }

        notice(s_NickServ, source,
             "\2%s\2 deleted from your autojoin list.", *autojoin);
        show_next_db(source, s_NickServ);
        if (readonly)
            notice(s_NickServ, source,
               "Warning: Services is in read-only mode; changes will not be saved.");

	free(*autojoin);
	--hni->ajoincount;
	if (i < hni->ajoincount)	/* if it wasn't the last entry... */
		memmove(autojoin, autojoin+1, (hni->ajoincount-i) * sizeof(char*));
	if (hni->ajoincount)		/* if there are any entries left... */
	    hni->autojoin = srealloc(hni->autojoin, hni->ajoincount * sizeof(char *));
	else {
	    free(hni->autojoin);
	    hni->autojoin = NULL;
	}

    } else if (stricmp(cmd, "LIST") == 0) {

	notice(s_NickServ, source, "\2Autojoin list:\2");
	for (autojoin = hni->autojoin, i = 0; i < hni->ajoincount; ++autojoin, ++i) {
	    if (chan && !match_wild(chan, *autojoin))
		continue;
            notice(s_NickServ, source, " %d)  %s", i+1, *autojoin);
	}
	notice(s_NickServ, source, ERR_EOL);
	if (!(hni->eflags & NIX_AUTOJOIN)) {
	    notice(s_NickServ, source, toknmsg[23], s_NickServ);
	}
    } else if (stricmp(cmd, "WIPE") == 0) {
        for (autojoin = hni->autojoin, i = 0; i < hni->ajoincount; ++autojoin, ++i) {
	      free(*autojoin);
	}
        hni->ajoincount=0;
        free(hni->autojoin);
        hni->autojoin = NULL;
	notice(s_NickServ,source,"Your Autojoin list has been wiped.");

    } else {

	notice(s_NickServ, source,
                "Syntax: \2AJOIN\2 [ADD|DEL|LIST|WIPE] #[chan]");
	notice(s_NickServ, source,
		"\2/msg %s HELP AJOIN\2 for more information.", s_NickServ);

    }
}
/************************************************************************/
static void do_comment(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    const char *chan = strtok(NULL, "");
    NickInfo *ni;
    User *u;
    int i;
    char **nickcom;

    if (cmd && stricmp(cmd, "LIST") == 0 && chan && is_services_admin(source)) {

	if (!(ni = findnick(chan))) {
	    notice(s_NickServ, source, "Nick \2%s\2 not registered!", chan);
	    return;
	}
	notice(s_NickServ, source, "Comment list for \2%s\2:", chan);
	for (nickcom = ni->comment, i = 0; i < ni->comline; ++nickcom, ++i)
	    notice(s_NickServ, source, "    %s", *nickcom);

    } else if (!cmd || (((stricmp(cmd,"LIST")==0) || (stricmp(cmd,"WIPE")==0)) ? !!chan : !chan)) {

	notice(s_NickServ, source,
                "Syntax: \2COMMENT\2 [ADD|DEL|LIST|WIPE] [Messages]");
	notice(s_NickServ, source,
		"\2/msg %s HELP COMMENT\2 for more information.", s_NickServ);

    } else if (!chan && (stricmp(cmd, "ADD") == 0)) {

	notice(s_NickServ, source,
		"\2/msg %s HELP COMMENT\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(source))) {

	notice(s_NickServ, source, "Your nick isn't registered.");

    } else if (!(u = finduser(source)) || !(ni->flags & NI_IDENTIFIED)) {

	notice(s_NickServ, source,
		"Password authentication required for that command.");
	notice(s_NickServ, source,
		"Retry after typing \2/msg %s IDENTIFY \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: SET from nonexistent user %s", s_NickServ, source);

    } else if (stricmp(cmd, "ADD") == 0) {

       if (strlen(chan) > 200) {
           notice(s_NickServ, source, "You comment to large to handle!");
           wallops(s_NickServ, "%s tried to add a HUGE comment", source);
           return;
       }

       if ((ni->comline > 6) && !is_services_admin(source)) {
            notice(s_NickServ, source,
                 "Sorry, Comment list full for nick %s", ni->nick);
            return;
        }
            
	ni->comline++;
	ni->comment = srealloc(ni->comment, sizeof(char *) * ni->comline);
	ni->comment[ni->comline-1] = sstrdup(chan);
	notice(s_NickServ, source, "\2%s\2 added to your comment list.", chan);
        show_next_db(source, s_NickServ);

        if (readonly)
            notice(s_NickServ, source,
               "Warning: Services is in read-only mode; changes will not be saved.");

    } else if (stricmp(cmd, "DEL") == 0) {
        int z, j=0;
        if (isNum(chan) && (i = atoi(chan)) > 0 && i <= ni->comline) {

            for (nickcom = ni->comment, z = 0; z < ni->comline;
                        ++nickcom, ++z) {

               j += 1;
               if (i == j) {
                   nickcom = &ni->comment[z];
                   break;
               }
               
            }
            if (!(i == j)) {
                notice(s_NickServ, source,
                        "No such entry (#%d) on %s comment list.", i, chan);
                return;
            }
            i = z;

        } else {
            notice(s_NickServ, source,
                        "No more comment to delete.");
	    return;
	}

        notice(s_NickServ, source,
             "\2%s\2 deleted from your comment list.", *nickcom);

        show_next_db(source, s_NickServ);

	free(*nickcom);
	--ni->comline;
	if (i < ni->comline)	/* if it wasn't the last entry... */
		memmove(nickcom, nickcom+1, (ni->comline-i) * sizeof(char *));
	if (ni->comline)		/* if there are any entries left... */
	    ni->comment = srealloc(ni->comment, ni->comline * sizeof(char *));
	else {
	    free(ni->comment);
	    ni->comment = NULL;
	}

        notice(s_NickServ, source,
             "Comment %s is deleted, you have %d comment left.", chan, ni->comline);

    } else if (stricmp(cmd, "LIST") == 0) {
	notice(s_NickServ, source, "\2Comment list:\2");
	for (nickcom = ni->comment, i = 0; i < ni->comline; ++nickcom, ++i) {
	    if (chan && !match_wild(chan, *nickcom))
		continue;
            notice(s_NickServ, source, " %d)  %s", i+1, *nickcom);
	}
	notice(s_NickServ, source, ERR_EOL);

    } else if (stricmp(cmd, "WIPE") == 0) {
        for (nickcom = ni->comment, i = 0; i < ni->comline; ++nickcom, ++i) {
	      free(*nickcom);
	}
        ni->comline=0;
        free(ni->comment);
        ni->comment = NULL;
	notice(s_NickServ,source,"Your comment list has been wiped.");

    } else {

	notice(s_NickServ, source,
                "Syntax: \2COMMENT\2 [ADD|DEL|LIST|WIPE] [Comment]");
	notice(s_NickServ, source,
		"\2/msg %s HELP COMMENT\2 for more information.", s_NickServ);

    }
}
/***************************************************************************/
static void do_noteit(const char *source)
{
    const char *cmd = strtok(NULL, " ");
    const char *chan;
    NickInfo *ni;
    User *u;
    int i;
    char **nicknote;

    if (!cmd) {
	notice(s_NickServ, source,
                "Syntax: \2NOTE\2 [ADD|DEL|LIST|WIPE] [Messages]");
	notice(s_NickServ, source,
		"\2/msg %s HELP NOTE\2 for more information.", s_NickServ);
	return;
    }
    if (!stricmp(cmd,"LIST") || !stricmp(cmd,"DEL")) {
	chan = strtok(NULL, " ");
    } else if (!stricmp(cmd,"ADD")) {
	chan = strtok(NULL, "");
    }

    if (!chan && (stricmp(cmd, "ADD") == 0)) {

	notice(s_NickServ, source,
		"\2/msg %s HELP NOTE\2 for more information.", s_NickServ);

    } else if (!(ni = findnick(source))) {

	notice(s_NickServ, source, "Your nick isn't registered.");

    } else if (!(u = finduser(source)) || !(ni->flags & NI_IDENTIFIED)) {

	notice(s_NickServ, source,
		"Password authentication required for that command.");
	notice(s_NickServ, source,
		"Retry after typing \2/msg %s IDENTIFY \37password\37.",
		s_NickServ);
	if (!u)
	    log("%s: SET from nonexistent user %s", s_NickServ, source);

    } else if (stricmp(cmd, "ADD") == 0) {

       if (strlen(chan) > 200) {
           notice(s_NickServ, source, "You note to large to handle!");
           return;
       }

       if ((ni->noteline > 5) && !is_services_admin(source)) {
            notice(s_NickServ, source,
                 "Sorry, Note list full for nick %s", ni->nick);
            return;
        }
            
	ni->noteline++;
	ni->note = srealloc(ni->note, sizeof(char *) * ni->noteline);
	ni->note[ni->noteline-1] = sstrdup(chan);
	notice(s_NickServ, source, "\2%s\2 added to your note list.", chan);
        show_next_db(source, s_NickServ);

        if (readonly)
            notice(s_NickServ, source,
               "Warning: Services is in read-only mode; changes will not be saved.");

    } else if (stricmp(cmd, "DEL") == 0) {
        int z, j=0;

        if (isNum(chan) && (i = atoi(chan)) > 0 && i <= ni->noteline) {

            for (nicknote = ni->note, z = 0; z < ni->noteline;
                        ++nicknote, ++z) {

               j += 1;
               if (i == j) {
                   nicknote = &ni->note[z];
                   break;
               }
               
            }
            if (!(i == j)) {
                notice(s_NickServ, source,
                        "No such entry (#%d) on %s note list.", i, chan);
                return;
            }
            i = z;

        } else {
	        notice(s_NickServ, source,
        	     "No more note to delete.");
		return;
	}

        notice(s_NickServ, source,
             "\2%s\2 deleted from your note list.", *nicknote);
        show_next_db(source, s_NickServ);


	free(*nicknote);
	--ni->noteline;
	if (i < ni->noteline)	/* if it wasn't the last entry... */
		memmove(nicknote, nicknote+1, (ni->noteline-i) * sizeof(char *));
	if (ni->noteline)		/* if there are any entries left... */
	    ni->note = srealloc(ni->note, ni->noteline * sizeof(char *));
	else {
	    free(ni->note);
	    ni->note = NULL;
	}

    } else if (stricmp(cmd, "LIST") == 0) {
	NickInfo *ni1;
	if (chan) {
		if (is_on_id_list(source, chan)) {
			ni1 = findnick(chan);
		}
	}
	if (!chan) {
		notice(s_NickServ, source, "\2Note list:\2");
		for (nicknote = ni->note, i = 0; i < ni->noteline; ++nicknote, ++i) {
	            notice(s_NickServ, source, " %d)  %s", i+1, *nicknote);
		}
	} else {
		notice(s_NickServ, source, "\2%s Note list:\2", ni1->nick);
		for (nicknote = ni1->note, i = 0; i < ni1->noteline; ++nicknote, ++i) {
	            notice(s_NickServ, source, " %d)  %s", i+1, *nicknote);
		}
	}
	notice(s_NickServ, source, ERR_EOL);

    } else if (stricmp(cmd, "WIPE") == 0) {
        for (nicknote = ni->note, i = 0; i < ni->noteline; ++nicknote, ++i) {
	      free(*nicknote);
	}
        ni->noteline=0;
        free(ni->note);
        ni->note = NULL;
	notice(s_NickServ,source,"Your note list has been wiped.");

    } else {

	notice(s_NickServ, source,
                "Syntax: \2NOTE\2 [ADD|DEL|LIST|WIPE] [Nots]");
	notice(s_NickServ, source,
		"\2/msg %s HELP NOTE\2 for more information.", s_NickServ);

    }
}

/********************************************************/
static void do_search(const char *source)
{
 char *cmd = strtok(NULL, " ");
 char *what = strtok(NULL, " ");
 char *asex = strtok(NULL, " ");
 NickInfo *ni;
 int i, j=0;
 int is_servadmins = rec_services_admin(source);

  if (!is_tokn(38) && !is_servadmins) {
	notice(s_NickServ, source, "This command disable by Admin.");
	return;
  }

  if (!cmd || !what) {
     if (rec_services_root(source)) {
	     notice(s_NickServ, source, "Syntax: \2SEARCH\2 <\2UIN|EMAIL|AGE|SEX|INFO|LMASK|URL\2> <value> [Male|Female]");
     } else {
	     notice(s_NickServ, source, "Syntax: \2SEARCH\2 <\2UIN|EMAIL|AGE|URL\2> <value>");
     }
     notice(s_NickServ, source,
         "\2/msg %s HELP SEARCH\2 for more information.", s_NickServ);
          return;
  } else {
	int v=0;
	if (strlen(what) > 3)
		v = strlen(what) - 3;

	if (!is_servadmins && (strspn(what, "*") > v)) {
		notice(s_NickServ, source, "This keyword not allowed.");
		return;
	}

	if (stricmp(cmd,"UIN")==0) {
		notice(s_NickServ, source, "Searching...");
	        for (i = 33; i < 256; ++i) {
        	    for (ni = nicklists[i]; ni; ni = ni->next) {
                	if (ni->flags & NI_VERBOTEN)
	                    continue;
                	if (ni->flags & NI_SLAVE)
	                    continue;
        	        if (!is_servadmins && (ni->flags & NI_PRIVATE))
                	    continue;
                	if (!ni->uin)
	                    continue;
			if (asex) {
				if (!ni->sex)
					continue;
				if (!(stricmp(ni->sex,asex)==0))
					continue;
			}
			if (match_wild_nocase(what,ni->uin)) {
			     j++;
			     notice(s_NickServ, source, "%d - %s - (%s)", j, ni->nick, ni->uin);
			}
		    } /* End for ni */
		} /* End for 33 */
 	        notice(s_NickServ, source, "\2%d\2 nick(s) founded!!.", j);
	} else if (stricmp(cmd,"AGE")==0) {
		notice(s_NickServ, source, "Searching...");
	        for (i = 33; i < 256; ++i) {
        	    for (ni = nicklists[i]; ni; ni = ni->next) {
                	if (ni->flags & NI_VERBOTEN)
	                    continue;
                	if (ni->flags & NI_SLAVE)
	                    continue;
        	        if (!is_servadmins && (ni->flags & NI_PRIVATE))
                	    continue;
                	if (!ni->age)
	                    continue;
			if (asex) {
				if (!ni->sex)
					continue;
				if (!(stricmp(ni->sex,asex)==0))
					continue;
			}
			if (match_wild_nocase(what,ni->age)) {
			     j++;
			     notice(s_NickServ, source, "%d - %s - (%s)", j, ni->nick, ni->age);
			}
		    } /* End for ni */
		} /* End for 33 */
 	        notice(s_NickServ, source, "\2%d\2 nick(s) founded!!.", j);
	} else if (stricmp(cmd,"EMAIL")==0) {
		notice(s_NickServ, source, "Searching...");
	        for (i = 33; i < 256; ++i) {
        	    for (ni = nicklists[i]; ni; ni = ni->next) {
                	if (ni->flags & NI_VERBOTEN)
	                    continue;
                	if (ni->flags & NI_SLAVE)
	                    continue;
                	if (ni->flags & NI_HIDE_EMAIL)
	                    continue;
        	        if (!is_servadmins && (ni->flags & NI_PRIVATE))
                	    continue;
                	if (!ni->email)
	                    continue;
			if (asex) {
				if (!ni->sex)
					continue;
				if (!(stricmp(ni->sex,asex)==0))
					continue;
			}
			if (match_wild_nocase(what,ni->email)) {
			     j++;
			    notice(s_NickServ, source, "%d - %s - (%s)", j, ni->nick, ni->email);
			}
		    } /* End for ni */
		} /* End for 33 */
 	        notice(s_NickServ, source, "\2%d\2 nick(s) founded!!.", j);
	} else if ((stricmp(cmd,"SEX")==0) && is_servadmins){
		notice(s_NickServ, source, "Searching...");
	        for (i = 33; i < 256; ++i) {
        	    for (ni = nicklists[i]; ni; ni = ni->next) {
                	if (ni->flags & NI_VERBOTEN)
	                    continue;
                	if (ni->flags & NI_SLAVE)
	                    continue;
                	if (ni->flags & NI_HIDE_EMAIL)
	                    continue;
        	        if (!is_servadmins && (ni->flags & NI_PRIVATE))
                	    continue;
                	if (!ni->sex)
	                    continue;
			if (asex) {
				if (!ni->sex)
					continue;
				if (!(stricmp(ni->sex,asex)==0))
					continue;
			}
			if (match_wild_nocase(what, ni->sex)) {
			     j++;
			    notice(s_NickServ, source, "%d - %s - (%s)", j, ni->nick, ni->sex);
			}
		    } /* End for ni */
		} /* End for 33 */
 	        notice(s_NickServ, source, "\2%d\2 nick(s) founded!!.", j);
	} else if ((stricmp(cmd,"INFO")==0) && is_servadmins) {
		notice(s_NickServ, source, "Searching...");
	        for (i = 33; i < 256; ++i) {
        	    for (ni = nicklists[i]; ni; ni = ni->next) {
                	if (ni->flags & NI_VERBOTEN)
	                    continue;
                	if (ni->flags & NI_SLAVE)
	                    continue;
                	if (ni->flags & NI_HIDE_EMAIL)
	                    continue;
        	        if (!is_servadmins && (ni->flags & NI_PRIVATE))
                	    continue;
                	if (!ni->info)
	                    continue;
			if (asex) {
				if (!ni->sex)
					continue;
				if (!(stricmp(ni->sex,asex)==0))
					continue;
			}
			if (match_wild_nocase(what, ni->info)) {
			     j++;
			    notice(s_NickServ, source, "%d - %s - (%s)", j, ni->nick, ni->info);
			}
		    } /* End for ni */
		} /* End for 33 */
 	        notice(s_NickServ, source, "\2%d\2 nick(s) founded!!.", j);
	} else if (stricmp(cmd,"URL")==0) {
		notice(s_NickServ, source, "Searching...");
	        for (i = 33; i < 256; ++i) {
        	    for (ni = nicklists[i]; ni; ni = ni->next) {
                	if (ni->flags & NI_VERBOTEN)
	                    continue;
                	if (ni->flags & NI_SLAVE)
	                    continue;
                	if (ni->flags & NI_HIDE_EMAIL)
	                    continue;
        	        if (!is_servadmins && (ni->flags & NI_PRIVATE))
                	    continue;
                	if (!ni->url)
	                    continue;
			if (asex) {
				if (!ni->sex)
					continue;
				if (!(stricmp(ni->sex,asex)==0))
					continue;
			}
			if (match_wild_nocase(what, ni->url)) {
			     j++;
			    notice(s_NickServ, source, "%d - %s - (%s)", j, ni->nick, ni->url);
			}
		    } /* End for ni */
		} /* End for 33 */
 	        notice(s_NickServ, source, "\2%d\2 nick(s) founded!!.", j);
	} else if ((stricmp(cmd,"LMASK")==0) && is_servadmins) {
		notice(s_NickServ, source, "Searching...");
	        for (i = 33; i < 256; ++i) {
        	    for (ni = nicklists[i]; ni; ni = ni->next) {
                	if (ni->flags & NI_VERBOTEN)
	                    continue;
                	if (ni->flags & NI_SLAVE)
	                    continue;
                	if (ni->flags & NI_HIDE_EMAIL)
	                    continue;
        	        if (!is_servadmins && (ni->flags & NI_PRIVATE))
                	    continue;
                	if (!ni->last_usermask)
	                    continue;
			if (asex) {
				if (!ni->sex)
					continue;
				if (!(stricmp(ni->sex,asex)==0))
					continue;
			}
			if (match_wild_nocase(what, ni->last_usermask)) {
			     j++;
			    notice(s_NickServ, source, "%d - %s - (%s)", j, ni->nick, ni->last_usermask);
			}
		    } /* End for ni */
		} /* End for 33 */
 	        notice(s_NickServ, source, "\2%d\2 nick(s) founded!!.", j);
	} else {
     if (rec_services_root(source)) {
	     notice(s_NickServ, source, "Syntax: \2SEARCH\2 <\2UIN|EMAIL|AGE|SEX|INFO|LMASK|URL\2> <value> [Male|Female]");
     } else {
	     notice(s_NickServ, source, "Syntax: \2SEARCH\2 <\2UIN|EMAIL|AGE|URL\2> <value>");
     }
     notice(s_NickServ, source,
         "\2/msg %s HELP SEARCH\2 for more information.", s_NickServ);
	} /* If UIN */
  }
}
