/* Auspice functions.
 *
 * Auspice Services is copyright (c) 2000-2001 In Mean.
 *     E-mail: <auspice@auspice.org>
 *
 * This program is free but copyrighted software; see the file LICENSE for
 * details.
 */
#include "../inc/services.h"
#include "../inc/timeout.h"

#ifdef DEFHOSTSERV

static int nhostline = 0;
static int hostline_size = 0;

static struct s_hline {
        char *nick;
        char *host;
        char who[NICKMAX];
        long reserved[4];
} *s_hlines = NULL;

void load_hline_db(void)
{
    FILE *f;
    int i;

    if (!(f = open_db(s_HostServ, HLINE_DB, "r", 1)))
        return;

    switch (i = get_file_version(f, HLINE_DB)) {
      case 6:
      case 5:
        nhostline = fgetc(f) * 256 + fgetc(f);
        if (nhostline < 8)
            hostline_size = 16;
        else
            hostline_size = 2*nhostline;
        s_hlines = smalloc(sizeof(*s_hlines) * hostline_size);
        if (!nhostline) {
            fclose(f);
            return;
        }
        if (nhostline != fread(s_hlines, sizeof(*s_hlines), nhostline, f))
            fatal_perror("Read error on %s", HLINE_DB);
        for (i = 0; i < nhostline; i++) {
            s_hlines[i].nick = read_string(f, HLINE_DB);
            s_hlines[i].host = read_string(f, HLINE_DB);
        }
        break;
      default:
        fatal("Unsupported version (%d) on %s", i, HLINE_DB);
    } /* switch (version) */

    close_db(f, HLINE_DB);
}

void save_hline_db(void)
{
    FILE *f;
    int i;

    f = open_db(s_HostServ, HLINE_DB, "w", AS_HL_VERSION);
    fputc(nhostline/256, f);
    fputc(nhostline & 255, f);
    if (fwrite(s_hlines, sizeof(*s_hlines), nhostline, f) != nhostline)
        fatal_perror("Write error on %s", HLINE_DB);
    for (i = 0; i < nhostline; i++) {
        write_string(s_hlines[i].nick, f, HLINE_DB);
        write_string(s_hlines[i].host, f, HLINE_DB);
    }
    close_db(f, HLINE_DB);

}

int is_hostline(const char *nick)
{
    int i;

    for (i = 0; i < nhostline; i++) {
        if (stricmp(s_hlines[i].nick,nick) == 0) {
            return 1;
        }
    }
    return 0;
}

void do_hostline(const char *nick, User *u)
{
    int i;
    for (i = 0; i < nhostline; i++) {
        if (stricmp(s_hlines[i].nick,nick) == 0) {
            send_cmd(s_Agent,"CHGHOST %s %s",nick,s_hlines[i].host);
            if (u->host)
                free(u->host);
            u->host = sstrdup(s_hlines[i].host);
            return;
        }
    }
}

int del_hostline(const char *nick)
{
    int i;
    for (i = 0; i < nhostline && stricmp(s_hlines[i].nick, nick) != 0; i++);
    if (i < nhostline) {
        free(s_hlines[i].nick);
        free(s_hlines[i].host);
        nhostline--;
        if (i < nhostline)
	memmove(s_hlines+i, s_hlines+i+1, sizeof(*s_hlines) * (nhostline-i));
        return 1;
    } else {
        return 0;
    }
}

void HostServ(const char *source, char *buf)
{
    char *cmd;
    char *s, *chan, *nick, *host, *info;
    int i;
    User *u = finduser(source);
    NickInfo *ni;
    cmd = strtok(buf, " ");

    if (!cmd) {
        return;
    } else if (stricmp(cmd, "HELP") == 0) {
	    const char *cmd2 = strtok(NULL, "");
	    char buf[BUFSIZE];
	    snprintf(buf, sizeof(buf), "%s%s", s_HostServ, cmd2 ? " " : "");
	    strscpy(buf+strlen(buf), cmd2 ? cmd2 : "", sizeof(buf)-strlen(buf));
	    helpserv(s_HostServ, source, buf);
    } else if (stricmp(cmd, "\1PING") == 0) {
        if (!(s = strtok(NULL, "")))
            s = "\1";
        notice(s_OperServ, source, "\1PING %s", s);
    } else if (stricmp(cmd, "JOIN") == 0) {

        if (!is_services_root(source)) {
                notice(s_HostServ, source, "Permision Denied.");
                return;
        }
        chan = strtok(NULL, " ");
        if (!chan) {
                notice(s_HostServ, source, "Syntax: JOIN \2Channel\2");
        } else {
                send_cmd (s_HostServ, "JOIN %s", chan);
                send_cmd (s_HostServ, "MODE %s +a %s", chan, s_HostServ);
        }
    } else if (stricmp(cmd, "PART") == 0) {

        if (!is_services_root(source)) {
                notice(s_HostServ, source, "Permision Denied.");
                return;
        }
        chan = strtok(NULL, " ");
        if (!chan) {
                notice(s_HostServ, source, "Syntax: PART \2Channel\2");
        } else {
                send_cmd (s_HostServ, "PART %s", chan);
        }
   } else if (stricmp(cmd, "LIST") == 0) {
        s = strtok(NULL, " ");
        if (!is_services_oper(source)) {
                notice(s_HostServ, source, "Permision Denied.");
                return;
        }
        if (!s)
            s = "*";
        slog("HO L %s (%s!%s@%s)", u->nick, u->nick, u->username, u->host);
        notice(s_HostServ, source, "\2Current Host list:\2");
        for (i = 0; i < nhostline; i++) {
            if (!s || match_wild(s, s_hlines[i].nick)) {
              ni = findnick(s_hlines[i].nick);
              if (ni)
                notice(s_HostServ, source, " %d) %-16s (%s) added by %s",
            i+1, s_hlines[i].nick, s_hlines[i].host, s_hlines[i].who);
              else
                notice(s_HostServ, source, " %d) %-16s (Not Registered)",
                                i+1, s_hlines[i].nick);
            }
        }
        notice(s_HostServ,source,"\2End of List\2");

   } else if (stricmp(cmd,"ADD") == 0) {
	char *vhost, *s;
	int legalhost = 1;
        nick = strtok(NULL, " ");
        host = strtok(NULL, "");
        if (!nick || !host) {
            notice(s_HostServ, source, "Syntax: \2ADD\2 <nick> <host>");
            return;
        }
        if (!findnick(nick)) {
                notice(s_HostServ, source, "Nick %s isn't registered!", nick);
                return;
        }
        if (!is_services_oper(source)) {
                notice(s_HostServ, source, "Permision Denied.");
                return;
        }
        if (strlen(host) > (HOSTLEN - 1)) {
                notice(s_HostServ, source, "Illegal host.");
		return;
	}
	vhost = host;
        for (s = vhost; *s; s++)
        {
       	        if (!isallowed(*s)) 
                {
       	                legalhost = 0;
                }
        }
	if (legalhost) {
        for (i = 0; i < nhostline; i++) {
            if (stricmp(s_hlines[i].nick,nick) == 0) {
                notice(s_HostServ,source,"%s is already present "
                    "on the Host list.", nick);
                return;
            }
	}

        if (nhostline >= hostline_size) {
            if (hostline_size < 8)
                hostline_size = 8;
            else
                hostline_size *= 2;
            s_hlines = srealloc(s_hlines, sizeof(*s_hlines) *hostline_size);
        }

        if (!nhostline || (nhostline < MAX_HOSTSERV) || (MAX_HOSTSERV == 0)) {
            s_hlines[nhostline].nick = sstrdup(nick);
            s_hlines[nhostline].host = sstrdup(host);
            strscpy(s_hlines[nhostline].who, source, NICKMAX);
	    notice(s_HostServ, source, "Added nick %s into auto change host list (%s).",nick,host);
            show_next_db(source, s_HostServ);
            slog("HO A %s %s by %s", nick, host, source);
            ++nhostline;
        } else {
	        notice(s_HostServ,source,"Sorry, Host list is full.");
		wallops(SERVER_NAME,"\2Warning\2: host(nick) list is full. (%d)", MAX_HOSTSERV);
        }
	} else {
                notice(s_HostServ, source, "Illegal host.");
	}
    } else if (stricmp(cmd, "CHANGE") == 0) {
	char *vhost, *s;
	int legalhost = 1;
        nick = strtok(NULL, " ");
        info = strtok(NULL, "");

        if (!is_services_oper(source)) {
            notice(s_HostServ, source, "Permission denied.");
            return;
        }
        if ((nick == NULL) || (info == NULL)) {
          notice(s_HostServ, source, "Syntax: \2CHANGE\2 <nick> <newhost>");
          return;
        }
        if (strlen(info) > (HOSTLEN - 1)) {
                notice(s_HostServ, source, "Illegal host.");
		return;
	}
	vhost = info;
        for (s = vhost; *s; s++)
        {
       	        if (!isallowed(*s)) 
                {
       	                legalhost = 0;
                }
        }
	if (legalhost) {
         for (i = 0; i < nhostline; i++) {
               if (stricmp(s_hlines[i].nick, nick) == 0) {
                   free(s_hlines[i].host);
                   s_hlines[i].host = sstrdup(info);
                   strscpy(s_hlines[i].who, source, NICKMAX);
                   notice(s_HostServ, source, "Nick %s auto change host to %s",
                             s_hlines[i].nick, sstrdup(info));
               }
         }
	} else {
                notice(s_HostServ, source, "Illegal host.");
	}
    } else if (stricmp(cmd, "DEL") == 0) {
        if (!is_services_oper(source)) {
                notice(s_HostServ, source, "Permision Denied.");
                return;
        }
        nick = strtok(NULL, " ");
        if (nick) {
            if (strspn(nick, "1234567890") == strlen(nick) &&
                                (i = atoi(nick)) > 0 && i <= nhostline) {
                strcpy(nick,s_hlines[i-1].nick);
            }
            if (del_hostline(nick)) {
                notice(s_HostServ, source, "%s removed from Host list.", nick);
                slog("HO D %s (%s!%s@%s)",
                nick, u->nick, u->username, u->host);
                if (readonly) {
                    notice(s_HostServ, source,
			 "\2Notice:\2 Changes will not be saved!  Services is in readonly mode.");
                }
            } else {
                notice(s_HostServ, source, "%s not found on Host list.",
                    nick);
            }
        } else {
            notice(s_HostServ, source, "Syntax: \2DEL\2 <nick>");
        }
    } else if (stricmp(cmd, "SETHOST") == 0) {
	char *vhost, *s;
	int legalhost = 1;
        if (!is_services_helpop(source)) {
                notice(s_HostServ, source, "Permision Denied.");
                return;
        }
        chan = strtok(NULL, " ");  /* Chan = Host */
        if (!chan) {
                notice(s_HostServ, source, "Syntax: SETHOST \2<host>\2");
        } else if (strlen(chan) > (HOSTLEN - 1)) {
                notice(s_HostServ, source, "Illegal host.");
        } else {
		vhost = chan;
	        for (s = vhost; *s; s++)
	        {
        	        if (!isallowed(*s)) 
	                {
        	                legalhost = 0;
	                }
	        }
		if (legalhost) {
	                send_cmd (s_HostServ, "CHGHOST %s %s", source, chan);
			notice(s_HostServ, source, "Your host name set to \2%s\2", chan);
			notice(s_HostServ, source, "To turn it off type /mode %s -x", source);
			slog("OPER HOST %s (%s!%s@%s) %s", u->nick, u->nick, u->username, u->host, chan);
			if (u) {
		                if (u->host)
        		        	free(u->host);
	        		u->host = sstrdup(chan);
			}
		} else {
	                notice(s_HostServ, source, "Illegal host.");
		}
        }
   }
}

#endif /* Def HostLine */

