/* Auspice functions.
 *
 * Auspice Services is copyright (c) 2000-2001 In Mean.
 *     E-mail: <auspice@auspice.org>
 *
 * This program is free but copyrighted software; see the file LICENSE for
 * details.
 */
#include "../inc/services.h"
#include "../inc/timeout.h"

static AuthList *authlists[256];
static AuthList *find_authlist(const char *nick);
static void alpha_insert_authlist(AuthList *ml);
static void del_authlist(AuthList *ml);
int is_authexist (const char *uwho, const char *chan);

/********************************************************/
void load_auth_db(void)
{
    FILE *f= fopen(AUTH_DB, "r");
    int ver, i, j;
    AuthList *authl;
    Auths *memos;

    if (!(f = open_db(s_MemoServ, AUTH_DB, "r", 1)))
        return;
    switch (ver = get_file_version(f, AUTH_DB)) {
      case 6:
      case 5:
      case 4:
      case 3:
      case 2:
      case 1:
        for (i = 33; i < 256; ++i) {
            while (fgetc(f) == 1) {
                authl = smalloc(sizeof(AuthList));
                if (1 != fread(authl, sizeof(AuthList), 1, f))
                    fatal_perror("Read error on %s", AUTH_DB);
                alpha_insert_authlist(authl);
		authl->memos = memos = smalloc(sizeof(Auths) * authl->n_memos);
                fread(memos, sizeof(Auths), authl->n_memos, f);
                for (j = 0; j < authl->n_memos; ++j, ++memos) {
                    memos->who = read_string(f, AUTH_DB);
                  if (memos->chan)
                    memos->chan = read_string(f, AUTH_DB);
                }
            }
        }
        break;
      default:
        fatal("Unsupported version number (%d) on %s", ver, AUTH_DB);
    } /* switch (version) */
    close_db(f, AUTH_DB);
}
/*****************************************************************/
void save_auth_db(void)
{
    FILE *f;
    int i, j;
    AuthList *authl;
    Auths *memos;

    if (!(f = open_db(s_MemoServ, AUTH_DB, "w", AUTHS_VERSION)))
        return;
    for (i = 33; i < 256; ++i) {
        for (authl = authlists[i]; authl; authl = authl->next) {
            fputc(1, f);
            if (1 != fwrite(authl, sizeof(AuthList), 1, f) ||
                    authl->n_memos !=
                        fwrite(authl->memos, sizeof(Auths), authl->n_memos, f))
                fatal_perror("Write error on %s", AUTH_DB);
            for (memos = authl->memos, j = 0; j < authl->n_memos; ++memos, ++j) {
                write_string(memos->who, f, AUTH_DB);
                if (memos->chan)
                   write_string(memos->chan, f, AUTH_DB);
            }
        }
        fputc(0, f);
    }
    close_db(f, AUTH_DB);
}
/****************************************************************/
static void alpha_insert_authlist(AuthList *ml)
{
    AuthList *ml2, *ml3;
    char *nick = ml->nick;

    for (ml3 = NULL, ml2 = authlists[tolower(*nick)];
                        ml2 && stricmp(ml2->nick, nick) < 0;
                        ml3 = ml2, ml2 = ml2->next)
        ;
    ml->prev = ml3;
    ml->next = ml2;
    if (!ml3)
        authlists[tolower(*nick)] = ml;
    else
        ml3->next = ml;
    if (ml2)
        ml2->prev = ml;
}
/******************************************************************/
static void del_authlist(AuthList *ml)
{
    if (ml->next)
        ml->next->prev = ml->prev;
    if (ml->prev)
        ml->prev->next = ml->next;
    else
        authlists[tolower(*ml->nick)] = ml->next;
    free(ml->memos);
    free(ml);
}
/*****************************************************************/
void clear_auth(const char *nick)
{
    AuthList *ml;
    ml = find_authlist(nick);
    if (ml)
        del_authlist(ml);
}
/*****************************************************************/
void expire_auth()
{
    AuthList *ml, *next;
    Auths *m;
    int i, i2, ix;
    const time_t expire_time = atoi(tokn[26])*24*60*60;
    time_t now = CTime;
    long count = 0;
    long xcount = 0;
    long xtcount = 0;
    long tcount = 0;
    for (i = 33; i < 256; ++i) {
        for (ml = authlists[i]; ml; ml = next) {
           next = ml->next;
           for (ix = 0, i2 = 0, m = ml->memos; i2 < ml->n_memos; i2++, m++) {
               if (now - m->time >= expire_time) {
                   ++ix;
                   ++xcount;
                   log("Expiring Auth for %s", ml->nick);
                   free(ml->memos[i2].who);
                   free(ml->memos[i2].chan);
                   --ml->n_memos;
                   if (i2 < ml->n_memos)     /* Move down the remain */
                       bcopy(ml->memos + i2+1, ml->memos + i2,
                                       sizeof(Auths) * (ml->n_memos - i2));
                  if (ml->n_memos == 0) {
                      ++xtcount;
                      del_authlist(ml);
                       slog("AUTH X %s [record]", ml->nick);
                      do_break_log("AUTH_X", "AUTH X %s [record]",
                         ml->nick);

                  }
               }
           }
           if (ix > 0) {
               slog("AUTH X %s [%d]",
                 ml->nick, ix);
               do_break_log("AUTH_X", "AUTH X %s [%d]",
                 ml->nick, ix);
           }
        }
    }
    for (i = 0; i < 256; i++) {
        for (ml = authlists[i]; ml; ml = ml->next) {
            tcount++;
            for (i2 = 0; i2 < ml->n_memos; i2++)
                count++;
        }
    }

}
/****************************************************************/
static AuthList *find_authlist(const char *nick)
{
    AuthList *ml;
    int i=0;

    for (ml = authlists[tolower(*nick)];
                        ml && (i = stricmp(ml->nick, nick)) < 0;
                        ml = ml->next)
        ;
    return i==0 ? ml : NULL;
}
/***************************************************************/
void do_auth(const char *source)
{
	NickInfo *ni = findnick(source);
	char *cmd = strtok(NULL, " ");
	if (!ni) {
		notice(s_NickServ, source, "Your nick is not registered.");
		return;
	}
	if (!cmd) {
		notice(s_NickServ, source, "Syntax: AUTH <command>.");
		return;
	}
	if (!stricmp(cmd, "LIST")) {
	    char *nick2 = strtok(NULL, " ");
	    AuthList *ml;
	    Auths *m;
	    int i;
	    char timebuf[64];
	    struct tm tm;
	    if (nick2) {
		if (!is_services_admin(source)) return;
		if (!(ml = find_authlist(nick2))) {
	        	notice(s_NickServ, source, "There is no auth request wait for %s", nick2);
			return;
		}
	    } else {
		if (!(ml = find_authlist(source))) {
	        	notice(s_NickServ, source, "There is no auth request wait for you");
			return;
		}
	    }
            notice(s_NickServ, source, "-=Auth Listing...");
	    for (i = 0, m = ml->memos; i < ml->n_memos; ++i, ++m) {
        	    tm = *localtime(&m->time);
		    *timebuf = 0;
	            strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z", &tm);
        	    timebuf[sizeof(timebuf)-1] = 0;     /* just in case */
	            notice(s_MemoServ, source, "%3d  %-16s %-4s%s%s%s%s%s%s%s %-10s %s",
                        	m->number, m->who, m->chan, " ",
				(m->flags & AUTH_UOP)?"UOP":"",
				(m->flags & AUTH_VOP)?"VOP":"",
				(m->flags & AUTH_HOP)?"HOP":"",
				(m->flags & AUTH_AOP)?"AOP":"",
				(m->flags & AUTH_SOP)?"SOP":"",
				(m->flags & AUTH_SUCCESS)?"SUCCESSOR":"",
				(m->flags & AUTH_CFOUNDER)?"CFOUNDER":"", timebuf);
	    }
            notice(s_NickServ, source, "-=End List");
	} else if (!stricmp(cmd, "READ")) {
		AuthList *ml;
		Auths *m;
		char *numstr = strtok(NULL, " ");
		int num;
		if (!numstr || (num = atoi(numstr)) <= 0) {
			notice(s_NickServ, source, "Syntax: READ <number>");
		} else if (!(ml = find_authlist(source))) {
		        notice(s_MemoServ, source, "You have no requests.");
		} else {
			int i;
			for (i = 0; i < ml->n_memos; i++) {
		                if (ml->memos[i].number == num)
                			break;
	                }
	        	if (i >= ml->n_memos) {
	        	    notice(s_NickServ, source, "Auth %d does not exist!", num);
			} else {
				char timebuf[64];
				struct tm tm;
		        	m = &ml->memos[i];
			        tm = *localtime(&m->time);
				*timebuf = 0;
			        strftime(timebuf, sizeof(timebuf), "%a %b %d %H:%M:%S %Y %Z",&tm);
			        timebuf[sizeof(timebuf)-1] = 0;
				notice(s_NickServ, source, "-=Requested by \2%s\2 since %s", m->who, timebuf);
				notice(s_NickServ, source, "Request add you to %s%s%s%s%s%s%s in channel %s.", 
				(m->flags & AUTH_UOP)?"UOP":"",
				(m->flags & AUTH_VOP)?"VOP":"",
				(m->flags & AUTH_HOP)?"HOP":"",
				(m->flags & AUTH_AOP)?"AOP":"",
				(m->flags & AUTH_SOP)?"SOP":"",
				(m->flags & AUTH_SUCCESS)?"SUCCESSOR":"",
				(m->flags & AUTH_CFOUNDER)?"CFOUNDER":"",m->chan);
				notice(s_NickServ, source, "-=To accept AUTH ACCEPT %d", num);
				notice(s_NickServ, source, "-=To deny AUTH DENY %d", num);
			}

		}
	} else if (!stricmp(cmd, "ACCEPT")) {
		AuthList *ml;
		Auths *m;
		char *numstr = strtok(NULL, " ");
		int num;
		if (!numstr || (num = atoi(numstr)) <= 0) {
			notice(s_NickServ, source, "Syntax: ACCEPT <number>");
		} else if (!(ml = find_authlist(source))) {
		        notice(s_MemoServ, source, "You have no requests.");
		} else {
			int i;
			for (i = 0; i < ml->n_memos; i++) {
		                if (ml->memos[i].number == num)
                			break;
	                }
	        	if (i >= ml->n_memos) {
	        	    notice(s_NickServ, source, "Auth %d does not exist!", num);
			} else {
				char timebuf1[64];
				char *timebuf;
				struct tm tm;
				int templevel = 0;
		        	m = &ml->memos[i];
			        tm = *localtime(&m->time);
				*timebuf1 = 0;
				strftime(timebuf1, sizeof(timebuf1), "%a %b %d %H:%M:%S %Y %Z", &tm);
			        timebuf1[sizeof(timebuf1)-1] = 0;
				timebuf = scalloc(strlen(m->who)+strlen(timebuf1)+3,1);
				sprintf(timebuf, "%s %s", m->who, timebuf1);
				if (m->flags & AUTH_UOP) {
					addbyauth(source, timebuf, m->chan, 2);
					notice(s_NickServ, source, "You are now an UOP in channel %s", m->chan);
					templevel = 2;
				} else if (m->flags & AUTH_VOP) {
					addbyauth(source, timebuf, m->chan, 3);
					notice(s_NickServ, source, "You are now an VOP in channel %s", m->chan);
					templevel = 3;
				} else if (m->flags & AUTH_HOP) {
					addbyauth(source, timebuf, m->chan, 4);
					notice(s_NickServ, source, "You are now an HOP in channel %s", m->chan);
					templevel = 4;
				} else if (m->flags & AUTH_AOP) {
					addbyauth(source, timebuf, m->chan, 5);
					notice(s_NickServ, source, "You are now an AOP in channel %s", m->chan);
					templevel = 5;
				} else if (m->flags & AUTH_SOP) {
					addbyauth(source, timebuf, m->chan, 10);
					notice(s_NickServ, source, "You are now an SOP in channel %s", m->chan);
					templevel = 10;
				} else if (m->flags & AUTH_CFOUNDER) {
					addbyauth(source, timebuf, m->chan, 13);
					notice(s_NickServ, source, "You are now an CFOUNDER in channel %s", m->chan);
					templevel = 13;
				} else if (m->flags & AUTH_SUCCESS) {
					addbyauth(source, timebuf, m->chan, 14);
					notice(s_NickServ, source, "You are now a SUCCESSOR  in channel %s", m->chan);
					templevel = 14;
				}
				free(timebuf);
				do_lautoop(source, ml->memos[i].chan, templevel);
				do_sendaccept(source, ml->memos[i].who, ml->memos[i].chan, templevel);
		                free(ml->memos[i].who);
		                free(ml->memos[i].chan);
		                if (i < ml->n_memos)     /* Move down */
				bcopy(ml->memos+i+1,ml->memos+i,sizeof(Auths)*(ml->n_memos-i));
				--ml->n_memos;           /* One less memo now */
				if (ml->n_memos == 0)
				        del_authlist(ml);
			}

		}
	} else if (!stricmp(cmd, "DENY")) {
		AuthList *ml;
		Auths *m;
		char *numstr = strtok(NULL, " ");
		int num;
		if (!numstr || (num = atoi(numstr)) <= 0) {
			notice(s_NickServ, source, "Syntax: DENY  <number>");
		} else if (!(ml = find_authlist(source))) {
		        notice(s_MemoServ, source, "You have no requests.");
		} else {
			int i;
			for (i = 0; i < ml->n_memos; i++) {
		                if (ml->memos[i].number == num)
                			break;
	                }
	        	if (i >= ml->n_memos) {
	        	    notice(s_NickServ, source, "Auth %d does not exist!", num);
			} else {
		        	m = &ml->memos[i];
		                free(ml->memos[i].who);
		                free(ml->memos[i].chan);
				notice(s_NickServ, source, "Auth %d has been deleted.", num);
		                if (i < ml->n_memos)     /* Move down */
				bcopy(ml->memos+i+1,ml->memos+i,sizeof(Auths)*(ml->n_memos-i));
				--ml->n_memos;           /* One less memo now */
				if (ml->n_memos == 0)
			        del_authlist(ml);
			}

		}
	} else if (!stricmp(cmd, "HELP")) {
       	    notice(s_NickServ, source, "-=Help Index");
       	    notice(s_NickServ, source, "LIST     List all auth request.");
       	    notice(s_NickServ, source, "READ     Read the request.");
       	    notice(s_NickServ, source, "ACCEPT   Accept Request.");
       	    notice(s_NickServ, source, "DENY     Deny Request.");
       	    notice(s_NickServ, source, "-=End");
	}

}
/**************************************************************/
int is_authexist (const char *uwho, const char *chan)
{
	AuthList *ml;
	int i;

	if (!(ml = find_authlist(uwho))) {
		return 0;
	} else {
            for (i = 0; i < ml->n_memos; i++) {
		if (!stricmp(ml->memos[i].chan, chan))
			return ml->memos[i].number;
            }
	}
	return 0;
}
/**************************************************************/
void check_auth(const char *nick)
{
    NickInfo *ni;
    AuthList *ml;

    if (!(ni = findnick(nick)))
        return;
    ml = find_authlist(nick);
    if (ml) {
	notice (s_NickServ, nick, "You have \2%d\2 request wait for you to decide. More info type /msg %s AUTH HELP", 
	ml->n_memos, s_NickServ);
    }
}
/**************************************************************/
void do_addauth (const char *source, const char *uwho, const char *chan, int alevel)
{
	AuthList *ml;
	NickInfo *ni = findnick(uwho);
        Auths *m;
	int authline =0;

	if (!ni)
		return;
	authline = is_authexist(uwho, chan);
        ml = find_authlist(uwho);
	if (!authline) {
        if (!ml) {
                ml = scalloc(sizeof(AuthList), 1);
                strscpy(ml->nick, uwho, NICKMAX);
		alpha_insert_authlist(ml);
        }
        ml->n_memos++;
        ml->memos = srealloc(ml->memos, sizeof(Auths) * ml->n_memos);
        m = &ml->memos[ml->n_memos-1];
	m->who = sstrdup(source);
        if (ml->n_memos > 1) {
		m->number = m[-1].number + 1;
                if (m->number < 1) {
                     int i;
	             for (i = 0; i < ml->n_memos; ++i)
                      ml->memos[i].number = i+1;
                }
        } else
                ml->memos[ml->n_memos-1].number = 1;
        m->time = CTime;
        m->chan = sstrdup(chan);
	m->flags = 0;
	if (finduser(uwho) && (ni->flags & NI_IDENTIFIED))
		notice(s_NickServ, uwho, "You have request pending for auth, Type /Msg %s AUTH HELP for more info", s_NickServ);
	switch (alevel) {
		case 1:
			m->flags |= AUTH_UOP;
			break;
		case 2:
			m->flags |= AUTH_VOP;
			break;
		case 3:
			m->flags |= AUTH_HOP;
			break;
		case 4:
			m->flags |= AUTH_AOP;
			break;
		case 5:
			m->flags |= AUTH_SOP;
			break;
		case 6:
			m->flags |= AUTH_CFOUNDER;
			break;
		case 7:
			m->flags |= AUTH_SUCCESS;
			break;
		default:
			break;
	}
   } else { /* have authline */
	m = &ml->memos[authline-1];
	if (stricmp(m->who, source)) {
		free(m->who);
		m->who = sstrdup(source);
	}
        m->time = CTime;
	m->flags = 0;
	switch (alevel) {
		case 1:
			m->flags |= AUTH_UOP;
			break;
		case 2:
			m->flags |= AUTH_VOP;
			break;
		case 3:
			m->flags |= AUTH_HOP;
			break;
		case 4:
			m->flags |= AUTH_AOP;
			break;
		case 5:
			m->flags |= AUTH_SOP;
			break;
		case 6:
			m->flags |= AUTH_CFOUNDER;
			break;
		case 7:
			m->flags |= AUTH_SUCCESS;
			break;
		default:
			break;
	}
   }
}
