/* Routines to maintain a list of online users.
 * Auspice Services is copyright (c) 2000-2001 In Mean.
 *     E-mail: <auspice@auspice.org>
 *
 * SirvNET Services is copyright (c) 1998-2000 Trevor Klingbeil.
 *     E-mail: <priority1@dal.net>
 * Originally based on EsperNet Services(c) by Andy Church.
 * This program is free but copyrighted software; see the file LICENSE for
 * details.
 */

#include "../inc/services.h"

User *userlist = NULL;
static void check_realname(User *user);
unsigned int usercnt = 0, opcnt = 0, maxusercnt = 0;
unsigned long connectscnt = 0;

/*************************************************************************/

/* Allocate a new User structure, fill in basic values, link it to the
 * overall list, and return it.  Always successful.
 */

static User *new_user(const char *nick)
{
    User *user;
    NickInfo *ni;

    user = scalloc(sizeof(User), 1);
    if (!nick) nick = "";
    strscpy(user->nick, nick, NICKMAX);
    user->floodlev = 0;
    user->next = userlist;

    if (userlist)
	userlist->prev = user;

    userlist = user;
    usercnt++;
    connectscnt++;

    if (usercnt > maxusercnt)
	maxusercnt = usercnt;

    if (is_tokn(34))
	    m_sendmotd(nick);

    if (is_tokn(20)) {
	ni = findnick(nick);
	if (!ni) {
	        notice(s_GlobalNoticer,nick, WELCOME_NOTICE,NETWORK_NAME, nick,NETWORK_NAME, s_NickServ, s_ChanServ);
	}
    }
    return user;
}

/*************************************************************************/

/* Change the nickname of a user, and move pointers as necessary. */

static void change_user_nick(User *user, const char *nick)
{
    strscpy(user->nick, nick, NICKMAX);
}

/*************************************************************************/

/* Remove and free a User structure. */

void delete_user(User *user)
{

    struct u_chanlist *c, *c2;
    struct u_chaninfolist *ci, *ci2;
    int i;

    usercnt--;
    if (user->mode & UMODE_O)
	opcnt--;

    check_del_memos(user);
    cancel_user(user);
    free(user->username);
    free(user->host);
    free(user->realhost);
    free(user->realname);
    free(user->server);

    c = user->chans;
    while (c) {
	c2 = c->next;
	chan_deluser(user, c->chan);
	free(c);
	c = c2;
    }

    ci = user->founder_chans;

    while (ci) {
	ci2 = ci->next;
	free(ci);
	ci = ci2;
    }

    for (i=0; user->id_nicks[i] && i < MAX_IDS; i++)
        free(user->id_nicks[i]);

    if (user->prev)
	user->prev->next = user->next;
    else
	userlist = user->next;

    if (user->next)
	user->next->prev = user->prev;
    free(user);
}

/*************************************************************************/

/* Return statistics.  Pointers are assumed to be valid. */

void get_user_stats(long *nusers, long *memuse)
{
    long count = 0, mem = 0;
    User *user;
    struct u_chanlist *uc;
    struct u_chaninfolist *uci;
    int i;

    for (user = userlist; user; user = user->next) {
	count++;
	mem += sizeof(*user);
	if (user->username)
	    mem += strlen(user->username)+1;
	if (user->host)
	    mem += strlen(user->host)+1;
	if (user->realhost)
	    mem += strlen(user->host)+1;
	if (user->realname)
	    mem += strlen(user->realname)+1;
	if (user->server)
	    mem += strlen(user->server)+1;
	for (uc = user->chans; uc; uc = uc->next)
	    mem += sizeof(*uc);
	for (uci = user->founder_chans; uci; uci = uci->next)
	    mem += sizeof(*uci);
        for (i=0; user->id_nicks[i] && i < MAX_IDS; i++)
            mem += strlen(user->id_nicks[i]);
    }
    *nusers = count;
    *memuse = mem;
}

/*************************************************************************/

/* check realname for some unwanted. */

static void check_realname(User *user)
{
   if (match_wild_nocase("*sub7*", user->realname)) {
	wallops(NULL, "%s's realname match sub7 user (%s)",
		user->nick, user->realname);
   } else if (match_wild_nocase("ircop", user->realname)) {
	wallops(NULL, "%s's realname is (%s)",
		user->nick, user->realname);
   } else if (match_wild_nocase("service", user->realname)) {
	wallops(NULL, "%s's realname is (%s)",
		user->nick, user->realname);
   }
}
/*************************************************************************/

/* Find a user by nick.  Return NULL if user could not be found. */

User *finduser(const char *nick)
{
    User *user = userlist;

    if (strchr(nick, '.'))
	return NULL;

    while (user && stricmp(user->nick, nick) != 0) user = user->next;
    return user;
}
int is_on_uid_list(User *u, const char *nick) {
    unsigned int i;
    if (!u)
	return 0;
    for (i=0; i < MAX_IDS && u->id_nicks[i]; i++) {
	if (stricmp(u->id_nicks[i], nick) == 0) {
	    return 1;
	}
    }
    return 0;
}
int is_on_id_list(const char *source, const char *nick)
{
    User *u = finduser(source);
    return is_on_uid_list(u, nick);
}

/*************************************************************************/

/* Return the first user in the user list. */

User *first_user(void)
{
    return userlist;
}

/*************************************************************************/
/*************************************************************************/

/* Handle a server NICK command.
 *	av[0] = nick
 *	If a new user:
 *		av[1] = hop count
 *		av[2] = signon time
 *		av[3] = username
 *		av[4] = hostname
 *		av[5] = user's server
 *              av[6] = 1 = service 0 = user
 *		av[7] = user's real name
 *         With Bahamut:
 *              av[3] = user modes
 *              av[4] = username
 *              av[5] = hostname
 *              av[6] = user's server
 *              av[7] = services stamp
 *              av[8] = realname
 *         With Unreal 3.0+ with NICKv2
 *              av[6] = 1 = service 0 = user
 *              av[7] = user modes
 *              av[8] = virtual host, not needed
 *              av[9] = user's real name
 *	Else:
 *		av[1] = time of change
 */

void do_nick(const char *source, int ac, char **av)
{
    User *user;
    int i = 0;
    if (!*source) {
	/* This is a new user; create a User structure for it. */

	if (debug)
	    log("debug: new user: %s", av[0]);

	/* We used to ignore the ~ which a lot of ircd's use to indicate no
	 * identd response.  That caused channel bans to break, so now we
	 * just take what the server gives us.  People are _still_
	 * encouraged to read the RFCs and stop doing anything to usernames
	 * depending on the result of an identd lookup.
	 */

	/* First check for AKILLs. */

#ifndef BAHAMUT
        if (check_akill(av[0], av[3], av[4]))
            return;
#else
        if (check_akill(av[0], av[4], av[5]))
            return;
#endif
	/* Allocate User structure and fill it in. */
	user = new_user(av[0]);
	user->signon = atol(av[2]);
#ifdef BAHAMUT
	i = 1;
#endif
        user->username = sstrdup(av[3 + i]);
#ifndef NICKV2
        user->host = sstrdup(av[4 + i]);
#endif
	user->realhost = sstrdup(av[4 + i]);
        user->server = sstrdup(av[5 + i]);
#ifdef NICKV2
	i = 2;
#endif
        user->realname = sstrdup(av[7 + i]);
#ifdef BAHAMUT
        av[1] = av[3];
        do_umode(av[0], 2, av);
#endif
#ifdef NICKV2
        av[1] = av[7];
	user->host = sstrdup(av[8]);
        do_umode(av[0], 2, av);
#endif
	user->games = 0;

	if (!stricmp(user->server, WEBTVSERV)) {
		user->iswebtv = 1;
	} else {
		user->iswebtv = 0;
	}

	/* Check to see if it looks like clones. */
        if (is_tokn(5))
           check_clones(user);

        /* Check to see if realname setting matches against warning list */

	if (is_tokn(18))
		check_realname (user);

	do_logonmsg(user->nick);
	do_autojoin(user->nick, user->host, user->username);

    } else {

	/* An old user changing nicks. */
	user = finduser(source);

	if (!user) {
	    log("user: NICK from nonexistent nick %s: %s", source, merge_args(ac, av));
	    return;
	}

	if (debug)
	    log("debug: %s changed nick to %s", source, av[0]);

	cancel_user(user);
	change_user_nick(user, av[0]);
	user->signon = atol(av[1]);

    }

    user->my_signon = CTime;

    if (validate_user(user)) {
	check_memos(user->nick);
	check_auth(user->nick);
    }

}

/*************************************************************************/
/* Handle a JOIN command.
 *	av[0] = channels to join
 */

void do_join(const char *source, int ac, char **av)
{
    User *user;
    char *s, *t;
    struct u_chanlist *c, *c2;
    Channel *cc;
    user = finduser(source);

    if (!user) {
	log("user: JOIN from nonexistent user %s: %s", source, 
		merge_args(ac, av));
		return;
    }
    t = av[0];

    while (*(s=t)) {
	t = s + strcspn(s, ",");
	if (*t)
		*t++ = 0;
	if (debug) {
		log("debug: %s joins %s", source, s);
	}

	if (*s == '0') {
        	 c = user->chans;
	         while (c)
        	 {
	             c2 = c->next;
        	     chan_deluser(user, c->chan);
	             free (c);
        	     c = c2;
	         }
        	 user->chans = NULL;
	         continue;
	}

        /* Make sure check_kick comes before chan_adduser, so banned users
	 * don't get to see things like channel keys. */

	cc = findchan(s);

	if (check_kick(user, s, cc))
	    continue;

	check_welcome(user, s);
	chan_adduser(user, s);

	c = scalloc(sizeof(*c),1);
	c->next = user->chans;

	if (user->chans)
	    user->chans->prev = c;
	user->chans = c;
	if (!cc) {
		c->chan = findchan(s);
	} else {
		c->chan = cc;
	}

	checkmassop(user, s, c->chan);
    }
}

/*************************************************************************/

/* Handle a PART command.
 *	av[0] = channels to leave
 *	av[1] = reason (optional)
 */

void do_part(const char *source, int ac, char **av)
{
    User *user;
    char *s, *t;
    struct u_chanlist *c;

    user = finduser(source);

    if (!user) {
	log("user: PART from nonexistent user %s: %s", source,
							merge_args(ac, av));
	return;
    }


    t = av[0];

    while (*(s=t)) {

	t = s + strcspn(s, ",");

	if (*t)
	    *t++ = 0;

	if (debug)
	    log("debug: %s leaves %s", source, s);
	for (c = user->chans; c && stricmp(s, c->chan->name) != 0; c = c->next)
	    ;
	if (c) {
            if (!c->chan) {
                log("parting %s channel entry found but c->chan NULL", s);
                return;
            }
	    chan_deluser(user, c->chan);

	    if (c->next)
		c->next->prev = c->prev;
	    if (c->prev)
		c->prev->next = c->next;
	    else
		user->chans = c->next;
	    free(c);

	}
    }
}

/*************************************************************************/

/* Handle a KICK command.
 *	av[0] = channel
 *	av[1] = nick(s) being kicked
 *	av[2] = reason
 */

void do_kick(const char *source, int ac, char **av)
{
    User *user;
    char *s, *t;
    struct u_chanlist *c;

    t = av[1];
    while (*(s=t)) {
	t = s + strcspn(s, ",");
	if (*t)
	    *t++ = 0;
	user = finduser(s);
	if (!user) {
	    log("user: KICK for nonexistent user %s on %s: %s", s, av[0],
						merge_args(ac-2, av+2));
	    continue;
	}
	if (debug)
	    log("debug: kicking %s from %s", s, av[0]);

	if (spyhost && !stricmp(spyhost, user->realhost))
		send_cmd(s_ChanServ, "PRIVMSG #Debug :kicking %s from %s", s, av[0]);

	for (c = user->chans; c && stricmp(av[0], c->chan->name) != 0;
								c = c->next)
	    ;
	if (c) {
	    chan_deluser(user, c->chan);
	    if (c->next)
		c->next->prev = c->prev;
	    if (c->prev)
		c->prev->next = c->next;
	    else
		user->chans = c->next;
	    free(c);
	}
    }
}

/*************************************************************************/

/* Handle a MODE command for a user.
 *	av[0] = nick to change mode for
 *	av[1] = modes
 */

void do_umode(const char *source, int ac, char **av)
{
    User *user;
    char *s;
    int add = 1;		/* 1 if adding modes, 0 if deleting */

    if (stricmp(source, av[0]) != 0) {
	log("user: MODE %s %s from different nick %s!", av[0], av[1], source);
	wallops(NULL, "%s attempted to change mode %s for %s", source, av[1], av[0]);
	return;
    }
    user = finduser(source);
    if (!user) {
	log("user: MODE %s for nonexistent nick %s: %s", av[1], source, merge_args(ac, av));
	return;
    }
    if (debug)
	log("debug: Changing mode for %s to %s", source, av[1]);

    s = av[1];
    while (*s) {
	switch (*s++) {
	    case '+': add = 1; break;
	    case '-': add = 0; break;
	    case 'i': add ? (user->mode |= UMODE_I) : (user->mode &= ~UMODE_I); break;
	    case 'w': add ? (user->mode |= UMODE_W) : (user->mode &= ~UMODE_W); break;
	    case 'g': add ? (user->mode |= UMODE_G) : (user->mode &= ~UMODE_G); break;
	    case 's': add ? (user->mode |= UMODE_S) : (user->mode &= ~UMODE_S); break;
	    case 'A': add ? (user->mode |= UMODE_A) : (user->mode &= ~UMODE_A); break;
	    case 'C': add ? (user->mode |= UMODE_C) : (user->mode &= ~UMODE_C); break;
#ifdef NETADMIN
	    case 'N':
		if (add) { 
			if (is_tokn(45)) {
				if  (is_oper_cando(user->nick, 1)) {
					user->mode |= UMODE_N;
				} else {
					send_cmd(s_NickServ, "%s %s -N 0", istoken?TOK_SVSMODE:MSG_SVSMODE, user->nick);
				}
			} else {
				user->mode |= UMODE_N;
			}
		} else
			user->mode &= ~UMODE_N;
		break;
	    case 'T': add ? (user->mode |= UMODE_T) : (user->mode &= ~UMODE_T); break;
	    case 'I': add ? (user->mode |= UMODE_IN) : (user->mode &= ~UMODE_IN); break;
	    case 'H': add ? (user->mode |= UMODE_HO) : (user->mode &= ~UMODE_HO); break;
#endif
	    case 'B': add ? (user->mode |= UMODE_B) : (user->mode &= ~UMODE_B); break;
#ifdef LIQUID
	    case '3': add ? (user->mode |= UMODE_HO) : (user->mode &= ~UMODE_HO); break;
#endif
	    case 'o':

		if (add) {
                    if (!(user->flags & U_NOOPER)) {
			user->mode |= UMODE_O;
#ifdef SVSJOIN
			if (is_tokn(21))
				send_cmd(s_Agent, "%s %s %s", istoken?TOK_SVSJOIN:MSG_SVSJOIN, user->nick, tokn[17]);
#endif
	      		do_operlogonmsg(user->nick);

                    	opcnt++;

		    } else {
			send_cmd(s_OperServ, "PRIVMSG #Snoop :Failed oper for %s", user->nick);
			send_cmd(NULL, "%s %s -oaA", istoken?TOK_SVSMODE:MSG_SVSMODE, user->nick);
			notice(s_OperServ, user->nick, "Sorry, You have the Abusive user mode set, You cannot oper.");
		    }

		} else {
		    user->mode &= ~UMODE_O;
		    --opcnt;
		}
		break;
	}
    }
}

/*************************************************************************/

/* Handle a QUIT command.
 *	av[0] = reason
 */

void do_quit(const char *source, int ac, char **av)
{
    User *user;
    NickInfo *ni, *hni;

    if (check_servii_kill(source, ac, av))
        return;

    if (!source)
	return;

    user = finduser(source);

    if (!user)
        return;
   
    if (debug)
	log("debug: %s quits", source);

    ni = findnick(source);
    if (ni) {
    	hni = nickhost(ni);
	if ((!(hni->flags & NI_VERBOTEN)) && (ni->flags & (NI_IDENTIFIED | NI_RECOGNIZED)))
	{	
		hni->last_seen = CTime;
	        if (hni->last_quit) free(hni->last_quit);
	        hni->last_quit = *av[0]?sstrdup(av[0]):NULL;
	}
    }
    delete_user(user);
}

/*************************************************************************/

/* Handle a KILL command.
 *	av[0] = nick being killed
 *	av[1] = reason
 */

void do_kill(const char *source, int ac, char **av)
{
    User *user;
    NickInfo *ni;

    user = finduser(av[0]);

    if (check_servii_kill(source, ac, av))
        return;

    if (!user)
        return;

    if (debug && av[0])
	log("debug: %s killed", av[0]);

    if ((ni = findnick(av[0])) && (!(ni->flags & NI_VERBOTEN)) && (ni->flags & (NI_IDENTIFIED | NI_RECOGNIZED)))
	ni->last_seen = CTime;
	
    delete_user(user);
}

/*************************************************************************/
/*************************************************************************/

/* Is the given nick an oper? */

int inline is_oper(const char *nick)
{
    User *user = finduser(nick);
    return user && (user->mode & UMODE_O);
}
int inline is_uoper(User *user)
{
    return user && (user->mode & UMODE_O);
}
/***********************************************************************/
#ifdef NETADMIN
int inline is_netadmin(const char *nick)
{
    User *user = finduser(nick);
    return user && (user->mode & UMODE_N);
}
/*************************************************************************/
int inline is_techadmin(const char *nick)
{
    User *user = finduser(nick);
    return user && (user->mode & UMODE_T);
}
/*************************************************************************/
int inline is_invisible(const char *nick)
{
    User *user = finduser(nick);
    return user && (user->mode & UMODE_IN);
}
/*************************************************************************/
int inline is_hideoper(const char *nick)
{
    User *user = finduser(nick);
    return user && (user->mode & UMODE_HO);
}
#endif
/*************************************************************************/
int inline is_admin(const char *nick)
{
    User *user = finduser(nick);
    return user && (user->mode & UMODE_A);
}
/*************************************************************************/
int inline is_servadmin(const char *nick)
{
    User *user = finduser(nick);
    return user && (user->mode & UMODE_A);
}
int inline is_coadmin(const char *nick)
{
    User *user = finduser(nick);
    return user && (user->mode & UMODE_C);
}
/*************************************************************************/
int shoulda(User *user) {
    if (!user)
	return 0;
    if (user->mode & UMODE_O)
	return 1;
#ifdef NETADMIN
    if (user->mode & UMODE_N)
	return 4;
    if (user->mode & UMODE_T)
	return 3;
#endif
    if (user->mode & UMODE_A)
	return 2;
    return 0;
}

int operlevel(const char *nick) {
    User *user = finduser(nick);
    if (!user)
	return 0;
#ifdef NETADMIN
    if (user->mode & UMODE_N)
	return 1;
    if (user->mode & UMODE_T)
	return 2;
#endif
    if (user->mode & UMODE_A)
	return 3;
    if (user->mode & UMODE_C)
	return 4;
    if (user->mode & UMODE_O)
	return 5;
    return 0;
}
/*************************************************************************/

/* Is the given nick on the given channel? */

int is_on_chan(const char *nick, const char *chan)
{
    User *u = finduser(nick);
    struct u_chanlist *c;

    if (!u)
	return 0;
    for (c = u->chans; c; c = c->next) {
	if (stricmp(c->chan->name, chan) == 0)
	    return 1;
    }
    return 0;
}
int is_on_uchan(User *u, const char *chan)
{
    struct u_chanlist *c;

    if (!u) return 0;
    for (c = u->chans; c; c = c->next) {
	if (stricmp(c->chan->name, chan) == 0)
	    return 1;
    }
    return 0;
}
/*************************************************************************/

/* Is the given nick a channel operator on the given channel? */

int is_chanop(const char *nick, const char *chan)
{
    Channel *c = findchan(chan);

    if (!c)
	return 0;
    return has_chan_status(finduser(nick), c, CSTATUS_OP);
    return 0;

}
/*************************************************************************/
#ifdef DEFHALFOP
int is_chanhop(const char *nick, const char *chan)
{
    Channel *c = findchan(chan);

    if (!c)
	return 0;
    return has_chan_status(finduser(nick), c, CSTATUS_HALFOP);
    return 0;
}
#endif
/*************************************************************************/

/* Is the given nick voiced (channel mode +v) on the given channel? */

int is_voiced(const char *nick, const char *chan)
{
    Channel *c = findchan(chan);

    if (!c)
	return 0;
    return has_chan_status(finduser(nick), c, CSTATUS_VOICE);
    return 0;
}
/*************************************************************************/
#ifdef DEFPROTECT
int is_prots(const char *nick, const char *chan)
{
    Channel *c = findchan(chan);
    if (!c)
	return 0;
    return has_chan_status(finduser(nick), c, CSTATUS_PROTECT);
    return 0;
}
#endif
/***********************************************************************/
#ifdef CFOUNDER
int is_cfounder(const char *nick, const char *chan)
{
    Channel *c = findchan(chan);

    if (!c)
	return 0;
    return has_chan_status(finduser(nick), c, CSTATUS_OWNER);
    return 0;
}
#endif
/*************************************************************************/
/*************************************************************************/

/* Does the user's usermask match the given mask (either nick!user@host or
 * just user@host)?
 */

int match_usermask(const char *mask, User *user)
{
    char *mask2 = sstrdup(mask);
    char *nick, *username, *host1, *nick2, *host2;
    int result;

    if (strchr(mask2, '!')) {
	nick = strlower(strtok(mask2, "!"));
	username = strtok(NULL, "@");
    } else {
	nick = NULL;
	username = strtok(mask2, "@");
    }

    host1 = strtok(NULL, "");

    if (!username || !host1) {
	free(mask2);
	return 0;
    }

    strlower(host1);

    if (nick)
	nick2 = strlower(sstrdup(user->nick));

    host2 = strlower(sstrdup(user->host));

    result = (nick ? match_wild(nick, nick2) : 1) &&
		match_wild(username, user->username) &&
		match_wild(host1, host2);

    free(mask2);
    if (nick) free(nick2);
    free(host2);
    return result;
}
/*************************************************************************/
int match_usermask2(const char *mask, User *u)
{
    char *urmask, *urmask2;
    int result = 0;

    if (!u || !mask) return 0;

    urmask = smalloc(strlen(u->nick) + strlen(u->username) + strlen(u->host) + 3);
    sprintf(urmask, "%s!%s@%s", u->nick, u->username, u->host);
    urmask2 = smalloc(strlen(u->nick) + strlen(u->username) + strlen(u->realhost) + 3);
    sprintf(urmask2, "%s!%s@%s", u->nick, u->username, u->realhost);
    if (urmask && urmask2) {
                if (match_wild_nocase(mask, urmask) ||
                        match_wild_nocase(mask, urmask2)) result = 1;
    }
    return result;
}

/*************************************************************************/

/* Split a usermask up into its constitutent parts.  Returned strings are
 * malloc()'d, and should be free()'d when done with.  Returns "*" for
 * missing parts.
 */

void split_usermask(const char *mask, char **nick, char **user, char **host1)
{
    char *mask2 = sstrdup(mask);

    *nick = strtok(mask2, "!");
    *user = strtok(NULL, "@");
    *host1 = strtok(NULL, "");
    /* Handle special case: mask == user@host */
    if (*nick && !*user && strchr(*nick, '@')) {
	*nick = NULL;
	*user = strtok(mask2, "@");
	*host1 = strtok(NULL, "");
    }
    if (!*nick)
	*nick = "*";
    if (!*user)
	*user = "*";
    if (!*host1)
	*host1 = "*";
    *nick = sstrdup(*nick);
    *user = sstrdup(*user);
    *host1 = sstrdup(*host1);
    free(mask2);
}

/*************************************************************************/

/* Given a user, return a mask that will most likely match any address the
 * user will have from that location.  For IP addresses, wildcards the
 * appropriate subnet mask (e.g. 35.1.1.1 -> 35.*; 128.2.1.1 -> 128.2.*);
 * for named addresses, wildcards the leftmost part of the name unless the
 * name only contains two parts.  If the username begins with a ~, delete
 * it.  The returned character string is malloc'd and should be free'd
 * when done with.
 */

char *create_mask(User *u)
{
    char *mask, *s, *end;

    /* Get us a buffer the size of the username plus hostname.  The result
     * will never be longer than this (and will often be shorter).
     */

    end = mask = smalloc(strlen(u->username) + strlen(u->host) + 2);
    end += sprintf(end, "%s@", u->username);
    if (strspn(u->host, "0123456789.") == strlen(u->host)
		&& (s = strchr(u->host, '.'))
		&& (s = strchr(s+1, '.'))
		&& (s = strchr(s+1, '.'))
		&& (   !strchr(s+1, '.'))) {	/* IP addr */
	s = sstrdup(u->host);
	*strrchr(s, '.') = 0;
	if (atoi(u->host) < 192)
	    *strrchr(s, '.') = 0;
	if (atoi(u->host) < 128)
	    *strrchr(s, '.') = 0;
	sprintf(end, "%s.*", s);
	free(s);
    } else {
	if ((s = strchr(u->host, '.')) && strchr(s+1, '.')) {
	    s = sstrdup(strchr(u->host, '.')-1);
	    *s = '*';
	} else {
	    s = sstrdup(u->host);
	}
	strcpy(end, s);
	free(s);
    }
    return mask;
}
/*********************************************************************/
int bcreate_mask(User *u,int btype, char *ret, int retlen)
{
    char *apoint;
    if (!u || !btype) return 0;
    switch (btype) {
	case 1: 
		apoint = create_mask(u);
		snprintf(ret, retlen, "%s", apoint);
		free(apoint);
		return 1;
	case 2: 
		snprintf(ret, retlen, "*!*@%s", u->host);
		return 1;
	default:
		return 0;
    }
}
/*********************************************************************/
void mass_check_akill()
{
    User *user;
    for (user = userlist; user; user = user->next) {
       if (check_akill(user->nick,user->username,user->realhost))
       	  return;
    }
}

void trace_host(const char *source)
{
    User *user;
    char *ttype = strtok(NULL, " ");
    char *host1 = strtok(NULL, " ");
    int i = 0;

    if (!ttype || !host1) {
       notice(s_OperServ, source, "Syntax: \2TRACE\2 HOST|SERVER mask");
       notice(s_OperServ, source, "Example: \2TRACE\2 HOST *pleaselook.com");
    } else if (!stricmp(ttype, "SERVER")) {
	    for (user = userlist; user; user = user->next) {
	       if(match_wild_nocase(host1, user->server)) {
			++i;
			notice(s_OperServ,source,"%d - %s virthost: (%s@%s) realhost: (%s@%s)", i, user->nick, user->username,user->host, user->username, user->realhost);
		}	
	    }
	    notice(s_OperServ, source, NS_NFEOL, i);
    } else {
	    for (user = userlist; user; user = user->next) {
	       if(match_wild_nocase(host1, user->realhost) || match_wild_nocase(host1, user->host)) {
			++i;
                	notice(s_OperServ,source,"%d - %s virthost: (%s@%s) realhost: (%s@%s)", i, user->nick, user->username,user->host, user->username, user->realhost);
		}	
	    }
	    notice(s_OperServ, source, NS_NFEOL, i);
    }
}

int check_akill_num(char *mask)
{
    User *user;
    int i=0;
    char *nick, *ident, *host1;

  nick = strtok(mask, "!");
  ident = strtok(mask, "@");
  host1 = strtok(mask, "");

  for (user = userlist; user; user = user->next) {
        if (match_wild_nocase(ident, user->username)
          && match_wild_nocase(host1, user->realhost))
             i += 1;
  }
  free(mask);
  return i;
}


/*************************************************************************/

int check_ignore_num(const char *mask)
{
    User *user;
    int i=0;
    char *mask2 = sstrdup(mask);
    char *nick = strtok(mask2, "!");
    char *ident2 = strtok(NULL, "!");
    char *ident = strtok(ident2, "@");
    char *host1 = strtok(NULL, "");

    for (user = userlist; user; user = user->next) {
        if (match_wild_nocase(nick, user->nick)
          && match_wild_nocase(ident, user->username)
          && match_wild_nocase(host1, user->realhost))
             i += 1;
    }
    free(mask2);
  return i;
}

/**********************************************************************/
#ifndef NONSUPER
void alljoin(char *chan)
{
    User *user;
        for (user = userlist; user; user = user->next) {
        	send_cmd(s_MassServ, "%s %s %s", istoken?TOK_SAJOIN:MSG_SAJOIN, user->nick,chan);
		notice(s_MassServ, user->nick, "You are invited to join channel %s by Network Admin",chan);
	}
}
void alloperjoin(char *chan)
{
	User *user;
        for (user = userlist; user; user = user->next) {
		if (user->mode & UMODE_O) {
	        	send_cmd(s_MassServ, "%s %s %s", istoken?TOK_SAJOIN:MSG_SAJOIN, user->nick,chan);
			notice(s_MassServ, user->nick, "You are invited to join channel %s by Network Admin",chan);
		}
	}
}
#endif
