/* Auspice functions.
 *
 * Auspice Services is copyright (c) 2000-2001 In Mean.
 *     E-mail: <auspice@auspice.org>
 *
 * This program is free but copyrighted software; see the file LICENSE for
 * details.
 */

#include "../inc/services.h"
#include "../inc/timeout.h"

static AdminInfo *makeadmin(const char *nick);
AdminInfo *adminlists[256];
static void alpha_insert_admin(AdminInfo *bi);
static void do_addoper(const char *source);
static void do_adminwhois(const char *source);
static void do_setserver(const char *source);
static void showadmin(const char *source, int dowhat);

int is_oper_override(const char *source,AdminInfo *ais);

/**********************************************************************/
void load_admin_db(void)
{
    FILE *f = fopen(ADMINSERV_DB, "r");
    int i, ver, j;
    AdminInfo *ais;

    if (!(f = open_db(s_AdminServ, ADMINSERV_DB, "r", 1)))
        return;

    switch (ver = get_file_version(f, ADMINSERV_DB)) {
	case 1:
        for (i = 33; i < 256; ++i) {
            while (fgetc(f) == 1) {
                   ais = smalloc(sizeof(AdminInfo));
                   if (1 != fread(ais, sizeof(AdminInfo), 1, f))
                       fatal_perror("Read error on %s", ADMINSERV_DB);

                if (ais->nick)
                    ais->nick = read_string(f, ADMINSERV_DB);
                alpha_insert_admin(ais);

                if (ais->host)
                    ais->host = read_string(f, ADMINSERV_DB);
                if (ais->who)
                    ais->who = read_string(f, ADMINSERV_DB);
                if (ais->server)
                    ais->server = read_string(f, ADMINSERV_DB);
                if (ais->markline) {
                    char **adminmark;
                    adminmark = smalloc(sizeof(char *) * ais->markline);
                    ais->mark = adminmark;
                    for (j = 0; j < ais->markline; ++j, ++adminmark)
                        *adminmark = read_string(f, ADMINSERV_DB);
                }

            } /* while (fgetc(f) == 1) */
        } /* for (i) */
	break;
      default:
        fatal("Unsupported version number (%d) on %s", i, ADMINSERV_DB);
    } /* switch (version) */

    close_db(f, ADMINSERV_DB);
}
/**********************************************************************/
static void alpha_insert_admin(AdminInfo *bi)
{
    AdminInfo *bi2, *bi3;
    char *nick = bi->nick;

    for (bi3 = NULL, bi2 = adminlists[tolower(*nick)];
                        bi2 && stricmp(bi2->nick, nick) < 0;
                        bi3 = bi2, bi2 = bi2->next)
        ;
    bi->prev = bi3;
    bi->next = bi2;
    if (!bi3)
        adminlists[tolower(*nick)] = bi;
    else
        bi3->next = bi;
    if (bi2)
        bi2->prev = bi;
}
/**********************************************************************/
void save_admin_db(void)
{
    FILE *f;
    int i, j;
    AdminInfo *ais;
    char **adminmark;

    if (!(f = open_db(s_AdminServ, ADMINSERV_DB, "w", ADS_VERSION)))
        return;
    for (i = 33; i < 256; ++i) {
        for (ais = adminlists[i]; ais; ais = ais->next) {
            fputc(1, f);

            if (1 != fwrite(ais, sizeof(AdminInfo), 1, f))
            fatal_perror("Write error on %s", ADMINSERV_DB);
            if (ais->nick) {
                write_string(ais->nick, f, ADMINSERV_DB);
            }
            if (ais->host) {
                write_string(ais->host, f, ADMINSERV_DB);
            }
            if (ais->who) {
                write_string(ais->who, f, ADMINSERV_DB);
            }
            if (ais->server) {
                write_string(ais->server, f, ADMINSERV_DB);
            }
	    for (adminmark = ais->mark, j = 0; j < ais->markline; ++adminmark, ++j)
                write_string(*adminmark, f, ADMINSERV_DB);

        }
        fputc(0, f);
    }
    close_db(f, ADMINSERV_DB);
}
/**********************************************************************/
static AdminInfo *makeadmin(const char *nick)
{
    AdminInfo *ais;
    ais = scalloc(sizeof(AdminInfo), 1);
    ais->nick = sstrdup(nick);
    alpha_insert_admin(ais);
    return ais;
}
/**********************************************************************/
static int deladmin(AdminInfo *ais)
{
    int i;
    if (ais->next)
        ais->next->prev = ais->prev;
    if (ais->prev)
        ais->prev->next = ais->next;
    else
        adminlists[tolower(*ais->nick)] = ais->next;
    if (ais->nick)
      free(ais->nick);
    if (ais->host)
      free(ais->host);
    if (ais->who)
      free(ais->who);
    if (ais->server)
      free(ais->server);
    if (ais->markline) {
        for (i = 0; i < ais->markline; i++) {
            if (ais->mark[i])
            free(ais->mark[i]);
        }
        free(ais->mark);
    }

    free(ais);
    return 1;
}
/**********************************************************************/
AdminInfo *findadmin(const char *nick)
{
    AdminInfo *ais;
    if (!nick || !*nick) return NULL;
    for (ais = adminlists[tolower(*nick)]; ais; ais = ais->next) {
        if (stricmp(ais->nick, nick) == 0)
            return ais;
    }
    return NULL;
}
/**********************************************************************/
void adminserv(const char *source, char *buf)
{
    AdminInfo *ais;
    char *cmd;
    User *u = finduser(source);
    cmd = strtok(buf, " ");

    slog("ADS (%s!%s@%s) [%s]",source,u->username,u->host,buf);
    do_break_log("ADS", "OS (%s!%s@%s) [%s]",source,u->username,u->host,buf);

    if (!cmd) {
	return;
    } else if (!stricmp(cmd,"\1PING")) {
	char *s;
        if (!(s = strtok(NULL, "")))
            s = "\1";
        notice(s_NickServ, source, "\1PING %s", s);
    } else if (!stricmp (cmd, "\1VERSION\1")) {
	char *s;
        if (!(s = strtok (NULL, "")))
            s = "\1";
        notice(s_AdminServ, source, "%sVERSION Auspice IRC Services %s", s, s);
    } else if (!is_services_admin(source)) {
        notice(s_AdminServ, source, "Permision Denied on Admin Services.");
    } else if (!stricmp(cmd, "HELP")) {
	char *subhelp = strtok(NULL, " ");
	if (!subhelp) {
		notice(s_AdminServ, source, "\2Help Index\2");
		notice(s_AdminServ, source, "OPER          Modified SRA/SA/SOP/HOP list");
		notice(s_AdminServ, source, "WHOIS         Whois an Oper");
		notice(s_AdminServ, source, "OPERSET       Change SRA/SA/SOP/HOP");
		notice(s_AdminServ, source, "SETODESC      Set SRA/SA/SOP/HOP info");
		notice(s_AdminServ, source, "FLAGS         Modify Services Oper flags");
		notice(s_AdminServ, source, "\2End Help\2");
	} else if (!stricmp(subhelp,"WHOIS")) {
		notice(s_AdminServ, source, "Syntax: WHOIS <nick>");
	} else if (!stricmp(subhelp,"OPER")) {
		notice(s_AdminServ, source, "Syntax: OPER ADD <nick> <SRA|SA|SOP|HOP>");
		notice(s_AdminServ, source, "Syntax: OPER DEL <nick>");
		notice(s_AdminServ, source, "Syntax: OPER LIST <nick> <SRA|SA|SOP|HOP>");
	} else if (!stricmp(subhelp,"OPERSET")) {
		notice(s_AdminServ, source, "Syntax: OPERSET <nick> <SRA|SA|SOP|HOP>");
	} else if (!stricmp(subhelp,"SETOFLAG")) {
		notice(s_AdminServ, source, "Syntax: SETOFLAG <nick> <oflag>");
		notice(s_AdminServ, source, "see in /msg %s HELP OFLAG", s_AdminServ);
	} else if (!stricmp(subhelp,"UNSETOFLAG")) {
		notice(s_AdminServ, source, "Syntax: UNSETOFLAG <nick> <oflag>");
		notice(s_AdminServ, source, "see in /msg %s HELP OFLAG", s_AdminServ);
	} else if (!stricmp(subhelp,"OFLAG")) {
		notice(s_AdminServ, source, "\2OFLAG list\2");
		if (is_services_coder(source)) {
			notice(s_AdminServ, source, "(SRA)  \2ADDSRA\2 Allow this SRA to modify SRA list");
			notice(s_AdminServ, source, "(SRA)  \2ADDSA \2 Allow this SRA to modify SA list");
			notice(s_AdminServ, source, "(ALL)  \2SINGLE\2 Allow this OPER have access in one server only");
			notice(s_AdminServ, source, "(ALL)  \2SUPEROP\2 Allow this OPER have AOP access in any channel");
			notice(s_AdminServ, source, "(ALL)  \2ABUSETM\2 Add this oper to Abuse Team");
		}
		if (is_oper_cando(source,7)) {
			notice(s_AdminServ, source, "(ALL)  \2SUSPEND\2 Suspend this Admin.");
			notice(s_AdminServ, source, "(ALL)  \2FJOIN\2   Enable use FJOIN and FPART.");
		}
		if (is_services_root(source)) {
			notice(s_AdminServ, source, "(SA)   \2HOLD  \2 Allow this SA to hold nick/chan");
			notice(s_AdminServ, source, "(SA)   \2FORBID\2 Allow this SA to forbid nick/chan");
			notice(s_AdminServ, source, "(SA)   \2GLOBAL\2 Allow this SA to send global notice");
		}
		if (is_services_admin(source)) {
			notice(s_AdminServ, source, "(SA)   \2AKILL\2 Allow this oper place akill");
		}
		notice(s_AdminServ, source, "\2End of OFLAG list.\2");
	}
    } else if (!stricmp(cmd, "OPER")) {
	do_addoper(source);
    } else if (!stricmp(cmd, "SETSERVER")) {
	do_setserver(source);
    } else if (!stricmp(cmd, "SETODESC")) {
	char *nick = strtok(NULL, " ");
	char *operdesc = strtok(NULL, "");
	if (!nick || !operdesc) {
		notice(s_AdminServ, source, "Syntax: SETODESC <nick> <desc|NULL>");
	} else if (!(ais = findadmin(nick))) {
		notice(s_AdminServ, source, "There is no such this oper.");
	} else if (strlen(operdesc) > 150) {
		notice(s_AdminServ, source, "Desc limite to 150 chars.");
	} else if (!stricmp(operdesc,"NULL")) {
		if (ais->host)
			free(ais->host);
		ais->host = NULL;
	} else {
		if (ais->host)
			free(ais->host);
		ais->host = sstrdup(operdesc);
		notice(s_AdminServ, source, "Set successful.");
	}
    } else if (!stricmp(cmd, "WHOIS") || !stricmp(cmd, "INFO")) {
	do_adminwhois(source);
    } else if (!stricmp(cmd, "OPERSET")) {
	char *nick = strtok(NULL, " ");
	char *operflag = strtok(NULL, " ");
	if (!nick || !operflag) {
		notice(s_AdminServ, source, "Syntax: OPERSET <nick> <flag>");
	} else if (!(ais = findadmin(nick))) {
		notice(s_AdminServ, source, "Oper not exist in list.");
	} else if (!stricmp(operflag, "SRA")) {
		if (is_oper_override(source, ais) && is_oper_cando(source, 1)) {
			ais->adflags = 0;
			ais->adflags |= ADF_SERVICEROOT;
			ais->flags |= AI_ADDSA;
			free(ais->who);
			ais->who = sstrdup(source);
			ais->added = CTime;
			wallops(NULL,"%s is now a Services Root set by %s", ais->nick, source);
		} else {
			notice(s_AdminServ, source, "Permision Denied.");
		}
	} else if (!stricmp(operflag, "SA")) {
		if (is_oper_override(source, ais) && is_services_root(source)) {
			free(ais->who);
			ais->who = sstrdup(source);
			ais->added = CTime;
			ais->adflags = 0;
			ais->adflags |= ADF_SERVICEADMIN;
			wallops(NULL,"%s is now a Services Admin set by %s", ais->nick, source);
		} else {
			notice(s_AdminServ, source, "Permision Denied.");
		}
	} else if (!stricmp(operflag, "SOP")) {
		if (is_oper_override(source, ais)) {
			free(ais->who);
			ais->who = sstrdup(source);
			ais->added = CTime;
			ais->adflags = 0;
			ais->adflags |= ADF_SERVICEOP;
			wallops(NULL,"%s is now a Services Oper set by %s", ais->nick, source);
		} else {
			notice(s_AdminServ, source, "Permision Denied.");
		}
	} else if (!stricmp(operflag, "HOP")) {
		if (is_oper_override(source, ais)) {
			free(ais->who);
			ais->who = sstrdup(source);
			ais->added = CTime;
			ais->adflags = 0;
			ais->adflags |= ADF_HELPOP;
			wallops(NULL,"%s is now a Services Helper set b y  %s", ais->nick, source);
		} else {
			notice(s_AdminServ, source, "Permision Denied.");
		}
	} else {
		notice(s_AdminServ, source, "incorrect syntax, or permision denied.");
	}
    } else if (!stricmp(cmd, "FLAGS")) {
	char *cmd = strtok(NULL, " ");
	char *nick = strtok(NULL, " ");
	char *operflag = strtok(NULL, " ");
	if (!cmd || !nick || !operflag) {
		notice(s_AdminServ, source, "Syntax: FLAGS ADD|DEL <nick> <oflag>");
	} else if (stricmp(cmd,"ADD") && stricmp(cmd,"DEL")) {
		notice(s_AdminServ, source, "Syntax: FLAGS ADD|DEL <nick> <oflag>");
	} else if (!(ais = findadmin(nick))) {
		notice(s_AdminServ, source, "There is no such this oper.");
	} else if (!stricmp(operflag, "ADDSRA")) {
		if (!is_services_coder(source)) {
			notice(s_AdminServ, source, PMD);
			return;
		}
		if (!stricmp(cmd,"ADD")) {
			if (!(ais->adflags & ADF_SERVICEROOT)) {
				notice(s_AdminServ, source, "This flag can set for SRA only.");
			} else {
				free(ais->who);
				ais->who = sstrdup(source);
				ais->added = CTime;
				ais->flags |= AI_ADDSRA;
				notice(s_AdminServ, source, "SRA %s have permision to modify SRA list.", ais->nick);
			}
		} else {
			if (!(ais->flags & AI_ADDSRA)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				free(ais->who);
				ais->who = sstrdup(source);
				ais->added = CTime;
				ais->flags &= ~AI_ADDSRA;
				notice(s_AdminServ, source, "SRA %s have not permision to modify SRA list.", ais->nick);
			}
		}
	} else if (!stricmp(operflag, "ABUSETM")) {
		if (!is_oper_cando(source,1)) {
			notice(s_AdminServ, source, PMD);
			return;
		}
		if (!stricmp(cmd,"ADD")) {
			free(ais->who);
			ais->who = sstrdup(source);
			ais->added = CTime;
			ais->flags |= AI_ABUSETM;
			notice(s_AdminServ, source, "%s is now an Abuse Team.", ais->nick);
		} else {
			if (!(ais->flags & AI_ABUSETM)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_ABUSETM;
				notice(s_AdminServ, source, "%s removed from abuse team.", ais->nick);
			}
		}
	} else if (!stricmp(operflag, "FJOIN")) {
		if (!is_oper_cando(source,1)) {
			notice(s_AdminServ, source, PMD);
			return;
		}
		if (!stricmp(cmd,"ADD")) {
			ais->flags |= AI_FORCEJOIN;
			notice(s_AdminServ, source, "%s is now able to do FJOIN.", ais->nick);
		} else {
			if (!(ais->flags & AI_FORCEJOIN)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_FORCEJOIN;
				notice(s_AdminServ, source, "%s disable FJOIN.", ais->nick);
			}
		}
	} else if (!stricmp(operflag, "SUSPEND")) {
		if (!is_oper_cando(source, 7)) {
			notice(s_AdminServ, source, ERR_MOREFLAG);
			return;
		}
		if (!stricmp(cmd,"ADD")) {
			if (!(ais->adflags & ADF_SERVICEROOT)) {
				notice(s_AdminServ, source, "Permision denied.");
			} else {
				ais->flags |= AI_QUIT;
				notice(s_AdminServ, source, "%s is now an Suspend.", ais->nick);
			}
		} else {
			if (!(ais->flags & AI_QUIT)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_QUIT;
				notice(s_AdminServ, source, "Unsupended.");
			}
		}
	} else if (!stricmp(operflag, "SINGLE")) {
		if (!is_services_coder(source)) {
			notice(s_AdminServ, source, PMD);
			return;
		}
		if (!stricmp(cmd,"ADD")) {
			ais->flags |= AI_SINGLE;
			notice(s_AdminServ, source, "This oper is only work in one server.");
		} else {
			if (!(ais->flags & AI_SINGLE)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_SINGLE;
				notice(s_AdminServ, source, "This OPER have global access.");
			}
		}
	} else if (!stricmp(operflag, "SUPEROP")) {
		if (!is_services_coder(source)) {
			notice(s_AdminServ, source, PMD);
		} 
		if (!stricmp(cmd,"ADD")) {
			ais->flags |= AI_SUPEROP;
			notice(s_AdminServ, source, "This oper have superop permision.");
		} else {
			if (!(ais->flags & AI_SUPEROP)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_SUPEROP;
				notice(s_AdminServ, source, "Removed SUPEROP Flag.");
			}
		}
	} else if (!stricmp(operflag, "AKILL")) {
		if (!is_services_root(source)) {
			notice(s_AdminServ, source, PMD);
		}
		if (!stricmp(cmd,"ADD")) {
			ais->flags |= AI_AKILL;
			notice(s_AdminServ, source, "This oper can place akill.");
		} else {
			if (!(ais->flags & AI_AKILL)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_AKILL;
				notice(s_AdminServ, source, "Removed AKILL Flag.");
			}
		}
	} else if (!stricmp(operflag, "ADDSA")) {
		if (!(ais->adflags & ADF_SERVICEROOT)) {
			notice(s_AdminServ, source, "This flag can set for SRA only.");
			return;
		} else if (!is_services_coder(source)) {
			notice(s_AdminServ, source, PMD);
			return;
		}
		if (!stricmp(cmd,"ADD")) {
			ais->flags |= AI_ADDSA;
			notice(s_AdminServ, source, "SRA %s have permision to modify SA list.", ais->nick);
		} else {
			if (!(ais->flags & AI_ADDSA)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_ADDSA;
				notice(s_AdminServ, source, "SRA %s have not permision to modify SA list.", ais->nick);
			}
		}
	} else if (!stricmp(operflag, "HOLD")) {
		if (!(ais->adflags & ADF_SERVICEADMIN)) {
			notice(s_AdminServ, source, "This flag can set for SA only.");
			return;
		} else if (!is_services_root(source)) {
			notice(s_AdminServ, source, PMD);
			return;
		}
		if (!stricmp(cmd,"ADD")) {
			ais->flags |= AI_HOLD;
			notice(s_AdminServ, source, "SA %s have permision to hold nick/chan.", ais->nick);
		} else {
			if (!(ais->flags & AI_HOLD)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_HOLD;
				notice(s_AdminServ, source, "SA %s have no permision to hold nick/chan.", ais->nick);
			}
		}
	} else if (!stricmp(operflag, "GLOBAL")) {
		if (!(ais->adflags & ADF_SERVICEADMIN)) {
			notice(s_AdminServ, source, "This flag can set for SA only.");
		} else if (!is_services_root(source)) {
			notice(s_AdminServ, source, PMD);
		}
		if (!stricmp(cmd,"ADD")) {
			ais->flags |= AI_GLOBAL;
			notice(s_AdminServ, source, "SA %s have permision to send global notice.", ais->nick);
		} else {
			if (!(ais->flags & AI_GLOBAL)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_GLOBAL;
				notice(s_AdminServ, source, "SA %s have no permision to send global notice.", ais->nick);
			}
		}
	} else if (!stricmp(operflag, "FORBID")) {
		if (!(ais->adflags & ADF_SERVICEADMIN)) {
			notice(s_AdminServ, source, "This flag can set for SA only.");
		} else if (!is_services_root(source)) {
			notice(s_AdminServ, source, PMD);
		}
		if (!stricmp(cmd,"ADD")) {
			ais->flags |= AI_FORBID;
			notice(s_AdminServ, source, "SA %s have permision to forbid nick/channel.", ais->nick);
		} else {
			if (!(ais->flags & AI_FORBID)) {
				notice(s_AdminServ, source, "already unset.");
			} else {
				ais->flags &= ~AI_FORBID;
				notice(s_AdminServ, source, "SA %s have no permision to forbid nick/channel.", ais->nick);
			}
		}
	} else {
		notice(s_AdminServ, source, "incorrect syntax, or permision denied.");
	}	
    } else {
        notice(s_AdminServ, source, NS_UNKNOWC, cmd, s_AdminServ);
    }
}
/**************************************************************/
static void do_setserver(const char *source) {
	char *nick = strtok(NULL, " ");
	char *aserver = strtok(NULL, " ");
	AdminInfo *ais;

	if (!nick || !aserver) {
		notice(s_AdminServ, source, "Syntax: SETSERVER <nick> <server>");
	} else if (!(ais = findadmin(nick))) {
		notice(s_AdminServ, source, "Nick %s does not exist in OPER list.", nick);
	} else {
		if (is_oper_override(source, ais)) {
			if (ais->server)
				free(ais->server);
			ais->server = sstrdup(aserver);
			notice(s_AdminServ, source, "Successful add %s server", nick);
	 	} else {
			notice(s_AdminServ, source, "Permision Denied!");
		}
	}
}
/**************************************************************/
static void do_adminwhois(const char *source) {
	char *nick = strtok(NULL, " ");
        struct tm tm;
        char timebuf[64];
	char buf[512];
	AdminInfo *ais;
	if (!nick) {
		notice(s_AdminServ, source, "Syntax: WHOIS <nick>");
	} else if (is_services_coder(nick)) {
		notice(s_AdminServ, source, "He can do everything!");
	} else if (!(ais = findadmin(nick))) {
		notice(s_AdminServ, source, "Syntax: This nick is not in oper list!");
	} else {
	notice(s_AdminServ, source, "-=Admin WHOIS on \2%s\2", nick);
	*timebuf = 0;
	if (ais->adflags & ADF_SERVICEROOT) {
		if (ais->flags & AI_ADDSRA) {
			notice(s_AdminServ, source, "Rank *\2Services Root\2");
			notice(s_AdminServ, source, "Flags: ADDSRA, ADDSA, HOLD, GLOBAL, FORBID, AKILL, FJOIN");
		} else {
			notice(s_AdminServ, source, "Rank \2Services Root\2");
			notice(s_AdminServ, source, "Flags: HOLD, GLOBAL, FORBID, AKILL");

		}
	} else if (ais->adflags & ADF_SERVICEADMIN) {
			notice(s_AdminServ, source, "Rank \2Services Admin\2");
			notice(s_AdminServ, source, "Flags: AKILL");
	} else if (ais->adflags & ADF_SERVICEOP) {
			notice(s_AdminServ, source, "Rank \2Services Operator\2");
	} else if (ais->adflags & ADF_HELPOP) {
			notice(s_AdminServ, source, "Rank \2Services Helpop\2");
	}
	notice(s_AdminServ, source, "Last modify by: \2%s\2", ais->who);
        tm = *localtime(&ais->added);
        strftime(timebuf, sizeof(timebuf), "%b %d %H:%M:%S %Y %Z", &tm);
        timebuf[sizeof(timebuf)-1] = 0;
	notice(s_AdminServ, source, "Added on      : \2%s\2", timebuf);
	if (ais->server)
	notice(s_AdminServ, source, "Server        : \2%s\2", ais->server);
	if (ais->host)
	notice(s_AdminServ, source, "Info          : \2%s\2", ais->host);
	*buf = 0;
	if (rec_services_coder(source)) {
	        if (ais->flags & AI_ADDSRA)
        	    strcat(buf, "ADDSRA");
	}
        if (ais->flags & AI_ADDSA) {
       	    if (*buf)
               	strcat(buf, ", ");
            strcat(buf, "ADDSA");
       	}
        if (ais->flags & AI_HOLD) {
       	    if (*buf)
               	strcat(buf, ", ");
            strcat(buf, "HOLD");
       	}
        if (ais->flags & AI_GLOBAL) {
       	    if (*buf)
               	strcat(buf, ", ");
            strcat(buf, "GLOBAL");
       	}
        if (ais->flags & AI_SUPEROP) {
       	    if (*buf)
               	strcat(buf, ", ");
            strcat(buf, "SUPEROP");
       	}
        if (ais->flags & AI_AKILL) {
       	    if (*buf)
               	strcat(buf, ", ");
            strcat(buf, "AKILL");
       	}
        if (ais->flags & AI_ABUSETM) {
       	    if (*buf)
               	strcat(buf, ", ");
            strcat(buf, "ABUSETM");
       	}
        if (ais->flags & AI_FORCEJOIN) {
       	    if (*buf)
               	strcat(buf, ", ");
            strcat(buf, "FJOIN");
       	}
        if (!*buf)
            strscpy(buf, "None", 4);
	notice(s_AdminServ, source, "Flags: %s", buf);

	if (ais->flags & AI_QUIT) {
		notice(s_AdminServ, source, "* Suspended");
        }
	notice(s_AdminServ, source, "-=End Whois");
	}
}
/**************************************************************/
static void do_addoper(const char *source) {
	AdminInfo *ais;
	char *cmd = strtok(NULL, " ");
	if (!cmd) {
		notice(s_AdminServ, source, "Syntax: OPER <command> <param>");
	} else if (!stricmp(cmd,"ADD")) {
		char *nick = strtok(NULL, " ");
		char *operflag = strtok(NULL, " ");
		if (!nick || !operflag) {
			notice(s_AdminServ, source, "Syntax: OPER ADD <nick> SRA|SA|SOP|HOP");
		} else if (!stricmp(operflag, "SRA") && is_oper_cando(source, 1)) {
			if (findadmin(nick)) {
				notice(s_AdminServ, source, "This nick is already an OPER please use command OPERSET to change status.");
				return;
			}
			ais = makeadmin(nick);
			if (!ais) {
				notice(s_AdminServ, source, "Fialed add.");
				return;
			}
			ais->nick = sstrdup(nick);
			ais->host = NULL;
			ais->server = NULL;
			ais->markline = 0;
			ais->adflags |= ADF_SERVICEROOT;
			ais->flags |= AI_ADDSA;
			ais->who = sstrdup(source);
			ais->added = CTime;
			notice(s_AdminServ, source, "%s now an Services Root.", nick);
			wallops(NULL,"%s is now a Services Root added by %s", nick, source);
		} else if (!stricmp(operflag, "SA") && is_oper_cando(source,2)) {
			if (findadmin(nick)) {
				notice(s_AdminServ, source, "This nick is already an OPER please use command OPERSET to change status.");
				return;
			}
			ais = makeadmin(nick);
			if (!ais) {
				notice(s_AdminServ, source, "Fialed add.");
				return;
			}
			ais->nick = sstrdup(nick);
			ais->host = NULL;
			ais->who = sstrdup(source);
			ais->server = NULL;
			ais->markline = 0;
			ais->adflags |= ADF_SERVICEADMIN;
			ais->added = CTime;
			notice(s_AdminServ, source, "%s now an Services Admin.", nick);
			wallops(NULL,"%s is now a Services Admin added by %s", nick, source);
		} else if (!stricmp(operflag, "SOP") && is_services_admin(source)) {
			if (findadmin(nick)) {
				notice(s_AdminServ, source, "This nick is already an OPER please use command OPERSET to change status.");
				return;
			}
			ais = makeadmin(nick);
			if (!ais) {
				notice(s_AdminServ, source, "Fialed add.");
				return;
			}
			ais->nick = sstrdup(nick);
			ais->host = NULL;
			ais->who = sstrdup(source);
			ais->server = NULL;
			ais->markline = 0;
			ais->adflags |= ADF_SERVICEOP;
			ais->added = CTime;
			notice(s_AdminServ, source, "%s now an Services Oper.", nick);
			wallops(NULL,"%s is now a Services Oper added by %s", nick, source);
		} else if (!stricmp(operflag, "HOP") && is_services_admin(source)) {
			if (findadmin(nick)) {
				notice(s_AdminServ, source, "This nick is already an OPER please use command OPERSET to change status.");
				return;
			}
			ais = makeadmin(nick);
			if (!ais) {
				notice(s_AdminServ, source, "Fialed add.");
				return;
			}
			ais->nick = sstrdup(nick);
			ais->host = NULL;
			ais->who = sstrdup(source);
			ais->server = NULL;
			ais->markline = 0;
			ais->adflags |= ADF_HELPOP;
			ais->added = CTime;
			notice(s_AdminServ, source, "%s is now a Services Helpop.", nick);
			wallops(NULL,"%s is now a Services Helpop added by %s", nick, source);
		} else {
			notice(s_AdminServ, source, "incorrect syntax, or permision denied.");
		} /* ADD TYPE */
	} else if (!stricmp(cmd, "DEL")) {
		char *nick = strtok(NULL, " ");
		if (!nick) {
			notice(s_AdminServ, source, "Syntax: DEL nick.");
		} else {
			ais = findadmin(nick);
			if (!ais) {
				notice(s_AdminServ, source, "This nick is not in oper list.");
				return;
			}
			if (!is_oper_override(source, ais)) {
				notice(s_AdminServ, source, "Permision denied");
				return;
			}
			deladmin(ais);
			notice(s_AdminServ, source, "OPER %s successful deleted.", nick);
		}
	} else if (!stricmp(cmd, "LIST")) {
		char *opertype = strtok(NULL, " ");
		if (!opertype) {
			notice(s_AdminServ, source, "Syntax: LIST <SRA|SA|SOP|HOP>.");
		} else if (!stricmp(opertype, "SRA")) {
			int i;
			notice(s_AdminServ, source, "\2SRA Listing\2.");
			for (i = 33; i < 256; ++i) {
				for (ais = adminlists[i]; ais; ais = ais->next)
					if (ais->adflags & ADF_SERVICEROOT)
					 notice(s_AdminServ, source, "%-10s \2%-15s\2", ais->nick,ais->who);
			}
			notice(s_AdminServ, source, "\2End of SRA Listing\2.");
		} else if (!stricmp(opertype, "SA")) {
			int i;
			notice(s_AdminServ, source, "\2SA Listing\2.");
			for (i = 33; i < 256; ++i) {
				for (ais = adminlists[i]; ais; ais = ais->next)
					if (ais->adflags & ADF_SERVICEADMIN)
					 notice(s_AdminServ, source, "%-10s \2%-15s\2", ais->nick,ais->who);
			}
			notice(s_AdminServ, source, "\2End of SA Listing\2.");
		} else if (!stricmp(opertype, "SOP")) {
			int i;
			notice(s_AdminServ, source, "\2SOP Listing\2.");
			for (i = 33; i < 256; ++i) {
				for (ais = adminlists[i]; ais; ais = ais->next)
					if (ais->adflags & ADF_SERVICEOP)
					 notice(s_AdminServ, source, "%-10s \2%-15s\2", ais->nick,ais->who);
			}
			notice(s_AdminServ, source, "\2End of SOP  Listing\2.");
		} else if (!stricmp(opertype, "HOP")) {
			int i;
			notice(s_AdminServ, source, "\2Helpop  Listing\2.");
			for (i = 33; i < 256; ++i) {
				for (ais = adminlists[i]; ais; ais = ais->next)
					if (ais->adflags & ADF_HELPOP)
					 notice(s_AdminServ, source, "%-10s \2%-15s\2", ais->nick,ais->who);
			}
			notice(s_AdminServ, source, "\2End of HOP  Listing\2.");
		} else {
			notice(s_AdminServ, source, "Syntax: LIST <SRA|SA|SOP|HOP>.");
		}
	}/* CMD */
} 

/**************************************************************/
/*
 * 1 Add SRA
 * 2 Add SA
 * 3 Can do hold
 * 5 Can have Super access
 * 6 Can place akill
 * 7 Abuse Team
 * 8 Forbid
 * 9 FJOIN
 */
int is_oper_cando(const char *source, int operflag) {
	AdminInfo *ais;
	int x=0;

	if (is_services_coder(source) && (operflag != 5))
		return 1;

	if (!(ais=findadmin(source))) /* No Access */
		return 0;

	if (!is_on_id_list(source, source)) /* Not Identify yourself? */
		return 0;

	if (ais->adflags & ADF_SERVICEROOT) {
		if ((operflag == 1) && (ais->flags & AI_ADDSRA))
			return 1;
		if ((operflag == 2) && (ais->flags & AI_ADDSA))
			return 1;
		x = 1;
	}

	if (ais->flags & AI_QUIT)
		return 0;

	if (x || ais->adflags & ADF_SERVICEADMIN) {
		if ((operflag == 3) && ((ais->flags & AI_HOLD) || x))
			return 1;
		if ((operflag == 4) && ((ais->flags & AI_GLOBAL) || x))
			return 1;
		if ((operflag == 8) && ((ais->flags & AI_FORBID) || x))
			return 1;
		x = 1;
	}

	if ((operflag == 6) && ((ais->flags & AI_AKILL) || x))
		return 1;

	if ((operflag == 7) && (ais->flags & AI_ABUSETM))
		return 1;

	if ((operflag == 5) && (ais->flags & AI_SUPEROP))
		return 1;

	if ((operflag == 9) && (ais->flags & AI_FORCEJOIN))
		return 1;

	return 0;
}
/**************************************************************/
int is_oper_override(const char *source, AdminInfo *ais) {
	
	if (!ais)
		return 1;
	if (rec_services_coder(source)) /* Coder Alway override */
		return 1;
	if (ais->flags & AI_ADDSRA) /* No one can override SRA* */
		return 0;
	if (is_oper_cando(source, 1)) /* SRA* Can do everything */
		return 1;
	if (ais->adflags & ADF_SERVICEROOT) /* SRA can't override SRA */
		return 0;
	if (rec_services_root(source))
		return 1; /* SRA override SA */
	if (rec_services_admin(source));
		return 1;
	return 0;
}
/**************************************************************/
int is_services_coder(const char *nick)
{
    if (!is_oper(nick))
            return 0;
    if (is_nickservices_coder(nick))
            return 1;
    return 0;
}
int is_nickservices_coder(const char *nick)
{
    if (!is_on_id_list(nick, SERVICES_MASTER))
            return 0;
    return 1;
}
int rec_services_coder(const char *nick)
{
    NickInfo *ni;
    if (!is_oper(nick))
            return 0;
    if (is_services_coder(nick))
        return 1;
    if (stricmp(nick, SERVICES_MASTER))
           return 0;
    ni = findnick(nick);
    if (ni && ni->flags & NI_RECOGNIZED)
        return 1;
    return 0;
}


/**************************************************************/
int rec_services_root(const char *nick)
{
    unsigned int i;
    User *u = finduser(nick);

    if (!is_oper(nick))
            return 0;

    if (is_services_root(nick))
	return 1;

    if (!u)
	return 0;

    for (i=0; i < MAX_IDS && u->id_nicks[i]; i++) {
        if (is_inolevel(u->id_nicks[i], 2)) {
            return 1;
        }
    }
    return 0;
}
/* -------------------------------------------- */
int is_services_root(const char *nick)
{
	int i=0;
	AdminInfo *ais;
	if (is_services_coder(nick))
		return 1;
	if (!(ais=findadmin(nick)))
		return 0;
	if (is_on_id_list(nick, nick))
		i = 1;
	if ((ais->adflags & ADF_SERVICEROOT) && i)
		return 1;
	return 0;
}
/************************************************/
int is_services_admin(const char *nick)
{
	int i=0;
	AdminInfo *ais;
	if (is_services_coder(nick))
		return 1;
	if (!(ais=findadmin(nick)))
		return 0;
	if (is_on_id_list(nick, nick))
		i = 1;
	if ((ais->adflags & ADF_SERVICEROOT) && i)
		return 1;
	if ((ais->adflags & ADF_SERVICEADMIN) && i)
		return 1;
	return 0;
}
/**************************************************/
int rec_services_admin(const char *nick)
{
    unsigned int i;
    User *u = finduser(nick);

    if (!is_oper(nick))
            return 0;

    if (is_services_admin(nick))
	return 1;
    if (!u)
	return 0;
    for (i=0; i < MAX_IDS && u->id_nicks[i]; i++) {
        if (is_inolevel(u->id_nicks[i], 3)) {
            return 1;
        }
    }
    return 0;
}
/**************************************************/
int is_services_oper(const char *nick)
{
	int i=0;
	AdminInfo *ais;
	if (is_services_coder(nick))
		return 1;
	if (!(ais=findadmin(nick)))
		return 0;
	if (is_on_id_list(nick, nick))
		i = 1;
	if ((ais->adflags & ADF_SERVICEROOT) && i)
		return 1;
	if ((ais->adflags & ADF_SERVICEADMIN) && i)
		return 1;
	if ((ais->adflags & ADF_SERVICEOP) && i)
		return 1;
	return 0;
}
/* ------------------------------------------------ */
int rec_services_oper(const char *nick)
{
    unsigned int i;
    User *u = finduser(nick);

    if (is_services_oper(nick))
	return 1;
    if (!u)
	return 0;
    for (i=0; i < MAX_IDS && u->id_nicks[i]; i++) {
        if (is_inolevel(u->id_nicks[i], 4)) {
            return 1;
        }
    }

    return 0;
}
/****************************************************/
int is_services_helpop(const char *nick) {
	int i=0;
	AdminInfo *ais;
	if (is_services_coder(nick))
		return 1;
	if (!(ais=findadmin(nick)))
		return 0;
	if (is_on_id_list(nick, nick))
		i = 1;
	if ((ais->adflags & ADF_SERVICEROOT) && i)
		return 1;
	if ((ais->adflags & ADF_SERVICEADMIN) && i)
		return 1;
	if ((ais->adflags & ADF_SERVICEOP) && i)
		return 1;
	if ((ais->adflags & ADF_HELPOP) && i)
		return 1;
	return 0;
}
/* --------------------------------------- */
int rec_services_helpop(const char *nick) {
    unsigned int i;
    User *u = finduser(nick);
    if (is_services_helpop(nick))
	return 1;
    if (!u)
	return 0;
    for (i=0; i < MAX_IDS && u->id_nicks[i]; i++) {
        if (is_inolevel(u->id_nicks[i],5)) {
            return 1;
        }
    }
    return 0;
}
/************************************************************/
void os_remove_nick(const char *nick)
{
    AdminInfo *ais;
    ais=findadmin(nick);
    if (ais)
	deladmin(ais);
}
/************************************************************/
static void showadmin(const char *source, int dowhat) {
   char buf[BUFSIZE];
   AdminInfo *ais;
   NickInfo *ni;

   int i, avail=0, all=0;
   *buf = 0;
   for (i = 33; i < 256; ++i) {
        for (ais = adminlists[i]; ais; ais = ais->next)
                if ((ais->adflags & (ADF_SERVICEADMIN|ADF_SERVICEROOT)) && !(ais->flags & AI_QUIT)) {
			all++;
			ni = findnick(ais->nick);
			if (ni && (ni->flags & NI_IDENTIFIED)) {
                		strcat(buf, "\2");
		                strcat(buf, ais->nick);
                		strcat(buf, "\2");
                		strcat(buf, " ");
				avail++;
				if (strlen(buf) > 60) {
					send_cmd(server_name, "372 %s :- %s", source, buf);
					*buf=0;
				}
			} else {
		                strcat(buf, ais->nick);
		                strcat(buf, " ");
				if (strlen(buf) > 60) {
					send_cmd(server_name, "372 %s :- %s", source, buf);
					*buf=0;
				}
			}
		}
    }
    send_cmd(server_name, "372 %s :- %s", source, buf);
    send_cmd(server_name, "372 %s :-", source);
    send_cmd(server_name, "372 %s :- %d of %d nicks available.",
         source, avail, all);
    send_cmd(server_name, "372 %s :- Note: The nicks that are in \2bold\2"
                 " are currently online", source);
}
/************************************************************/
void sendmotd(const char *source)
{
    NickInfo *ni;

    send_cmd(server_name, "372 %s :- This list of people can retrieve"
                " lost passwords:", source);
    send_cmd(server_name, "372 %s :-", source);
    ni = findnick(SERVICES_MASTER);
    if (ni && (ni->flags & NI_IDENTIFIED)) {
	    send_cmd(server_name, "372 %s :- Services Master \2%s\2", source, SERVICES_MASTER);
    } else {
	    send_cmd(server_name, "372 %s :- Services Master %s", source, SERVICES_MASTER);
    }
    send_cmd(server_name, "372 %s :-", source);
    showadmin(source,1) ;
}

/***********************************************/
int is_inolevel(const char *source, int alevel)
{
	AdminInfo *ais;

	if (!stricmp(source, SERVICES_MASTER) && (alevel == 1))
		return 1;

	ais=findadmin(source);

	if (!ais)
  	   return 0;

	if ((ais->adflags & ADF_SERVICEROOT) && (alevel == 2))
		return 1;

	if ((ais->adflags & ADF_SERVICEADMIN) && (alevel == 3))
		return 1;
	if ((ais->adflags & ADF_SERVICEOP) && (alevel == 4))
		return 1;
	if ((ais->adflags & ADF_HELPOP) && (alevel == 5))
		return 1;
	if ((ais->flags & AI_ABUSETM) && (alevel == 6))
		return 1;

	return 0;
}

int is_adminlevel(const char *source)
{
	AdminInfo *ais;

	if (!stricmp(source, SERVICES_MASTER))
		return 1;

	ais=findadmin(source);

	if (!ais)
  	   return 0;

	if (ais->adflags & ADF_SERVICEROOT)
		return 2;
	if (ais->adflags & ADF_SERVICEADMIN)
		return 3;
	if (ais->adflags & ADF_SERVICEOP)
		return 4;
	if (ais->adflags & ADF_HELPOP)
		return 5;

	return 0;
}

int is_inilevel(const char *source, int alevel)
{
	AdminInfo *ais;

	if (!stricmp(source, SERVICES_MASTER))
		return 1;

	ais=findadmin(source);

	if (!ais)
  	   return 0;

	return 1;
}
