/*
 *  This file is provided for use with the unix-socket-faq.  It is public
 *  domain, and may be copied freely.  There is no copyright on it.  The
 *  original work was by Vic Metcalfe (vic@brutus.tlug.org), and any
 *  modifications made to that work were made with the understanding that
 *  the finished work would be in the public domain.
 *
 *  If you have found a bug, please pass it on to me at the above address
 *  acknowledging that there will be no copyright on your work.
 *
 *  The most recent version of this file, and the unix-socket-faq can be
 *  found at http://www.interlog.com/~vic/sock-faq/.
 */

#include "sockhelp.h"
#include "options.h"
#include "list.h"
#include "aClient.h"
#include "extern.h"
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fstream>
#include <stdlib.h>
#include <fcntl.h>
#include <signal.h>
//#include <sys/types.h>
//#include <sys/wait.h>
//#include <errno.h>



using std::ofstream;
using std::ifstream;
using std::cerr;
using std::endl;

char *PACKAGE = "GameServ";
char *VERSION = "1.1.8";

int sock;
int day;

List<aClient> clients;

void save_day();
void load_day();
void prettyIntro();

// Make this a daemon
int daemon(int nochdir, int noclose);

// Close all file descriptors from >= fd
void closeall(int fd);

int main(int argc, char *argv[])
{
  char buffer[1024], buf[1024];
  int connected = 1;
  char *cmd, *source = NULL, *conf = "gameserv.conf";
  srand(time(NULL));

  if (argc > 1)
  {
	if ( argc > 2 || stricmp(argv[1], "--help") == 0)
	{
	    cout << "Usage: gameserv [options] [configfile]" << endl;
	    cout << "Options:" << endl;
	    cout << "--help                        Displays this help dialogue" << endl;
	    return 1;
	}
	conf = new char[strlen(argv[1])];
	strcpy(conf, argv[1]);
  }

  prettyIntro();

  if (load_config_file(conf))
  {
  	cout << "Config file loaded ok...\n"
             << "Turning into a daemon" << endl;
  }
  else
  	exit(2);

    // Turn into a daemon
    if (daemon(1,0) < 0)
    {
        perror("Could not turn into a daemon");
        exit(2);
    }

  ignore_pipe();
  sock = make_connection(remoteport, SOCK_STREAM, remoteserver);
  if (sock == -1) {
    fprintf(stderr,"make_connection failed.\n");
    unload_config_file();
    return -1;
  }

#ifdef UNREAL
	raw("PROTOCTL NICKv2 VHP");
	raw("PASS :%s", remotepass);
	raw("SERVER %s 1 :%s", servername, servername);
	raw("NICK %S 1 %d %S %s %s %d +owghraAxNt %s :%s v%s", time(NULL), gshost, 
		servername, time(NULL), gshost, PACKAGE, VERSION);
	raw(":%S JOIN %s", c_Forest);
	raw(":%S MODE %s +mtn", c_Forest);
#elif defined(BAHAMUT)
	raw("PASS %s :TS", remotepass);
	raw("SERVER %s 1 :%s", servername, servername);
        raw("NICK %S 1 %d +o %s %s %s 0 :GameServ", time(NULL), gsident, gshost, 
		servername);
	raw(":%s SJOIN %d %d %s +mnt :@%S", servername, time(NULL), time(NULL), c_Forest);
#elif defined(HYBRID)
	raw("PASS %s :TS", remotepass);
	raw("SERVER %s 1 :%s", servername, servername);
	raw("NICK %S 1 %d +o %s %s %s :GameServ", time(NULL), gsident, gshost,
		servername);
	// Sending a timestamp of 1 to force ops.
	raw(":%s SJOIN 1 %s +ntm :@%S", servername, c_Forest);
#elif defined(P10)
	// Server numeric is: []  <-- must be unique
	raw("PASS :%s", remotepass);
	raw("SERVER %s 1 %d %d P10 []AAF :%s", servername, time(NULL), time(NULL), servername);
	raw("[] N %S 1 %d %s %s DAqAoB %s :%S", time(NULL), gsident, gshost, gsnum);
	raw("[] B %s %d +tnm %s:o", c_Forest, time(NULL) - 864000, gsnum);
#endif

#if defined(P10)
	raw("%s T %s :%s", gsnum, c_Forest, c_ForestTopic);
	raw("[] EB");  // End burst
#else 
#ifndef HYBRID
	raw(":%S MODE %s +o %S", c_Forest);
#endif
	raw(":%S TOPIC %s :%s", c_Forest, c_ForestTopic);
#endif

  sock_gets(sock,buffer,sizeof(buffer)-1); /* -1 added thanks to
    David Duchene <dave@ltd.com> for pointing out the possible
    buffer overflow resulting from the linefeed added below. */


  #ifdef DEBUGMODE
      log("Server: %s",buffer);
  #endif

  init_masters();
  load_gs_dbase();
  load_day();
  long int loadtime = time(NULL);
  long int currentTime;
  long int oldTime = loadtime;
  bool loaded = false;

  if (load_monsters() == false)
	goto end;

  while (connected) {
      if (sock_gets(sock,buffer,sizeof(buffer)) == -1) {
        connected = 0;
      }
	strcpy(buf, buffer);

      #if !defined(P10)
        if (buffer[0] == ':')
	{
	    source = strtok(buf, " ");
	    cmd = strtok(NULL, " ");
	}
	else
	    cmd = strtok(buf, " ");
      #else
	    source = strtok(buf, " ");
	    cmd = strtok(NULL, " ");
      #endif

	#ifdef DEBUGMODE
	    log("Server: %s", buffer);
	#endif

	// Wait N seconds then we're loaded.
        if (!loaded)
	{
	    if (time(NULL) >= welcomedelay + loadtime)
		loaded = true;
	}

	// Save the player data every updateperiod seconds
	currentTime = time(NULL);
	if (currentTime - oldTime >= updateperiod)
	{
	    oldTime = currentTime;
	    save_gs_dbase();
	}


      #if !defined(P10)
	if (stricmp(cmd, "PING") == 0) {
	    char *timestamp;
	    timestamp = strtok(NULL, "");
	    raw("PONG %s", timestamp);
      #else
	if (stricmp(cmd, "G") == 0) {
	    char *timestamp;
	    timestamp = strtok(NULL, " ");
	    raw("[] Z [] %s 0 %s", timestamp + 1, timestamp);
      #endif
	#ifdef P10
	} else if (stricmp(cmd, "EB") == 0) {
	    raw("[] EA");
	#endif
	} else if (stricmp(cmd, "VERSION") == 0) {
	    char *server;
	    server = strtok(NULL, " ");
	    server++;
	    raw(":%s 351 %s %s_%s. %s", servername, source+1, PACKAGE, VERSION, servername);
      #if !defined(P10)
	} else if (strncmp(cmd, "NICK", 4) == 0) {
	    if (buffer[0] == ':')
	    {
		aClient *tempPtr;
		if ((tempPtr = find((source + 1))))
		{
		    char *nick;
		    nick = strtok(NULL, " ");
		    tempPtr->setNick(nick);
		}
	    }
	    else
	    {
		char *nick;
      #else
       } else if (stricmp(cmd, "N") == 0 && strlen(source) == 2) {
	    {	
		char *nick, *realnick;
		realnick = strtok(NULL, " ");

		for (int x = 0; x < 5; x++)
		      nick = strtok(NULL, " ");

		if (nick[0] == '+')
		{
		    #ifdef DEBUGMODE
		    log ("aClient has modes");
		    #endif

		    // Searching for the +r mode (extra parameter)
		    for (unsigned int count = 1; count < strlen(nick); count++)
		    {
			if (nick[count] == 'r')
			{
			    nick = strtok(NULL, " ");
			    break;
			}
		    }
		    nick = strtok(NULL, " ");
		}
      #endif
		aClient *newuser;

		nick = strtok(NULL, " ");

		#ifdef P10
		    newuser = new aClient(nick, realnick);
		#else
		    newuser = new aClient(nick);
		#endif


		if (loaded)
		#ifdef P10
		    notice(s_GameServ, nick, welcomemsg, realnick);
		#else
		    notice(s_GameServ, nick, welcomemsg, nick);
		#endif

		clients.insertAtBack(newuser);
		delete newuser;
	    }
      #if defined(P10)
	} else if (stricmp(cmd, "Q") == 0) {
      #else
	} else if (stricmp(cmd, "QUIT") == 0) {
      #endif
	    aClient *quitter;
	    char z = source[0];

	    if (z == ':')
		source++;

	    if ((quitter = find(source)))
		clients.remove(quitter);
	    if ((quitter = findIRCplayer(source)))
	    {
		quitter->setNick("!NULL!");
		quitter->stats->user = NULL; // Unidentify them
	    }

	    if (z == ':')
		source--;

      #if defined(P10)
	} else if (stricmp(cmd, "P") == 0) {
	    char *rest, *dest;
	    char *longname;
	    longname = new char[strlen(s_GameServ) + strlen(servername) + 2];

	    sprintf(longname, "%S@%s", servername);

	    dest = strtok(NULL, " ");
	    rest = strtok(NULL, "");
	    if (stricmp(dest, gsnum) == 0 || stricmp(dest, longname) == 0)
	    {
		delete [] longname;
		gameserv(source, rest);
	    }
	    else if (stricmp(dest, c_Forest) == 0)
	    {
		delete [] longname;
		forest(source, rest);
	    }
      #else
	} else if (stricmp(cmd, "PRIVMSG") == 0) {
	    char *rest, *dest;
	    dest = strtok(NULL, " ");
	    rest = strtok(NULL, "");
	    if (strnicmp(dest, s_GameServ, strlen(s_GameServ)) == 0)
		gameserv(source, rest);
	    else if (stricmp(dest, c_Forest) == 0)
		forest(source, rest);
      #endif
	} else if (stricmp(cmd, "JOIN") == 0) {
	    char *channel;
	    channel = strtok(NULL, " ");
	    if (stricmp(channel, c_Forest) == 0 && is_playing(source + 1))
		raw(":%S MODE %s +v %s", c_Forest, (source + 1));

	#if defined(BAHAMUT)
	} else if (stricmp(cmd, "SJOIN") == 0) {
	    char *channel, *nick, *tmp, *rest;
	    strtok(NULL, " "); // Ignore the TS
#ifndef HYBRID
	    strtok(NULL, " "); // Ignore the TS
#endif
	    channel = strtok(NULL, " ");
           rest = strtok(NULL, "");
           tmp = strchr(rest, ':');
           tmp++;
           nick = strtok(tmp, " ");
           while (nick != NULL)
           {
               if (*nick == '@')
                   nick++;
               if (*nick == '+')
                   nick++; // Assume for users set op and voice, they
                            // are never passed as +@nick
               if (stricmp(channel, c_Forest) == 0 && is_playing(nick))
                   raw(":%S MODE %s +v %s", channel, nick);

               nick = strtok(NULL, " ");
           }
#endif
	} else {
	    #ifdef DEBUGMODE
		log("Unrecognized Message: cmd = %s   source = %s", cmd, source);
	    #endif
	}
  }

  end:

  save_gs_dbase();
  save_day();

  delete_monsters();
  delete_masters();

  #ifdef DEBUGMODE
      log("<CLOSED>");
  #endif

  close(sock);
  unload_config_file();
  return 0;
}

aClient *find(char *nick)
{
	return findbynick(nick);
}

aClient *find(const char *nick)
{
	return findbynick(nick);
}

#ifdef P10

aClient *findbyrealnick(char *realnick)
{
   ListNode <aClient> *newPtr;
    newPtr = clients.First();

    aClient *client = NULL;

    while (newPtr)
    {
	client = newPtr->getData();
	    if (stricmp(client->getRealNick(), realnick) == 0)
	    return client;
	client = NULL;
	newPtr = newPtr->Next();
    }
    return client;    
}

#else

aClient *findbyrealnick(char *realnick)
{
  return findbynick(realnick);
}

#endif

aClient *findbynick(char *nick)
{
    ListNode <aClient> *newPtr;
    newPtr = clients.First();

    aClient *client = NULL;

    while (newPtr)
    {
	client = newPtr->getData();
	#ifdef P10
	    if (strcmp(client->getNick(), nick) == 0)
	#else
	    if (stricmp(client->getNick(), nick) == 0)
	#endif
	    return client;
	client = NULL;
	newPtr = newPtr->Next();
    }
    return client;    
}

aClient *findIRCplayer(const char *nick)
{
    ListNode <aClient> *newPtr;
    aClient *p = NULL;

    for (newPtr = players.First(); newPtr; newPtr = newPtr->Next())
    {
	p = newPtr->getData();
	#ifdef P10
	    if (strcmp(p->getNick(), nick) == 0)
	#else
	    if (stricmp(p->getNick(), nick) == 0)
	#endif
	    return p;
	p = NULL;
    }
    return NULL;
}
aClient *findplayer(const char *name)
{
    ListNode <aClient> *newPtr;
    Player *p = NULL;

    for (newPtr = players.First(); newPtr; newPtr = newPtr->Next())
    {
	p = newPtr->getData()->stats;
	if (stricmp(p->name, name) == 0)
	    return newPtr->getData();
	p = NULL;
    }
    return NULL;
}

aClient *findbynick(const char *nick)
{
    ListNode <aClient> *newPtr;
    newPtr = clients.First();

    aClient *client = NULL;

    while (newPtr)
    {
	client = newPtr->getData();
	#ifdef P10
	    if (strcmp(client->getNick(), nick) == 0)
	#else
	    if (stricmp(client->getNick(), nick) == 0)
	#endif
	    return client;
	client = NULL;
	newPtr = newPtr->Next();
    }
    return client;    
}

void load_day()
{
    ifstream infile;

    infile.open(".gsday");

    if (infile.fail())
    {
	#ifdef DEBUGMODE
	    log("Error opening .gsday");
	#endif

	generate:
        #ifdef DEBUGMODE
	    log("Generating new day");
	#endif
    	struct tm *tm;
	time_t ti;
	time(&ti);
	tm = localtime(&ti);

	day = tm->tm_mday;

        save_day();
	return;
    }

    infile >> day;
    infile.close();
    if (day < 1 || day > 31)
	goto generate;
}

void save_day()
{
    ofstream outfile;

    outfile.open(".gsday");

    if (outfile.fail())
    {
	log("Error creating new file .gsday");
	return;
    }

    outfile << day << endl;

    outfile.close();
}

/* daemon() - detach process from user and disappear into the background
 * returns -1 on failure, but you can't do much except exit in that case
 * since we may already have forked. This is based on the BSD version,
 * so the caller is responsible for things like the umask, etc.
 */

/* believed to work on all Posix systems */

int daemon(int nochdir, int noclose)
{
    pid_t pid;
    switch (pid = fork())
    {
        case 0:  break;
        case -1: return -1;
        default: _exit(0);          /* exit the original process */
    }

    if (setsid() < 0)               /* shoudn't fail */
      return -1;

    /* dyke out this switch if you want to acquire a control tty in */
    /* the future -- not normally advisable for daemons */

    switch (pid = fork())
    {
        case 0:  break;
        case -1: return -1;
        default: 
		ofstream outfile;
		outfile.open("gameserv.pid");
    		if (outfile.fail())
		    cerr << "Unable to open gameserv.pid" << endl;
		outfile << pid << endl;
    		outfile.close();

	_exit(0);
    }

    if (!nochdir)
      chdir("/");

    if (!noclose)
    {
        closeall(0);
        open("/dev/null",O_RDWR);
        dup(0); dup(0);
    }

    return 0;
}


/* closeall() -- close all FDs >= a specified value */

void closeall(int fd)
{
    int fdlimit = sysconf(_SC_OPEN_MAX);

    while (fd < fdlimit)
      close(fd++);
}

void prettyIntro()
{
cout << endl;
cout << "  GGGG     AAA   MM    MM EEEEEEE  SSSSS  EEEEEEE RRRRRR  VV     VV " << endl;
cout << " GG  GG   AAAAA  MMM  MMM EE      SS      EE      RR   RR VV     VV " << endl;
cout << "GG       AA   AA MM MM MM EEEEE    SSSSS  EEEEE   RRRRRR   VV   VV  " << endl;
cout << "GG   GGG AAAAAAA MM    MM EE           SS EE      RR  RR    VV VV   " << endl;
cout << "G     G  AA   AA MM    MM EEEEEEE  SSSSS  EEEEEEE RR   RR    VVV" << endl;
cout << " GGGGG                                                        V\n\n" << endl;
cout << "Version: " << VERSION << endl;
}
