/* Header for common mask data structure.
 *
 * IRC Services is copyright (c) 1996-2004 Andrew Church.
 *     E-mail: <achurch@achurch.org>
 * Parts written by Andrew Kempe and others.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#ifndef MASKDATA_H
#define MASKDATA_H

/*************************************************************************/

/* This structure and the corresponding functions are used by the autokill,
 * exception, and S-line modules.
 */

typedef struct maskdata_ MaskData;
struct maskdata_ {
    MaskData *next, *prev;
    int modified;
    int locked;

    int num;		/* Index number */
    char *mask;
    int16 limit;	/* For exceptions only */
    char *reason;
    char who[NICKMAX];
    time_t time;
    time_t expires;	/* Or 0 for no expiry */
    time_t lastused;	/* Last time used, 0 if never used */
};

/* Types of mask data: */
#define MD_AKILL	0
#define MD_EXCLUSION	1
#define MD_EXCEPTION	2
#define MD_SGLINE	'G'	/* DO NOT CHANGE: some code relies on */
#define MD_SQLINE	'Q'	/* these values being the same as the */
#define MD_SZLINE	'Z'	/* corresponding S-line letter        */

/* Maximum number of mask data items for a single type: */
#define MAX_MASKDATA	32767

/*************************************************************************/

/* Database functions: */

E MaskData *add_maskdata(uint8 type, MaskData *data);
E void del_maskdata(uint8 type, MaskData *data);
E MaskData *get_maskdata(uint8 type, const char *mask);
E MaskData *get_matching_maskdata(uint8 type, const char *str);
E void put_maskdata(uint8 type, MaskData *data);
E MaskData *first_maskdata(uint8 type);
E MaskData *next_maskdata(uint8 type);
E int maskdata_count(uint8 type);

/* Exported for database use: */
E int check_expire_maskdata(uint8 type, MaskData *md);

/*************************************************************************/
/*************************************************************************/

/* OperServ internal stuff. */

/*************************************************************************/

/* Data structure for MaskData-related commands: */

typedef struct {
    const char *name;		/* Command name */
    uint8 md_type;		/* MaskData type */
    time_t *def_expiry_ptr;	/* Pointer to default expiry time */

    /* Various messages.  sprintf() parameters are given in order in
     * parentheses. */

    /* Syntax message (%s: command name) */
    int msg_syntax;
    /* Syntax messages for ADD and DEL subcommands (%s: command name) */
    int msg_add_syntax, msg_del_syntax;
    /* Message for adding to a full list (%s: command name) */
    int msg_add_too_many;
    /* Message for adding a duplicate record (%s: mask, %s: command name) */
    int msg_add_exists;
    /* Message for successful add (%s: mask, %s: command name) */
    int msg_added;
    /* Message for no record found on delete (%s: mask, %s: command name) */
    int msg_del_not_found;
    /* Message for successful delete (%s: mask, %s: command name) */
    int msg_deleted;
    /* Message for list header (%s: command name) */
    int msg_list_header;
    /* Message for LIST format (%s: mask, %s: reason) */
    int msg_list_list_format;
    /* Message for VIEW format (%s: mask, %s: who, %s: set time, %s: last
     * used time, %s: expire time, %s: reason) */
    int msg_list_view_format;
    /* Message for VIEW format for unused records (%s: mask, %s: who, %s:
     * set time, %s: expire time, %s: reason) */
    int msg_list_view_unused;
    /* Message for record count (%d: count, %s: command name) */
    int msg_count;

    /* Helper functions called by do_maskdata_cmd(). */

    /* Make any alterations to the mask necessary for addition or deletion;
     * if not NULL, called before any other checks (other than for NULL)
     * are performed. */
    void (*mangle_mask)(char *mask);
    /* Check whether the mask is appropriate; return 1 if OK, else 0.
     * The mask and expiry time may be modified.  If NULL, any mask and
     * expiry time are accepted. */
    int (*check_add_mask)(User *u, char *mask, time_t *expiry_ptr);
    /* Operations to perform on mask addition.  If NULL, nothing is done. */
    void (*do_add_mask)(User *u, uint8 type, MaskData *md);
    /* Operations to perform on mask deletion.  If NULL, nothing is done. */
    void (*do_del_mask)(User *u, uint8 type, MaskData *md);
    /* Handler for unknown commands; should return 1 if the command was
     * handled, 0 otherwise. */
    int (*do_unknown_cmd)(User *u, const char *cmd, char *mask);
} MaskDataCmdInfo;

/*************************************************************************/

/* MaskData command related functions: */

E void do_maskdata_cmd(MaskDataCmdInfo *info, User *u);
E char *make_reason(const char *format, const MaskData *data);

/*************************************************************************/

#endif	/* MASKDATA_H */
