/* Include file for high-level encryption routines.
 *
 * IRC Services is copyright (c) 1996-2005 Andrew Church.
 *     E-mail: <achurch@achurch.org>
 * Parts written by Andrew Kempe and others.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "services.h"
#include "encrypt.h"

/*************************************************************************/

/* Default (no encryption) high-level routines.  Used when no encryption
 * module is loaded.  See encrypt.h for function documentation. */

static int default_encrypt(const char *src, int len, char *dest, int size)
{
    if (size < len)
	return -1;
    memset(dest, 0, size);
    memcpy(dest, src, len);
    return 0;
}

static int default_encrypt_in_place(char *buf, int size)
{
    return 0;
}

static int default_encrypt_check_len(int passlen, int bufsize)
{
    if (bufsize <= 1)
	fatal("default_encrypt_check_len(): bufsize too small (%d)", bufsize);
    if (passlen < bufsize)
	return 0;
    else
	return bufsize-1;
}

static int default_decrypt(const char *src, char *dest, int size)
{
    if (strlen(src) >= size)
	return -1;
    strcpy(dest, src);
    return 1;
}

static int default_check_password(const char *plaintext, const char *password)
{
    if (strcmp(plaintext, password) == 0)
	return 1;
    else
	return 0;
}

/*************************************************************************/

/* Function pointers for high-level routines.  Encryption modules should
 * set these pointers to point to their own functions when loaded, and
 * should call reset_encrypt() (below) if unloaded.
 */

int (*encrypt)(const char *src, int len, char *dest, int size)
     = default_encrypt;
int (*encrypt_in_place)(char *buf, int size)
     = default_encrypt_in_place;
int (*encrypt_check_len)(int passlen, int bufsize)
     = default_encrypt_check_len;
int (*decrypt)(const char *src, char *dest, int size)
     = default_decrypt;
int (*check_password)(const char *plaintext, const char *password)
     = default_check_password;


void reset_encrypt(void)
{
    encrypt = default_encrypt;
    encrypt_in_place = default_encrypt_in_place;
    encrypt_check_len = default_encrypt_check_len;
    decrypt = default_decrypt;
    check_password = default_check_password;
}

/*************************************************************************/
