/* Database file handling routines.
 *
 * Services is copyright (c) 1996-1998 Andy Church.
 *     E-mail: <achurch@dragonfire.net>
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "services.h"
#include <fcntl.h>

/*************************************************************************/
/*************************************************************************/

/* Return the version number on the file.  Panic if there is no version
 * number or the number doesn't make sense (i.e. less than 1 or greater
 * than FILE_VERSION).
 */

int get_file_version(FILE *f, const char *filename)
{
    int version = fgetc(f)<<24 | fgetc(f)<<16 | fgetc(f)<<8 | fgetc(f);
    if (ferror(f))
	fatal_perror("Error reading version number on %s", filename);
    else if (ferror(f))
	fatal("Error reading version number on %s: End of file detected",
		filename);
    else if (version > FILE_VERSION || version < 1)
	fatal("Invalid version number (%d) on %s", version, filename);
    return version;
}

/*************************************************************************/

/* Write the current version number to the file.  Return 0 on error, 1 on
 * success.
 */

int write_file_version(FILE *f, const char *filename)
{
    if (
	fputc(FILE_VERSION>>24 & 0xFF, f) < 0 ||
	fputc(FILE_VERSION>>16 & 0xFF, f) < 0 ||
	fputc(FILE_VERSION>> 8 & 0xFF, f) < 0 ||
	fputc(FILE_VERSION     & 0xFF, f) < 0
    ) {
	log_perror("Error writing version number on %s", filename);
	return 0;
    }
    return 1;
}

/*************************************************************************/
/*************************************************************************/

static FILE *open_db_read(const char *service, const char *filename)
{
    FILE *f = fopen(filename, "rb");
    if (!f) {
	if (errno != ENOENT)
	    log_perror("Can't read %s database %s", service, filename);
	return NULL;
    }
    return f;
}

/*************************************************************************/

static FILE *open_db_write(const char *service, const char *filename)
{
    char namebuf[NAME_MAX+1];
    FILE *f;

    *namebuf = 0;
    snprintf(namebuf, sizeof(namebuf), "%s.save", filename);
    if (!*namebuf || strcmp(namebuf, filename) == 0) {
	errno = ENAMETOOLONG;
	log_perror("Can't back up %s database %s", service, filename);
	return NULL;
    }
    unlink(namebuf);
    if (rename(filename, namebuf) < 0 && errno != ENOENT) {
	static int walloped = 0;
	if (!walloped) {
	    walloped++;
	    wallops(NULL, "Can't back up %s database %s", service, filename);
	}
	log_perror("Can't back up %s database %s", service, filename);
#ifndef NO_BACKUP_OKAY
	return NULL;
#endif
    }
    f = fopen(filename, "wb");
    if (!f || !write_file_version(f, filename)) {
	static int walloped = 0;
	if (!walloped) {
	    walloped++;
	    wallops(NULL, "Can't write to %s database %s", service, filename);
	}
	log_perror("Can't write to %s database %s", service, filename);
	if (f) {
	    fclose(f);
	    unlink(filename);
	}
	if (rename(namebuf, filename) < 0
#ifdef NO_BACKUP_OKAY
		&& errno != ENOENT
#endif
	) {
	    /* Better quit; something might be seriously wrong */
	    fatal_perror("Cannot restore backup copy of %s", filename);
	}
	return NULL;
    }
    return f;
}

/*************************************************************************/

/* Open a database file for reading (*mode == 'r') or writing (*mode == 'w').
 * Return the stream pointer, or NULL on error.  When opening for write, it
 * is an error for rename() to return an error (when backing up the original
 * file) other than ENOENT, if NO_BACKUP_OKAY is not defined; it is an error
 * if the version number cannot be written to the file; and it is a fatal
 * error if opening the file for write fails and the backup was successfully
 * made but cannot be restored.
 */

FILE *open_db(const char *service, const char *filename, const char *mode)
{
    if (*mode == 'r') {
	return open_db_read(service, filename);
    } else if (*mode == 'w') {
	return open_db_write(service, filename);
    } else {
	errno = EINVAL;
	return NULL;
    }
}

/*************************************************************************/

/* Close a database file.  If the file was opened for write, remove the
 * backup we (may have) created earlier.
 */

void close_db(FILE *dbfile, const char *filename)
{
    int flags;

    flags = fcntl(fileno(dbfile), F_GETFL);
    if ((flags != -1) &&
		(((flags & O_ACCMODE) == O_WRONLY) ||
		 ((flags & O_ACCMODE) == O_RDWR)))
    {
	char namebuf[NAME_MAX+1];
	snprintf(namebuf, sizeof(namebuf), "%s.save", filename);
	if (*namebuf && strcmp(namebuf, filename) != 0)
	    remove(namebuf);
    }
    fclose(dbfile);
}

/*************************************************************************/
/*************************************************************************/

/* Read and write 2- and 4-byte quantities, pointers, and strings.  All
 * multibyte values are stored in big-endian order (most significant byte
 * first).  Apointer is stored as a byte, either 0 if NULL or 1 if not, and
 * read pointers are returned as either (void *)0 or (void *)1.  A string
 * is stored with a 2-byte unsigned length first; a length of 0 indicates
 * that the string pointer is NULL.  Written strings are truncated silently
 * at 65534 bytes, and are always null-terminated.
 *
 * All routines return -1 on error, 0 otherwise.
 */


int read_int16(uint16 *ret, FILE *f)
{
    int c1, c2;

    c1 = fgetc(f);
    c2 = fgetc(f);
    if (c1 == EOF || c2 == EOF)
	return -1;
    *ret = c1<<8 | c2;
    return 0;
}

int write_int16(uint16 val, FILE *f)
{
    if (fputc((val>>8) & 0xFF, f) == EOF || fputc(val & 0xFF, f) == EOF)
	return -1;
    return 0;
}


int read_int32(uint32 *ret, FILE *f)
{
    int c1, c2, c3, c4;

    c1 = fgetc(f);
    c2 = fgetc(f);
    c3 = fgetc(f);
    c4 = fgetc(f);
    if (c1 == EOF || c2 == EOF || c3 == EOF || c4 == EOF)
	return -1;
    *ret = c1<<24 | c2<<16 | c3<<8 | c4;
    return 0;
}

int write_int32(uint32 val, FILE *f)
{
    if (fputc((val>>24) & 0xFF, f) == EOF)
	return -1;
    if (fputc((val>>16) & 0xFF, f) == EOF)
	return -1;
    if (fputc((val>> 8) & 0xFF, f) == EOF)
	return -1;
    if (fputc((val    ) & 0xFF, f) == EOF)
	return -1;
    return 0;
}


int read_ptr(void **ret, FILE *f)
{
    int c;

    c = fgetc(f);
    if (c == EOF)
	return -1;
    *ret = (c ? (void *)1 : (void *)0);
    return 0;
}

int write_ptr(const void *ptr, FILE *f)
{
    if (fputc(ptr ? 1 : 0, f) == EOF)
	return -1;
    return 0;
}


int read_string(char **ret, FILE *f)
{
    char *s;
    uint16 len;

    if (read_int16(&len, f) < 0)
	return -1;
    if (len == 0) {
	*ret = NULL;
	return 0;
    }
    s = smalloc(len);
    if (len != fread(s, 1, len, f)) {
	free(s);
	return -1;
    }
    *ret = s;
    return 0;
}

int write_string(const char *s, FILE *f)
{
    uint32 len;

    if (!s)
	return write_int16(0, f);
    len = strlen(s);
    if (len > 65534)
	len = 65534;
    if (write_int16((uint16)(len+1), f) < 0)
	return -1;
    if (len > 0 && fwrite(s, 1, len, f) != len)
	return -1;
    if (fputc(0, f) == EOF)
	return -1;
    return 0;
}

/*************************************************************************/
/*************************************************************************/
