/* Initalization and related routines.
 *
 * Services is copyright (c) 1996-1998 Andy Church.
 *     E-mail: <achurch@dragonfire.net>
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "services.h"

/*************************************************************************/

/* Send a NICK command for the given pseudo-client.  If `user' is NULL,
 * send NICK commands for all the pseudo-clients. */

#if defined(IRC_DALNET)
# ifdef IRC_DAL4_4_15
#  define NICK(nick,name) \
    do { \
	send_cmd(NULL, "NICK %s 1 %ld %s %s %s 0 :%s", (nick), time(NULL), \
		services_user, services_host, server_name, (name)); \
    } while (0)
# else
#  define NICK(nick,name) \
    do { \
	send_cmd(NULL, "NICK %s 1 %ld %s %s %s :%s", (nick), time(NULL), \
		services_user, services_host, server_name, (name)); \
    } while (0)
# endif
#elif defined(IRC_UNDERNET)
# define NICK(nick,name) \
    do { \
	send_cmd(server_name, "NICK %s 1 %ld %s %s %s :%s", (nick), time(NULL),\
		services_user, services_host, server_name, (name)); \
    } while (0)
#elif defined(IRC_TS8)
# define NICK(nick,name) \
    do { \
	send_cmd(NULL, "NICK %s :1", (nick)); \
	send_cmd((nick), "USER %ld %s %s %s :%s", time(NULL), \
		services_user, services_host, server_name, (name)); \
    } while (0)
#else
# define NICK(nick,name) \
    do { \
	send_cmd(NULL, "NICK %s :1", (nick)); \
	send_cmd((nick), "USER %s %s %s :%s", \
		services_user, services_host, server_name, (name)); \
    } while (0)
#endif

void introduce_user(const char *user)
{
    /* Watch out for infinite loops... */
#define LTSIZE 20
    static int lasttimes[LTSIZE];
    if (lasttimes[0] >= time(NULL)-3)
	fatal("introduce_user() loop detected");
    memmove(lasttimes, lasttimes+1, sizeof(lasttimes)-sizeof(int));
    lasttimes[LTSIZE-1] = time(NULL);
#undef LTSIZE

    if (!user || stricmp(user, s_NickServ) == 0) {
	NICK(s_NickServ, NICKSERV_REALNAME);
#ifndef IRC_DALNET
	send_cmd(s_NickServ, "MODE %s +o", s_NickServ);
#endif
    }
    if (!user || stricmp(user, s_ChanServ) == 0) {
	NICK(s_ChanServ, CHANSERV_REALNAME);
#ifndef IRC_DALNET
	send_cmd(s_ChanServ, "MODE %s +o", s_ChanServ);
#endif
    }
    if (!user || stricmp(user, s_HelpServ) == 0) {
	NICK(s_HelpServ, HELPSERV_REALNAME);
    }
    if (!user || stricmp(user, IRCIIHELP_NAME) == 0) {
	NICK(IRCIIHELP_NAME, IRCIIHELP_REALNAME);
    }
    if (!user || stricmp(user, s_MemoServ) == 0) {
	NICK(s_MemoServ, MEMOSERV_REALNAME);
#ifndef IRC_DALNET
	send_cmd(s_MemoServ, "MODE %s +o", s_MemoServ);
#endif
    }
    if (!user || stricmp(user, s_OperServ) == 0) {
	NICK(s_OperServ, OPERSERV_REALNAME);
#ifndef IRC_DALNET
	send_cmd(s_OperServ, "MODE %s +oi", s_OperServ);
#else
	send_cmd(s_OperServ, "MODE %s +i", s_OperServ);
#endif
    }
#ifdef DEVNULL_NAME
    if (!user || stricmp(user, DEVNULL_NAME) == 0) {
	NICK(DEVNULL_NAME, DEVNULL_REALNAME);
	send_cmd(DEVNULL_NAME, "MODE %s +i", DEVNULL_NAME);
    }
#endif
    if (!user || stricmp(user, s_GlobalNoticer) == 0) {
	NICK(s_GlobalNoticer, GLOBAL_NOTICER_REALNAME);
	send_cmd(s_GlobalNoticer, "MODE %s +io", s_GlobalNoticer);
    }
}

#undef NICK

/*************************************************************************/

/* Set GID if necessary.  Return 0 if successful (or if RUNGROUP not
 * defined), else print an error message to logfile and return -1.
 */

static int set_group(void)
{
#if defined(RUNGROUP) && defined(HAVE_SETGRENT)
    struct group *gr;

    setgrent();
    while (gr = getgrent()) {
	if (strcmp(gr->gr_name, RUNGROUP) == 0)
	    break;
    }
    endgrent();
    if (gr) {
	setgid(gr->gr_gid);
	return 0;
    } else {
	log("Unknown group `%s'\n", RUNGROUP);
	return -1;
    }
#else
    return 0;
#endif
}

/*************************************************************************/

/* Parse command-line options.  Return 0 if all went well, -1 for an error
 * with an option, or 1 for -help.
 */

static int parse_options(int ac, char **av)
{
    int i;
    char *s, *t;

    for (i = 1; i < ac; i++) {
	s = av[i];
	if (*s == '-') {
	    s++;
	    if (strcmp(s, "remote") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-remote requires hostname[:port]\n");
		    return -1;
		}
		s = av[i];
		if (t = strchr(s, ':')) {
		    *t++ = 0;
		    if (atoi(t) > 0)
			remote_port = atoi(t);
		    else {
			fprintf(stderr, "-remote: port number must be a positive integer.  Using default.");
			return -1;
		    }
		}
		remote_server = s;
	    } else if (strcmp(s, "local") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-local requires hostname or [hostname]:[port]\n");
		    return -1;
		}
		s = av[i];
		if (t = strchr(s, ':')) {
		    *t++ = 0;
		    if (atoi(t) >= 0)
			local_port = atoi(t);
		    else {
			fprintf(stderr, "-local: port number must be a positive integer or 0.  Using default.");
			return -1;
		    }
		}
		local_host = s;
	    } else if (strcmp(s, "name") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-name requires a parameter\n");
		    return -1;
		}
		server_name = av[i];
	    } else if (strcmp(s, "desc") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-desc requires a parameter\n");
		    return -1;
		}
		server_desc = av[i];
	    } else if (strcmp(s, "user") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-user requires a parameter\n");
		    return -1;
		}
		services_user = av[i];
	    } else if (strcmp(s, "host") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-host requires a parameter\n");
		    return -1;
		}
		services_host = av[i];
	    } else if (strcmp(s, "dir") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-dir requires a parameter\n");
		    return -1;
		}
		services_dir = av[i];
	    } else if (strcmp(s, "log") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-log requires a parameter\n");
		    return -1;
		}
		log_filename = av[i];
	    } else if (strcmp(s, "update") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-update requires a parameter\n");
		    return -1;
		}
		s = av[i];
		if (atoi(s) <= 0) {
		    fprintf(stderr, "-update: number of seconds must be positive");
		    return -1;
		} else
		    update_timeout = atol(s);
	    } else if (strcmp(s, "expire") == 0) {
		if (++i >= ac) {
		    fprintf(stderr, "-expire requires a parameter\n");
		    return -1;
		}
		s = av[i];
		if (atoi(s) <= 0) {
		    fprintf(stderr, "-expire: number of seconds must be positive");
		    return -1;
		} else
		    expire_timeout = atol(s);
	    } else if (strcmp(s, "debug") == 0) {
		debug++;
	    } else if (strcmp(s, "readonly") == 0) {
		readonly = 1;
	    } else if (strcmp(s, "nofork") == 0) {
		nofork = 1;
	    } else {
		fprintf(stderr, "Unknown option -%s\n", s);
		return -1;
	    }
	} else {
	    fprintf(stderr, "Non-option arguments not allowed\n");
	    return -1;
	}
    }
    return 0;
}

/*************************************************************************/

/* Remove our PID file.  Done at exit. */

static void remove_pidfile(void)
{
    remove(PID_FILE);
}

/*************************************************************************/

/* Create our PID file and write the PID to it. */

static void write_pidfile(void)
{
    FILE *pidfile;

    pidfile = fopen(PID_FILE, "w");
    if (pidfile) {
	fprintf(pidfile, "%d\n", (int)getpid());
	fclose(pidfile);
	atexit(remove_pidfile);
    } else {
	log_perror("Warning: cannot write to PID file %s", PID_FILE);
    }
}

/*************************************************************************/

/* Overall initialization routine.  Returns 0 on success, -1 on failure. */

int init(int ac, char **av)
{
    int i;
    int openlog_failed = 0, openlog_errno = 0;
    int started_from_term = isatty(0) && isatty(1) && isatty(2);

    /* Imported from main.c */
    extern void sighandler(int signum);


    /* Set file creation mask and group ID. */
#if defined(DEFUMASK) && HAVE_UMASK
    umask(DEFUMASK);
#endif
    if (set_group() < 0)
	return -1;
    
    /* Parse options. */
    parse_options(ac, av);

    /* Chdir to Services data directory. */
    if (chdir(services_dir) < 0) {
	fprintf(stderr, "chdir(%s): %s\n", services_dir, strerror(errno));
	return -1;
    }

    /* Open logfile, and complain if we didn't. */
    if (open_log() < 0) {
	openlog_errno = errno;
	if (started_from_term) {
	    fprintf(stderr, "Warning: unable to open log file %s: %s\n",
			log_filename, strerror(errno));
	} else {
	    openlog_failed = 1;
	}
    }

    /* Detach ourselves if requested. */
    if (!nofork) {
	if ((i = fork()) < 0) {
	    perror("fork()");
	    return -1;
	} else if (i != 0) {
	    exit(0);
	}
	if (started_from_term) {
	    close(0);
	    close(1);
	    close(2);
	}
	if (setpgid(0, 0) < 0) {
	    perror("setpgid()");
	    return -1;
	}
    }

    /* Write our PID to the PID file. */
    write_pidfile();

    /* Announce ourselves to the logfile. */
#ifdef SKELETON
    log("Services %s (skeleton version, compiled for %s) starting up",
		version_number, version_protocol);
#else
    if (debug || readonly) {
	log("Services %s (compiled for %s) starting up (options:%s%s)",
		version_number, version_protocol,
		debug ? " debug" : "", readonly ? " readonly" : "");
    } else {
	log("Services %s (compiled for %s) starting up",
		version_number, version_protocol);
    }
#endif
    start_time = time(NULL);

    /* If in read-only mode, close the logfile again. */
    if (readonly)
	close_log();

    /* Set signal handlers.  Catch certain signals to let us do things or
     * panic as necessary, and ignore all others.
     */
#ifdef NSIG
    for (i = 1; i <= NSIG; i++)
#else
    for (i = 1; i <= 32; i++)
#endif
	signal(i, SIG_IGN);

    signal(SIGINT, sighandler);
    signal(SIGTERM, sighandler);
    signal(SIGQUIT, sighandler);
    signal(SIGSEGV, sighandler);
    signal(SIGBUS, sighandler);
    signal(SIGQUIT, sighandler);
    signal(SIGHUP, sighandler);
    signal(SIGILL, sighandler);
    signal(SIGTRAP, sighandler);
#ifdef SIGIOT
    signal(SIGIOT, sighandler);
#endif
    signal(SIGFPE, sighandler);

    signal(SIGUSR1, sighandler);  /* This is our "out-of-memory" panic switch */

    /* Initialize multi-language support */
    lang_init();
    if (debug)
	log("debug: Loaded languages");

    /* Load up databases */
#ifndef SKELETON
    load_ns_dbase();
    if (debug)
	log("debug: Loaded %s database (1/4)", s_NickServ);
    load_cs_dbase();
    if (debug)
	log("debug: Loaded %s database (2/4)", s_ChanServ);
#endif
    load_os_dbase();
    if (debug)
	log("debug: Loaded %s database (3/4)", s_OperServ);
    load_akill();
    if (debug)
	log("debug: Loaded AKILL database (4/4)");
    log("Databases loaded");

    /* Connect to the remote server */
    if (!*local_host)
	local_host = NULL;
    servsock = conn(remote_server, remote_port, local_host, local_port);
    if (servsock < 0)
	fatal_perror("Can't connect to server");
    send_cmd(NULL, "PASS :%s", PASSWORD);
#ifdef IRC_UNDERNET_NEW
    send_cmd(NULL, "SERVER %s 1 %lu %lu P09 :%s",
		server_name, start_time, start_time, server_desc);
#else
    send_cmd(NULL, "SERVER %s 1 :%s", server_name, server_desc);
#endif
    sgets2(inbuf, sizeof(inbuf), servsock);
    if (strnicmp(inbuf, "ERROR", 5) == 0) {
	/* Close server socket first to stop wallops, since the other
	 * server doesn't want to listen to us anyway */
	disconn(servsock);
	servsock = -1;
	fatal("Remote server returned: %s", inbuf);
    }

    /* Announce a logfile error if there was one */
    if (openlog_failed) {
	wallops(NULL, "Warning: couldn't open logfile: %s",
		strerror(openlog_errno));
    }

    /* Bring in our pseudo-clients */
    introduce_user(NULL);

    /* Success! */
    return 0;
}

/*************************************************************************/
