/*
 * Copyright (C) 1997-1998 Kai 'Oswald' Seidler <oswald@cs.tu-berlin.de>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <gdbm.h>

#include "service.h"

#include "noteserv.h"

struct Clients *List_OnlineUsers[256];
struct Clients *List_Subscriber = NULL;

extern int Service_Burst;
GDBM_FILE Database_FILE;

void Client_expireMessages(struct Clients *client, time_t now)
{
    struct Message *message, *last_message;
    int i = 1;

    message = client->messages;
    while (message)
    {
	if ((message->timeout && message->timeout < now) || message->timeout == 0)
	{
	    if (message->timeout == 0)
		Client_printMessage(i, client, message->pattern, message->message, -1, message->timestamp, 60);
	    else
		Client_printMessage(i, client, message->pattern, message->message, -1, 0, 61);
	    if (message == client->messages)
	    {
		client->messages = message->next;
		free(message);
		message = client->messages;	/* I know... */
	    } else
	    {
                /* last_message possibly used uninitiated here... */
		last_message->next = message->next;
		free(message);
		message = last_message->next;
	    }
	    client->status |= STATUS_NEEDTOSAVE;
	    client->msg_count--;
	    i++;
	    continue;
	}
	i++;
	last_message = message;
	message = message->next;
    }
    Client_writeDatabase(client);
}

void Client_expireExpressions(struct Clients *client, time_t now)
{
    struct Expression *expression, *last_expression;
    int i = 1;

    expression = client->expressions;
    while (expression)
    {
	if (expression->timeout && expression->timeout < now)
	{
	    Client_printExpression(i, client, expression->pattern, expression->comment, -1, 65);
	    if (expression == client->expressions)
	    {
		client->expressions = expression->next;
		free(expression);
		expression = client->expressions;	/* I know... */
	    } else
	    {
                /* uninitialized? */
		last_expression->next = expression->next;
		free(expression);
		expression = last_expression->next;
	    }
	    client->status |= STATUS_NEEDTOSAVE;
	    client->expr_count--;
	    i++;
	    continue;
	}
	i++;
	last_expression = expression;
	expression = expression->next;
    }
    Client_writeDatabase(client);
}

void Client_deleteMessage(char *nickname, char *delete)
{
    struct Clients *found;
    struct Message *message, *tmp, *last_message=NULL;
    int i = 1, d = 0, flag=0;
    char *pattern = NULL;

    if (!(found = List_find(&List_Subscriber, nickname, FIND_NICKNAME)))
	return;

    if (strlen(delete) == 1)
    {
	d = (int) toupper(*delete) - 64;
    }
    if (d < 1 || d > found->msg_count)
    {
	d = 0;
	pattern = delete;
    }
    message = found->messages;
    while (message)
    {
	if ((d && i == d) || (pattern && (!match(pattern, message->pattern)
				  || !match(pattern, message->message))))
	{
	    if (last_message)
		last_message->next = message->next;
	    else
		found->messages = message->next;
	    Client_printMessage(i, found, message->pattern, message->message, -1, 0, 62);
	    found->msg_count--;
	    found->status |= STATUS_NEEDTOSAVE;
	    tmp = message->next;
	    free(message);
	    message=tmp;
	    flag=1;
	}
	else
	{
	    last_message = message;
	    message = message->next;
	}
	i++;
    }
    if(flag)
	    Client_writeDatabase(found);
}

void Client_deleteExpression(char *nickname, char *delete)
{
    struct Clients *found;
    struct Expression *expression, *tmp, *last_expression=NULL;
    int i = 1, d = atoi(delete),flag=0;
    char *pattern = NULL;

    if (!(found = List_find(&List_Subscriber, nickname, FIND_NICKNAME)))
	return;

    if (d < 1 || d > found->expr_count)
    {
	pattern = delete;
	d = 0;
    }
    expression = found->expressions;
    while (expression)
    {
	if ((d && i == d) || (pattern && (!match(pattern, expression->pattern)
			       || !match(pattern, expression->comment))))
	{
	    if (last_expression)
		last_expression->next = expression->next;
	    else
		found->expressions = expression->next;
	    Client_printExpression(i, found, expression->pattern, "", -1, 66);
	    found->expr_count--;
	    found->status |= STATUS_NEEDTOSAVE;
	    tmp = expression->next;
	    free(expression);
	    expression = tmp;
	    flag=1;
	}
	else
	{
		last_expression = expression;
		expression = expression->next;
	}
	i++;
    }
    if(flag)
	    Client_writeDatabase(found);
}

char *Client_serialize(struct Clients *client, int *size)
{
    struct Expression *expression;
    struct Message *message;
    char *buffer, *ptr;
    int l;

    l = sizeof(struct Clients) + sizeof(struct Expression) * client->expr_count
    + sizeof(struct Message) * client->msg_count;

    if (!(buffer = malloc(l)))
    {
	Service_reportError("Client_serialize().malloc(%d)", l);
	exit(1);
    }
    memcpy(buffer, client, sizeof(struct Clients));
    ptr = buffer + sizeof(struct Clients);

    expression = client->expressions;
    while (expression)
    {
	memcpy(ptr, expression, sizeof(struct Expression));
	ptr += sizeof(struct Expression);
	expression = expression->next;
    }

    message = client->messages;
    while (message)
    {
	memcpy(ptr, message, sizeof(struct Message));
	ptr += sizeof(struct Message);
	message = message->next;
    }

    *size = l;
    return buffer;
}

void Database_addKline(struct Clients *to, char *key)
{
	struct Clients *client;

	client = Client_create(key, "", "", STATUS_NEEDTOSAVE|STATUS_KLINED);

	List_addClient(&List_Subscriber, client);
	Client_writeDatabase(client);
	Client_sendNotice(to, 36, key, "added");
}

void Database_deleteUser(char *keystring)
{
    datum key;
	
    key.dptr = keystring;
    key.dsize = strlen(keystring) + 1;

    gdbm_delete(Database_FILE, key);
}

void Database_delKline(struct Clients *to,char *key)
{
	struct Clients *client;
	
	if((client=Key_isKlined(key)))
	{
		List_removeClient(&List_Subscriber, client);
		Client_sendNotice(to, 36, key, "deleted");
	}
}

void Client_writeDatabase(struct Clients *client)
{
    datum data, key;
    char buffer[KEYLEN + 1];

    if (!(client->status && STATUS_NEEDTOSAVE))
	return;

    if (strlen(client->username))
    {
	sprintf(buffer, "!%s", client->username);
	key.dptr = buffer;
	key.dsize = strlen(buffer) + 1;
    } else
    {
	key.dptr = client->key;
	key.dsize = strlen(client->key) + 1;
    }

    if (client->expr_count == 0 && client->msg_count == 0 && !(client->status && STATUS_KLINED))
	gdbm_delete(Database_FILE, key);
    else
    {
	client->timestamp = time(NULL);
	data.dptr = Client_serialize(client, &data.dsize);

	gdbm_store(Database_FILE, key, data, GDBM_REPLACE);

	free(data.dptr);
    }

    client->status &= ~STATUS_NEEDTOSAVE;

}

void Client_free(struct Clients *client_to_delete)
{
    struct Expression *expression, *last_expression;
    struct Message *message, *last_message;

    if (client_to_delete == NULL)
	return;

    if ((expression = client_to_delete->expressions) != NULL)
    {
	while (expression)
	{
	    last_expression = expression;
	    expression = expression->next;
	    free(last_expression);
	}
    }
    if ((message = client_to_delete->messages) != NULL)
    {
	while (message)
	{
	    last_message = message;
	    message = message->next;
	    free(last_message);
	}
    }
    free(client_to_delete);
}

struct Clients *Client_readDatabaseClient(char *keystring)
{
    static datum data, key;
    char *ptr;
    struct Clients *new_client = NULL;
    /* Check on these last_ ones */
    struct Expression *new_expression, *last_expression;
    struct Message *new_message, *last_message;
    int i;

    if (*keystring != '!')
    {
	if ((new_client = List_find(&List_Subscriber, keystring, FIND_KEY | FIND_OFFLINE)) != NULL &&
	    !*new_client->username)
	{
	    System_Debug("%s back ONLINE\n", new_client->key);
	    Client_unsetStatus(new_client, STATUS_OFFLINE);
	    return NULL;
	}
    }
    if (!strcmp(keystring, "first"))
    {
	key = gdbm_firstkey(Database_FILE);
	if (!key.dptr)
	    return NULL;
    } else if (!strcmp(keystring, "next"))
    {
	key = gdbm_nextkey(Database_FILE, key);
	if (!key.dptr)
	    return NULL;
    } else if (!strcmp(keystring, "delete"))
    {
	gdbm_delete(Database_FILE, key);
	return NULL;
    } else
    {
	key.dptr = keystring;
	key.dsize = strlen(keystring) + 1;
    }

    data = gdbm_fetch(Database_FILE, key);

    if (!data.dptr)
	return NULL;


    if (!(new_client = malloc(sizeof(struct Clients))))
    {
	Service_reportError("Client_readDatabaseClient().malloc()");
	exit(2);
    }
    ptr = data.dptr;
    memcpy(new_client, ptr, sizeof(struct Clients));
    ptr += sizeof(struct Clients);

    new_client->expressions = NULL;
    new_client->status |= STATUS_NEEDTOSAVE;
    new_client->messages = NULL;

    for (i = 1; i <= new_client->expr_count; i++)
    {
	if (!(new_expression = malloc(sizeof(struct Expression))))
	{
	    Service_reportError("Client_readDatabaseClient().malloc()");
	    exit(3);
	}
	memcpy(new_expression, ptr, sizeof(struct Expression));
	new_expression->next = NULL;
	ptr += sizeof(struct Expression);

	if (i == 1)
	    new_client->expressions = new_expression;
	else
	    last_expression->next = new_expression;

	last_expression = new_expression;
    }

    for (i = 1; i <= new_client->msg_count; i++)
    {
	if (!(new_message = malloc(sizeof(struct Message))))
	{
	    Service_reportError("Client_readDatabaseClient().malloc()");
	    exit(4);
	}
	memcpy(new_message, ptr, sizeof(struct Message));
	new_message->next = NULL;
	ptr += sizeof(struct Message);

	if (i == 1)
	    new_client->messages = new_message;
	else
	    last_message->next = new_message;

	last_message = new_message;
    }

    return new_client;
}

void Client_changeNickname(struct Clients *found, char *tonick)
{
    char buffer[KEYLEN + 1];
    char *user;

    if (!found)
	return;

    user = strchr(found->key, '!');
    *user++ = '\0';
    sprintf(buffer, "%s!%s", tonick, user);
    strcpy(found->key, buffer);
}

void MClient_changeNickname(char *from_nick, char *to_nick)
{
    struct Clients *client, *last_client = NULL;

    client = List_OnlineUsers[(int) from_nick[0]];

    while (client)
    {
	if (Key_compareNickname(client->key, from_nick))
	{
	    if (last_client)
	    {
		last_client->next = client->next;
	    } else
	    {
		List_OnlineUsers[(int) from_nick[0]] = client->next;
	    }
	    client->next = List_OnlineUsers[(int) to_nick[0]];
	    List_OnlineUsers[(int) to_nick[0]] = client;
	    Client_changeNickname(client, to_nick);
	}
	last_client = client;
	client = client->next;
    }
}

void Client_setStatus(struct Clients *client, int status)
{
    client->status |= status;
    /* Client_writeDatabase(client); */
}

void Client_unsetStatus(struct Clients *client, int status)
{
    client->status &= ~status;
    /* Client_writeDatabase(client); */
}

void Client_setMode(struct Clients *client, char *mode)
{
    if (!client)
	return;

    System_Debug("%d %s ", client->status, mode);
    if (mode[0] == '+')
    {
	if (strchr(mode, 'i'))
	{
	    Subscriber_notifyaboutInvisible(client);
	    client->status |= STATUS_MODE_I;
	}
	if (strchr(mode, 'o'))
	    client->status |= STATUS_MODE_O;
    } else
    {
	if (strchr(mode, 'i'))
	{
	    client->status &= ~STATUS_MODE_I;
	    Subscriber_notifyaboutVisible(client);
	}
	if (strchr(mode, 'o'))
	    client->status &= ~STATUS_MODE_O;
    }
    System_Debug("%d\n", client->status);
}


struct Clients *Client_create(char *key, char *server, char *mode, int status)
{
    struct Clients *new_page;

    if (!(new_page = malloc(sizeof(struct Clients))))
    {
	Service_reportError("Client_create().malloc()");
	exit(5);
    }
    strcpy(new_page->key, key);
    strcpy(new_page->server, server);
    new_page->timestamp = time(NULL) + SIGNON_DELAY;
    new_page->username[0] = '\0';
    new_page->status = status;
    new_page->expressions = NULL;
    new_page->expr_count = 0;
    new_page->messages = NULL;
    new_page->msg_count = 0;
    Client_setMode(new_page, mode);

    return new_page;
}

struct Clients *Key_isKlined(char *key)
{
    return List_find(&List_Subscriber, key, FIND_KLINED);
}

void Client_addMessage(char *nickname, char *pattern, int seconds, char *text)
{
    struct Clients *found, *founda;
    struct Message *new_message, *message;
    int i, n;

    if (!(found = List_find(&List_OnlineUsers[(int) nickname[0]], nickname, FIND_NICKNAME)))
	    return;

    if (Key_isKlined(found->key))
    {
	Client_sendNotice(found, 86);
	return;
    }

    if ((n = MList_count(List_OnlineUsers, pattern, FIND_USEMATCH)) > 1)
    {
	Client_sendNotice(found, 0);
	return;
    }
    if (n == 1)
    {
	Client_sendNotice(found, 1);
	return;
    }

    if (!String_isPattern(pattern))
    {
	Client_sendNotice(found, 83, pattern);
    }
    if (!(founda = List_find(&List_Subscriber, nickname, FIND_NICKNAME)))
    {
	founda = List_addKey(&List_Subscriber, found->key, "", "", STATUS_NEEDTOSAVE);
	founda->status = (found->status & 0xf) | (founda->status & ~0xf);
    }
    found = founda;


    if (found->msg_count >= MAX_MSGS)
    {
	Client_sendNotice(found, 2, MAX_MSGS);
	return;
    }
    if (!(new_message = malloc(sizeof(struct Message))))
    {
	Service_reportError("Client_addMessage().malloc()");
	exit(6);
    }
    strcpy(new_message->pattern, pattern);
    strcpy(new_message->message, text + 1);
    if (seconds)
	new_message->timeout = time(NULL) + seconds;
    else
	new_message->timeout = 0;
    new_message->next = NULL;
    new_message->timestamp = time(NULL);

    message = found->messages;
    if (message)
    {
	i = 1;
	while (message)
	{
	    if (!strcasecmp(message->pattern, new_message->pattern) && !strcasecmp(message->message, new_message->message))
	    {
		if (seconds)
		    message->timeout = time(NULL) + seconds;
		else
		    message->timeout = 0;

		message->timestamp = time(NULL);

		Client_printMessage(i, found, pattern, new_message->message, seconds, 0, 52);
		found->status |= STATUS_NEEDTOSAVE;
		Client_writeDatabase(found);
		return;
	    }
	    i++;
	    if (message->next)
		message = message->next;
	    else
		break;
	}

	message->next = new_message;
    } else
	found->messages = new_message;
    found->msg_count++;

    Client_printMessage(found->msg_count, found, pattern, new_message->message, seconds, 0, 54);

    found->status |= STATUS_NEEDTOSAVE;
    Client_writeDatabase(found);

    return;
}

void Client_printExpression(int i, struct Clients *client, char *pattern, char *comment, time_t seconds, int num)
{
    char buffer[SMALLBUFFER];

    if (*comment)
    {
	sprintf(buffer, " [%s]", comment);
    } else
    {
	*buffer = '\0';
    }
    Client_sendNotice(client, num, i, pattern, buffer, Timeout_toString(client, seconds));
}

void Client_printMessage(int i, struct Clients *client, char *pattern, char *message, time_t seconds, time_t stamp, int num)
{
    char buffer[SMALLBUFFER];

    Client_sendNotice(client, num, i + 64, pattern, Timeout_toString(client, seconds));
    if (stamp)
    {
	strftime(buffer, SMALLBUFFER, "%c", gmtime(&stamp));
	Client_sendNotice(client, 6, buffer);
    }
    Client_sendNotice(client, 7, message);
}

void Client_addExpression(char *nickname, char *pattern, int seconds, char *comment)
{
    struct Clients *found, *founda;
    struct Expression *new_expression, *expression;
    int i;

    if (!(found = List_find(&List_OnlineUsers[(int) nickname[0]], nickname, FIND_NICKNAME)))
	    return;

    if (Key_isKlined(found->key))
    {
	Client_sendNotice(found, 86);
	return;
    }

    if (MList_count(List_OnlineUsers, pattern, FIND_USEMATCH) >= MAX_MATCH)
    {
	Client_sendNotice(found, 8);
	return;
    }
    if (!String_isPattern(pattern))
    {
	Client_sendNotice(found, 83, pattern);
    }
    if (!(founda = List_find(&List_Subscriber, nickname, FIND_NICKNAME)))
    {
	founda = List_addKey(&List_Subscriber, found->key, "", "", STATUS_NEEDTOSAVE);
	founda->status = (found->status & 0xf) | (founda->status & ~0xf);
    }
    found = founda;

    if (found->expr_count >= MAX_SPYS)
    {
	Nickname_sendNotice(nickname, 9, MAX_SPYS);
	return;
    }
    if (!(new_expression = malloc(sizeof(struct Expression))))
    {
	Service_reportError("Client_addExpression().malloc()");
	exit(7);
    }
    strcpy(new_expression->pattern, pattern);
    strcpy(new_expression->comment, comment);
    if (seconds)
	new_expression->timeout = time(NULL) + seconds;
    else
	new_expression->timeout = 0;
    new_expression->next = NULL;

    expression = found->expressions;
    if (expression)
    {
	i = 1;
	while (expression)
	{
	    if (!strcasecmp(expression->pattern, new_expression->pattern))
	    {
		strcpy(expression->comment, comment);
		if (seconds)
		    expression->timeout = time(NULL) + seconds;
		else
		    expression->timeout = 0;

		Client_printExpression(i, found, pattern, comment, seconds, 67);
		found->status |= STATUS_NEEDTOSAVE;
		Client_writeDatabase(found);
		return;
	    }
	    i++;
	    if (expression->next)
		expression = expression->next;
	    else
		break;
	}

	expression->next = new_expression;
    } else
	found->expressions = new_expression;

    found->expr_count++;

    Client_printExpression(found->expr_count, found, pattern, comment, seconds, 68);

    found->status |= STATUS_NEEDTOSAVE;
    Client_writeDatabase(found);

    return;
}


void Client_SignonOrchangeNick(int argc, char *argv[])
{
    struct Clients *client, *subscriber_client, *online_client;
    char *nickname = argv[0] + 1;

    if (*argv[0] == ':')	/* NICK CHANGE */
    {
	subscriber_client = List_find(&List_Subscriber, argv[0] + 1, FIND_NICKNAME);
	if (subscriber_client)
	{
	    /* Subscriber has a username, and we will follow */
	    if (*subscriber_client->username)
	    {
                /* Hybrid does not prepend NICK changes with : */
		Client_changeNickname(subscriber_client, argv[2]);
	    } else
	    {
		List_removeClient(&List_Subscriber, subscriber_client);
		subscriber_client = NULL;	/* segv fault 281197.2310 */
	    }
	}
	online_client = List_find(&List_OnlineUsers[(int) nickname[0]], nickname, FIND_NICKNAME);

	if (!online_client)
	{
		/* trying to stop the above segv faults */
		Service_reportError("ILLEGAL: %s NICK %s",argv[0],argv[2]);
		return;
	}

	Subscriber_notifyaboutNickchange(online_client, argv[2]); /* segv fault core.220198.23:10:07 */

	MClient_changeNickname(nickname, argv[2]);
	Subscriber_notifyaboutNicksignon(online_client);

	if (online_client->timestamp == 0)	/* segv fault core.091297.22:30:04 */
	    User_deliverMessages(online_client);

	if ((client = Client_readDatabaseClient(online_client->key)) != NULL)
	{
	    List_removeClient(&List_Subscriber, subscriber_client);
	    List_addClient(&List_Subscriber, client);
	    Client_unsetStatus(client, STATUS_OFFLINE);
	    online_client->status = (client->status & 0xf) | (online_client->status & ~0xf);
	}
	return;
    } else
	/* SIGNON */
    {
	nickname = argv[1];
	online_client = List_addKey(&List_OnlineUsers[(int) nickname[0]],
			    Key_buildfromNUH(nickname, argv[5], argv[6]),
				    argv[7], argv[4], STATUS_DONTSAVE);
                            /* nick, user, host, server, mode */


	if ((client = Client_readDatabaseClient(Key_buildfromNUH(argv[1], argv[5], argv[6]))) != NULL)
	{
	    List_addClient(&List_Subscriber, client);
	    Client_unsetStatus(client, STATUS_OFFLINE);
	    online_client->status = (client->status & 0xf) | (online_client->status & ~0xf);

            Client_setStatus(client, STATUS_NEEDTOSAVE);
            Client_writeDatabase(client);
	}
    }
}

int Client_IsOnline(struct Clients *client)
{
    if (!client)
	return 0;
    if (client->status & STATUS_OFFLINE)
	return 0;
    return 1;
}

int Client_IsAUser(struct Clients *client)
{
    if (!client)
	return 0;
    if (*client->username)
	return 1;
    return 0;
}

void Client_changeMode(int argc, char *argv[])
{
    char *nickname = argv[0] + 1;

    if (Service_Burst)
    {
	Service_Burst = 0;
	Service_reportError("Service is up!");
    }
    Client_setMode(List_find(&List_OnlineUsers[(int) nickname[0]], nickname, FIND_NICKNAME), argv[3] + 1);
}

void Client_Signoff(int argc, char *argv[])
{
    char *nickname = argv[0] + 1;
    struct Clients *client;

    if (!(client = List_find(&List_OnlineUsers[(int) nickname[0]], nickname, FIND_NICKNAME)))
	return;

    List_removeClient(&List_Subscriber, List_find(&List_Subscriber, nickname, FIND_NICKNAME));
    List_removeClientbyNickandnotifyaboutSignoff(&List_OnlineUsers[(int) nickname[0]], nickname);

}

void Client_Kill(int argc, char *argv[])
{ 
    char *nickname = argv[2];
    struct Clients *client;
 
    if (!(client = List_find(&List_OnlineUsers[(int) nickname[0]], nickname, FIND_NICKNAME)))
        return;   

    List_removeClient(&List_Subscriber, List_find(&List_Subscriber, nickname, FIND_NICKNAME));
    List_removeClientbyNickandnotifyaboutSignoff(&List_OnlineUsers[(int) nickname[0]], nickname);

}

