/*
 * Copyright (C) 1997-1998 Kai 'Oswald' Seidler <oswald@cs.tu-berlin.de>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>

#ifdef SYSTIME
#include <sys/time.h>
#endif

#include "service.h"

#include "noteserv.h"

extern int Service_Burst;

struct Clients *MList_find(struct Clients *list[], char *pattern, int type)
{
    static struct Clients *client;
    static int i;

    if (!Find_Init(type))
    {
	i = FIRST_CHAR;
	client = list[i];
    } else
    {
	if (client)
	    client = client->next;
    }

    while (1)
    {
	while (client == NULL)
	{
	    i++;
	    if (i > LAST_CHAR)
		return NULL;
	    client = list[i];
	}

	do
	{
	    if (Status_IsOffline(client->status) && !Find_Offline(type))
		continue;

	    if (Find_Everything(type))
		return client;

	    if (Find_RespectInvisible(type) && Status_IsInvisible(client->status))
		continue;

	    if (Find_Nickname(type) && !Status_IsOffline(client->status)
		&& Key_compareNickname(client->key, pattern))
		return client;

	    if (Find_UseMatch(type) && !match(pattern, client->key))
		return client;

	    if (Find_Key(type) && !strcmp(client->key, pattern))
		return client;

	    if (Find_Username(type) && !strcmp(pattern, client->username))
		return client;
	}
	while ((client = client->next) != NULL);
    }
}

struct Clients *List_find(struct Clients **list, char *pattern, int type)
{
    static struct Clients *client;

    if (!Find_Init(type))
    {
	client = *list;
    } else
    {
	if (client)
	    client = client->next;
    }

    if (client == NULL)
	return NULL;

    do
    {
	if (Status_IsKlined(client->status))
	{
	    if (Find_Klined(type) && (Find_Everything(type) || !match(client->key,pattern)))
		    return client;
	    continue;
	}
	    
	if (Status_IsOffline(client->status) && !Find_Offline(type))
	    continue;

	if (Find_Everything(type))
	    return client;

	if (Find_RespectInvisible(type) && Status_IsInvisible(client->status))
	    continue;

	if (Find_Nickname(type) && !Status_IsOffline(client->status)
	    && Key_compareNickname(client->key, pattern))
	    return client;

	if (Find_UseMatch(type) && !match(pattern, client->key))
	    return client;

	if (Find_Key(type) && !strcmp(client->key, pattern))
	    return client;

	if (Find_Username(type) && !strcmp(pattern, client->username))
	    return client;
    }
    while ((client = client->next) != NULL);
    return NULL;
}

void List_addClient(struct Clients **list, struct Clients *client)
{

    if (!client)
	return;
    client->next = *list;
    *list = client;

    User_sendOnlineSpys(client);
}

void List_destroyClient(struct Clients **list, struct Clients *client_to_delete)
{
    if (!client_to_delete)
	return;
    System_Debug("List_destroyClient(%s)\n", client_to_delete->key);
    Client_unsetStatus(client_to_delete, STATUS_NEEDTOSAVE);
    client_to_delete->msg_count = 0;
    List_removeClient(list, client_to_delete);
}

void List_removeClient(struct Clients **list, register struct Clients *client_to_delete)
{
    register struct Clients *client, *last_page;

    if (!client_to_delete)
	return;

    System_Debug("List_removeClient(%s)\n", client_to_delete->key);

    if (client_to_delete->msg_count)
    {
	System_Debug("%s OFFLINE\n", client_to_delete->key);
	/* client_to_delete->key[0]='\0'; */
	Client_setStatus(client_to_delete, STATUS_OFFLINE);
	Client_writeDatabase(client_to_delete);
	return;
    }
    if (Status_SaveNeeded(client_to_delete->status))
	Client_writeDatabase(client_to_delete);

    client = *list;

    do
    {
	if (client == client_to_delete)
	{
	    if (client == *list)
		*list = (*list)->next;
	    else
		last_page->next = client->next;
	    Client_free(client);
	    return;
	}
	last_page = client;
    }
    while ((client = client->next) != NULL);
    return;
}

void List_removeClientbyNickandnotifyaboutSignoff(struct Clients **list, char *nickname)
{
    register struct Clients *client, *last_page;

    client = *list;

    do
    {
	if (Key_compareNickname(client->key, nickname))
	{
	    if (client == *list)
		*list = (*list)->next;
	    else
		last_page->next = client->next;
	    Subscriber_notifyaboutSignoff(client);
	    Client_free(client);
	    return;
	}
	last_page = client;
    }
    while ((client = client->next) != NULL);
    return;
}

struct Clients *List_addKey(struct Clients **list, char *key, char *server, char *mode, int status)
{
    struct Clients *new_page;

    System_Debug("%s added...\n", key);

    new_page = Client_create(key, server, mode, status);
    new_page->next = *list;
    *list = new_page;

    return new_page;
}

int MList_count(struct Clients *list[], char *pattern, int defaulttype)
{
    struct Clients *found;
    int type = defaulttype;
    int i = 0;


    while ((found = MList_find(list, pattern, type)) != NULL)
    {
	type |= FIND_CONTINUE;
	i++;
    }

    return i;
}

int List_count(struct Clients **list, char *pattern, int defaulttype)
{
    struct Clients *found;
    int type = defaulttype;
    int i = 0;

    while ((found = List_find(list, pattern, type)) != NULL)
    {
	type |= FIND_CONTINUE;
	i++;
    }

    return i;
}
