/*
 * Copyright (C) 1997-1998 Kai 'Oswald' Seidler <oswald@cs.tu-berlin.de>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include <gdbm.h>

#include "service.h"

#include "noteserv.h"

#ifdef SYSTIME
#include <sys/time.h>
#endif

extern GDBM_FILE Database_FILE;
extern struct Clients *List_Subscriber;

char *messages[MAX_LANGUAGES][100];
char languages[MAX_LANGUAGES][10];

double Time_stop ()
{
  struct timeval tv;
  static struct timeval last_tv;
  double time_diff = 0;

  gettimeofday (&tv, NULL);
  if (timerisset (&last_tv))
    {
      time_diff = (tv.tv_sec + tv.tv_usec / 1000000.0) - (last_tv.tv_sec + last_tv.tv_usec / 1000000.0);
    }
  last_tv = tv;

  return time_diff;
}

int String_isPattern (char *string)
{
  if (strchr (string, '!') && strchr (string, '@'))
    return 1;
  else
    return 0;
}

char *String_toLower (char *string)
{
  char *ptr;

  for (ptr = string; *ptr != '\0'; ptr++)
    *ptr = tolower (*ptr);

  return string;
}

char *Timeout_toString (struct Clients *client, int i)
{
  static char buffer[SMALLBUFFER];
  int k, days, hours, minutes, seconds;

  *buffer = '\0';

  if (i == 0)
    return Client_getMessage (client, 79);

  if (i == -1)
    return "";

  days = i / SECSADAY;
  k = days * SECSADAY;
  hours = (i - k) / SECSANHOUR;
  k += hours * SECSANHOUR;
  minutes = (i - k) / 60;
  k += minutes * 60;
  seconds = i - k;

  if (days > 1)
    {
      sprintf (buffer, " (%d %s)", days, Client_getMessage (client, 70));
    }
  else if (days > 0)
    {
      sprintf (buffer, " (%d %s + %d %s)", days, days == 1 ? Client_getMessage (client, 69) : Client_getMessage (client, 70),
	       hours, hours == 1 ? Client_getMessage (client, 71) : Client_getMessage (client, 72));
    }
  else if (hours >= 1)
    {
      sprintf (buffer, " (%d %s)", hours, hours == 1 ? Client_getMessage (client, 71) : Client_getMessage (client, 72));
    }
  else if (hours > 0)
    {
      sprintf (buffer, " (%d %s + %d %s)", hours, hours == 1 ? Client_getMessage (client, 71) : Client_getMessage (client, 72),
	       minutes, minutes == 1 ? Client_getMessage (client, 73) : Client_getMessage (client, 74));
    }
  else if (minutes >= 1)
    {
      sprintf (buffer, " (%d %s)", minutes, minutes == 1 ? Client_getMessage (client, 73) : Client_getMessage (client, 74));
    }
  else if (minutes > 0)
    {
      sprintf (buffer, " (%d %s + %d %s)", minutes, minutes == 1 ? Client_getMessage (client, 73) : Client_getMessage (client, 74),
	       seconds, seconds == 1 ? Client_getMessage (client, 75) : Client_getMessage (client, 76));
    }
  else
    {
      sprintf (buffer, " (%d %s)", seconds, seconds == 1 ? Client_getMessage (client, 75) : Client_getMessage (client, 76));
    }

  return buffer;
}

void Database_Sync ()
{
  gdbm_sync (Database_FILE);
}

void Database_readofflineSubscribers (int expireonly)
{
  struct Clients *client;
  static char buffer[SMALLBUFFER];
  time_t stamp, now;

  /* gdbm_reorganize (Database_FILE); */
  now = time (NULL);

  if ((client = Client_readDatabaseClient ("first")) != NULL)
    {
      do
	{
	  if (Status_IsKlined (client->status))
	    {
		if(expireonly==0)
		{
		      List_addClient (&List_Subscriber, client);
		}
	    }
	  else
	    {
	      if ((now - client->timestamp) > EXPIRE_USER)
		{
		  stamp = client->timestamp;
		  strftime (buffer, SMALLBUFFER, "%c", gmtime (&stamp));
		  printf ("deleting %s (%s) last seen: %s (%ld seconds ago)\n", client->key, client->username, buffer,
			  now - client->timestamp);
		  /* Client_deleteDatabaseClient(client); */
		  if (strlen (client->username))
		    {
		      sprintf (buffer, "!%s", client->username);
		      Database_deleteUser (buffer);
		    }
		  else
		    {
		      Database_deleteUser (client->key);
		    }
		  client = Client_readDatabaseClient ("first");
		  continue;
		}
	      if (client->msg_count && expireonly==0)
		{
		  List_addClient (&List_Subscriber, client);
		  Client_unsetStatus (client, STATUS_NEEDTOSAVE);
		  Client_setStatus (client, STATUS_OFFLINE);
		  /* if (strlen(client->username))
		     {
		     client->key[0] = '\0';
		     } */
		}
	    }
	}
      while ((client = Client_readDatabaseClient ("next")) != NULL);
    }
}

void Messages_init ()
{
  char line[MAXBUFFER], *ptr, *ptr2, *new;
  int i = 0, j;
  FILE *fp;

  for (i = 0; i < MAX_LANGUAGES; i++)
    {
      *languages[i] = '\0';
    }

  i = 0;

  sprintf (line, "%s/messages.dat", DIRECTORY);

  if ((fp = fopen (line, "r")) == NULL)
    {
      fprintf (stderr, "Couldn't read messages file.\n");
      exit (8);
    }
  while (fgets (line, MAXBUFFER, fp))
    {
      if (line[0] == '#')
	continue;

      if ((ptr = strchr (line, '\n')))
	{
	  *ptr = '\0';
	}
      if ((ptr = strchr (line, ':')) == 0)
	{
	  fprintf (stderr, "Syntax error in messages file.\n");
	  exit (9);
	}
      *ptr = '\0';

      Messages_addLanguage (line);
      if ((i = Messages_getLanguage (line)) < 0)
	{
	  fprintf (stderr, "Syntax error in messages file (2).\n");
	  exit (10);
	}
      ptr2 = ++ptr;

      if ((ptr = strchr (ptr2, ':')) == 0)
	{
	  fprintf (stderr, "Syntax error in messages file (3).\n");
	  exit (11);
	}
      *ptr = '\0';

      j = atoi (ptr2);

      ptr++;

      if (!(new = malloc (strlen (ptr) + 1)))
	{
	  fprintf (stderr, "Out of memory.\n");
	  exit (12);
	}
      strcpy (new, ptr);
      messages[i][j] = new;
    }
  fclose (fp);
}

int Messages_addLanguage (char *lang)
{
  int i;

  if (Messages_getLanguage (lang) != -1)
    return -2;

  for (i = 0; i < MAX_LANGUAGES; i++)
    {
      if (*languages[i] == '\0')
	{
	  strcpy (languages[i], lang);
	  /* printf("Language %d '%s' added.\n", i, languages[i]); */
	  return i;
	}
    }
  return -1;
}

void Messages_getLanguages (char *buffer)
{
  int i;

  /* Terminate the string so we don't get repeated entries of the languages
  ** or other "crud" in the NOTICE -Hwy
  */
  buffer[0] = '\0';

  for (i = 0; i < MAX_LANGUAGES && *languages[i]; i++)
    sprintf (buffer, "%s %s", buffer, languages[i]);
}

int Messages_getLanguage (char *lang)
{
  int i;

  for (i = 0; i < MAX_LANGUAGES && *languages[i]; i++)
    {
      if (!strcmp (languages[i], lang))
	return i;
    }
  return -1;
}

char *Client_getMessage (struct Clients *client, int num)
{
  return messages[client->status & 0xf][num];
}
