/* NoteServ gdbm database examiner/debugger
** A generic gdbm file dumper
**
** W. Campbell (20 fv 2002)
** wcampbel@botbay.net
**
** $Id: ns-dumpdb.c,v 1.2 2002/05/25 01:16:15 wcampbel Exp $
**
** This code can be used in any way you please as long as
** my name remains with it.
*/

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include "gdbm.h"

#define TRUE 1
#define FALSE 0

#define TYPE_UNDEF	0
#define TYPE_LAST	1
#define TYPE_USER	2

/* These are NoteServ's structs and additional definitions */
#define HOSTLEN         63   
#define NICKLEN          9
#define USERLEN         10
#define KEYLEN          NICKLEN + USERLEN + HOSTLEN + 3

struct Last
{ 
    char key[KEYLEN + 1];
    time_t last;
};

void usage(char *);
void print_content(FILE *, char *, void *, unsigned int);

int main(int argc, char *argv[])
{
  char c;
  char *file_name = NULL;
  char *output_file = NULL;
  GDBM_FILE dbf;
  datum key;
  datum content;
  datum nextkey;
  extern char *optarg;
  FILE *outf = stdout;
  int outisfile = FALSE;
  unsigned int type = TYPE_UNDEF;

  while( (c = getopt(argc, argv, "f:o:ulh?")) != -1)
  {
    switch(c)
    {
      case 'h':
      case '?':
        usage(argv[0]);
        exit(0);
        /* NOT REACHED */
        break;
      case 'f':
        file_name = optarg;
        break;
      case 'o':
        output_file = optarg;
        break;
      case 'u':
        type = TYPE_USER;
        break;
      case 'l':
        type = TYPE_LAST;
        break;
      default:
        usage(argv[0]);
        exit(0);
        /* NOT REACHED */
        break;
    }
  }

  if (file_name == NULL)
  {
    fprintf(stderr, "No filename specified.\n");
    usage(argv[0]);
    exit(1);
  }

  if (output_file != NULL)
  {
    outf = fopen(output_file, O_RDONLY);
    outisfile = TRUE;
    if (outf == NULL)
    {
      fprintf(stderr, "Error opening output file: %s (%s)\n", output_file,
              strerror(errno));
      fprintf(stderr, "Terminating\n");
      exit(2);
    }
  }

  /* Open the gdbm file with the default blocksize */
  dbf = gdbm_open(file_name, 0, GDBM_READER, 00600, NULL);

  if (dbf == NULL)
  {
    fprintf(stderr, "GDBM open failed: %s\n", gdbm_strerror(gdbm_errno));
    exit(3);
  }

  /* Process the gdbm hash, extracting each key and value pair */
  key = gdbm_firstkey(dbf);
  while (key.dptr != NULL)
  {
    nextkey = gdbm_nextkey(dbf, key);
    content = gdbm_fetch(dbf, key);
    print_content(outf, key.dptr, content.dptr, type);
    free(content.dptr);
    free(key.dptr);
    key = nextkey;
  }

  if (outisfile == TRUE)
  {
    fclose(outf);
  }

  gdbm_close(dbf);

  return 0;
}

void usage(char *arg0)
{
  fprintf(stderr, "%s [-l|-u] -f file_name [-o output_file]\n", arg0);
}

void print_content(FILE *outf, char *key, void *content, unsigned int type)
{
  struct Last *lptr;

  switch(type)
  {
    case TYPE_UNDEF:
      /* Process it as if it were a simple char * pointer */
      fprintf(outf, "%s => %s\n", key, content);
      break;
    case TYPE_LAST:
      /* lastlog entry for NoteServ */
      lptr = (struct Last *) content;
      fprintf(outf, "%s => (%ld) %s\n", key, lptr->last, lptr->key);
      break;
  }
}
