/*
 * Copyright (C) 1997-1998 Kai 'Oswald' Seidler <oswald@cs.tu-berlin.de>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <stdarg.h>
#include <signal.h>

#include <gdbm.h>

/* #include <ircd/service_def.h> */

#include "service.h"

#include "noteserv.h"

/* Local defines for Service_sendHelpFlag() */
#define HELP_ADMIN 1
#define HELP_INFO  2

extern struct Clients *List_OnlineUsers[];
extern struct Clients *List_Subscriber;
extern GDBM_FILE Database_FILE;

int Service_Burst = 1;
int Server_Socket;
int Service_dies = 0;

struct CommandEntry
{
    char *IRC_Command;
    char *Service_Command;
    void (*Function) ();
};

struct CommandEntry CommandList[] =
{

    {"NICK", NULL, Client_SignonOrchangeNick},	/* IRC PROTOCOL */
    {"QUIT", NULL, Client_Signoff},
    {"KILL", NULL, Client_Kill},
    {"MODE", NULL, Client_changeMode},

    {"PING", NULL, Service_Ping},
/*    {"383", NULL, Service_setServ}, */
    {"VERSION", NULL, Service_sendSVersion},

    {NULL, "help", Service_Help},	/* GENERIC SERVICE COMMANDS */
    {NULL, "admin", Service_Admin},
    {NULL, "info", Service_Info},
    {NULL, "version", Service_Version},
    {NULL, "die", Service_Die},

    {NULL, "status", User_Status},	/* NOTESERV COMMANDS */
    {NULL, "stat", User_Stat},
    {NULL, "spy", User_Spy},
    {NULL, "notify", User_Spy},
    {NULL, "send", User_Send},
    {NULL, "login", User_Login},
    {NULL, "logout", User_Login},
    {NULL, "passwd", User_Passwd},
    {NULL, "find", User_Find},
    {NULL, "ls", User_List},
    {NULL, "rm", User_Remove},
    {NULL, "cont", User_Continue},
    {NULL, "trash", User_Trash},
    {NULL, "lang", User_Language},
    
    {NULL, "kline", User_Kline},
    {NULL, "unkline", User_Unkline},
    {NULL, "deluser", User_DeleteUser},

/*    {NULL, "seeyou", User_seeYou}, */

    {NULL, NULL, NULL}
};

void Server_sendto(char *format,...)
{
    char buffer[MAXBUFFER];
    va_list va;

    va_start(va, format);
    vsprintf(buffer, format, va);
    System_Debug(">> %s", buffer);
    write(Server_Socket, buffer, strlen(buffer));
    va_end(va);
}

char *Nickname_getMessage(char *nickname, int num)
{
    return Client_getMessage(NULL, num);
}

void Nickname_sendNotice(char *nickname, int num,...)
{
    va_list va;

    va_start(va, num);
    Client_sendNoticeVA(List_find(&List_OnlineUsers[(int) nickname[0]], nickname, FIND_NICKNAME), num, va);
    va_end(va);
}

void Client_sendNotice(struct Clients *client, int num,...)
{
    va_list va;

    va_start(va, num);
    Client_sendNoticeVA(client, num, va);
    va_end(va);
}

void Client_sendNoticeVA(struct Clients *client, int num, va_list va)
{
    char buffer[MAXBUFFER];
    char format[MAXBUFFER];

    sprintf(format, ":%s NOTICE %s :%s\n", NICK, Key_getNickname2(client->key), Client_getMessage(client, num));
    vsprintf(buffer, format, va);
    System_Debug(">> %s", buffer);
    write(Server_Socket, buffer, strlen(buffer));
}

/*
 * Reports error
 *
 */

void Service_reportError(char *format,...)
{
#ifdef ERRORADDR
    char buffer[MAXBUFFER];
    va_list va;

    va_start(va, format);
    vsprintf(buffer, format, va);
    System_Debug(">> %s", buffer);
    Server_sendto(":%s NOTICE %s :DEBUG %s\n", NICK, ERRORADDR, buffer);
    va_end(va);
#endif
}

int Server_receiveFrom(int fd, char *buffer)
{
    int n = read(fd, buffer, MAXBUFFER-1);
    buffer[n] = '\0';
    return n;
}


void Service_sendHelp(char *nickname, char *topic)
{
    FILE *fp;
    char buffer[MAXBUFFER];
    char *ptr;
    struct Clients *client;

    if (topic == NULL)
	topic = ".index";
    else
    {
	String_toLower(topic);
    }

    if((ptr = strchr(topic, '/')))
    {
	*ptr='\0';
    }

    sprintf(buffer, "%s/help/%s", DIRECTORY, topic);

    client = List_find(&List_OnlineUsers[(int) nickname[0]], nickname, FIND_NICKNAME);

    if ((fp = fopen(buffer, "r")) == NULL)
    {
	Client_sendNotice(client, 10, topic);
	return;
    }
    while (fgets(buffer, MAXBUFFER, fp))
    {
	Client_sendNotice(client, 35, buffer);
    }
    fclose(fp);
}

void Service_sendHelpFlag(char *nickname, int type)
{
    FILE *fp;
    char buffer[MAXBUFFER];
    char topic[13];
    struct Clients *client;

    if (type == HELP_ADMIN)
    {
      sprintf(buffer, "%s/help/.admin", DIRECTORY);
      strcpy(topic, ".admin");
    }
    else if (type == HELP_INFO)
    {
      sprintf(buffer, "%s/help/.information", DIRECTORY);
      strcpy(topic, ".information");
    }

    client = List_find(&List_OnlineUsers[(int) nickname[0]], nickname, FIND_NICKNAME);

    if ((fp = fopen(buffer, "r")) == NULL)
    {
        Client_sendNotice(client, 13, topic);
        return;
    }
    while (fgets(buffer, MAXBUFFER, fp))
    {
        Client_sendNotice(client, 35, buffer);
    }
    fclose(fp);
}

void Service_sendVersion(char *nickname)
{
    Server_sendto(":%s NOTICE %s :This is NoteServ version %s\n", NICK,
                  nickname, VERSION);
}

void Service_sendSVersion(int argc, char *argv[])
{
    Server_sendto(":%s 351 %s %s %s :TSora\n", SERVERNAME, argv[0] + 1, VERSION,
                  SERVERNAME);
}

void Service_Die(int argc, char *argv[])
{
    int i;
    char buffer[MAXBUFFER];

    if (strcmp(argv[4], PASSWORD))
    {
	Nickname_sendNotice(argv[0] + 1, 12);
	return;
    }
    *buffer = '\0';
    Database_Sync();
    Service_dies = 1;
    gdbm_close(Database_FILE);
    Nickname_sendNotice(argv[0] + 1, 13);

    for (i = 5; i <= argc; i++)
	sprintf(buffer, "%s %s", buffer, argv[i]);

    if (*buffer)
	Server_sendto("QUIT :%s\n", buffer + 1);
    else
	Server_sendto("QUIT\n");
}

void Service_Version(int argc, char *argv[])
{
    Service_sendVersion(argv[0] + 1);
}

void Service_Admin(int argc, char *argv[])
{
    Service_sendHelpFlag(argv[0] + 1, HELP_ADMIN);
}

void Service_Info(int argc, char *argv[])
{
    Service_sendHelpFlag(argv[0] + 1, HELP_INFO);
}

void Service_Help(int argc, char *argv[])
{
    if (*argv[4])
    {
	Service_sendHelp(argv[0] + 1, argv[4]);
    } else
    {
	Service_sendVersion(argv[0] + 1);
	Service_sendHelp(argv[0] + 1, NULL);
    }
}

void Service_interpretQuery(char *line)
{
    static char *argv[MAXARGS];
    char *ptr, *command, *service_command = NULL;
    int argc = 0, i;

    if (Service_dies)
	return;

    System_Debug(">> %s\n", line);

    while ((ptr = strchr(line, ' ')) && argc < MAXARGS)
    {
	argv[argc++] = line;
	*ptr = '\0';
	line = ptr + 1;
    }
    argv[argc] = line;

    for (i = argc + 1; i < MAXARGS; i++)
	argv[i] = "";

    if (*argv[0] == ':')
	command = argv[1];
    else
	command = argv[0];

    if (!strcmp(command, "PRIVMSG"))
    {
	service_command = argv[3] + 1;
    }
    /* 
     *  :fu-berlin.de 383 NoteServ@fu-berlin.de :You are service NoteServ@fu-berlin.de 
     *  :grebbie QUIT :Leaving
     *  :c-man NICK :chemieman
     *  NICK Talisos 1 Talisos max2-08.hrz.uni-giessen.de fu-berlin.de + :DGH
     *  :Oswald SQUERY NoteServ@fu-berlin.de :huhu 
     *  :Bas MODE Bas :+iw
     *  PING :fu-berlin.de 
     *
     */


    for (i = 0; CommandList[i].Function; i++)
    {
	if (CommandList[i].IRC_Command &&
	    !strcmp(command, CommandList[i].IRC_Command))
	{
	    CommandList[i].Function(argc, argv);
	    return;
	}
	if (service_command && CommandList[i].Service_Command &&
	    !strcasecmp(service_command, CommandList[i].Service_Command))
	{
	    CommandList[i].Function(argc, argv);
	    return;
	}
    }
    if (service_command)
	Nickname_sendNotice(argv[0] + 1, 14);
}

void Service_processPacket(char *buffer)
{
    char *ptr;
    char buffer2[MAXBUFFER];
    static char tmp[MAXBUFFER];

    while ((ptr = strchr(buffer, '\n')))
    {
	*ptr = '\0';
	if (*(ptr - 1) == '\r')
	    *(ptr - 1) = '\0';
	sprintf(buffer2, "%s%s", tmp, buffer);
	*tmp = '\0';
	Service_interpretQuery(buffer2);
	buffer = ptr + 1;
    }
    if (*buffer)
    {
	strcpy(tmp, buffer);
    }
}



int main(int argc, char *argv[])
{
    int nfds, i;
    struct timeval timeout;
    time_t time2write, time2notify, now;
    fd_set read_set;
    fd_set write_set;
    char buffer[MAXBUFFER];
    struct hostent *hp;
    struct sockaddr_in sairc;
    struct sockaddr_un suirc;
    int quit = 0;

    Messages_init();

    if (argc == 1)
    {
	fprintf(stderr, "Usage: %s <irc-server> <irc-port>\n       %s <unix domain socket>\n", argv[0], argv[0]);
	exit(14);
    }
    signal(SIGTRAP, SIG_IGN);
    memset(&sairc, 0, sizeof(sairc));
    if (argv[1][0] == '/' || argv[1][0] == '.')
    {
	Server_Socket = socket(AF_UNIX, SOCK_STREAM, 0);
	suirc.sun_family = AF_UNIX;
	strcpy(suirc.sun_path, argv[1]);
	if (connect(Server_Socket, (struct sockaddr *) &suirc, strlen(argv[1]) + 2) == -1)
	{
	    perror("connect");
	    exit(15);
	}
    } else
    {
	if ((Server_Socket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
	    perror("socket-irc");
	    exit(16);
	}
	if (!(hp = gethostbyname(argv[1])))
	{
	    perror("gethostbyname-irc");
	    exit(17);
	}
	memmove((char *) &sairc.sin_addr, hp->h_addr, hp->h_length);
	sairc.sin_port = htons(atoi(argv[2]));
	sairc.sin_family = AF_INET;

	if (connect(Server_Socket, (struct sockaddr *) &sairc, sizeof(sairc)) == -1)
	{
	    perror("connect");
	    exit(18);
	}
    }

    Server_sendto("PASS %s :TS\n", PASSWORD);
    Server_sendto("SERVER %s 1 :NoteService %s\n", SERVERNAME, VERSION);
    Server_sendto("NICK %s 1 1 + %s %s %s :NoteService %s\n", NICK, NICK,
                  SERVERNAME, SERVERNAME, VERSION);

    sprintf(buffer, "%s/%s", DIRECTORY, USERFILE);
    if (!(Database_FILE = gdbm_open(buffer, 512, GDBM_WRCREAT | GDBM_FAST, 0600, 0)))
    {
	perror("gdbm_open()");
	exit(19);
    }
    gdbm_reorganize(Database_FILE);
    Database_readofflineSubscribers(1); /* EXPIRE DATABASE */
    Database_readofflineSubscribers(0);


    time2write = time(NULL) + WRITETIMEOUT;
    time2notify = time(NULL);

    for (i = 0; i < 256; i++)
	List_OnlineUsers[i] = NULL;

    while (!quit)
    {
	timeout.tv_usec = 1000;
	timeout.tv_sec = 10;

	FD_ZERO(&read_set);
	FD_ZERO(&write_set);

	FD_SET(Server_Socket, &read_set);

	if ((nfds = select(FD_SETSIZE, &read_set, &write_set, NULL, &timeout)) == -1)
	{
	    perror("select");
	}
	if (FD_ISSET(Server_Socket, &read_set))
	{
	    if (!Server_receiveFrom(Server_Socket, buffer))
	    {
		printf("Connection to IRC-Server lost...\n");
		quit = 1;
	    }
	    Service_processPacket(buffer);
	}
	now = time(NULL);
	if (time2write < now)
	{
	    time2write = now + WRITETIMEOUT;
	    Database_Sync();
	}
	if (time2notify < now)
	{
	    time2notify = now + 2;
	    Subscribers_Expiration(List_Subscriber, now);
	    Subscriber_notifyaboutSignons(now);
	}
    }
    close(Server_Socket);
    Database_Sync();
    gdbm_close(Database_FILE);
    exit(0);
}

void Service_Ping(int argc, char *argv[])
{
    /* Handle inter-server PING's, used for EOB emulation and lag detection
    ** as well as client /sping's
    */
    if (argv[0][0] == ':')
    {
        if (argv[3] && argv[3][0])
          Server_sendto(":%s PONG %s :%s\n", SERVERNAME, argv[3] + 1, argv[2]);
    }
    else
    {
        Server_sendto("PONG %s\n", argv[1] + 1);
    }
}
