/*
 * tools/rsa_respond/respond.c
 * A simple RSA authentification challenge response generator for the
 * ircd-hybrid CHALLENGE command.
 *  This code is Copyright(C)2001 by the past and present ircd-hybrid
 *  developers.
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *  $Id: winrespond.c,v 1.9 2003/01/02 02:36:44 wcampbel Exp $
 */
#include <stdio.h>
#include <openssl/err.h>
#include <openssl/rsa.h>
#include <openssl/pem.h>
#include <openssl/md5.h>
#include <unistd.h>
#include <windows.h>
#include "resource.h"

#define DOC_FILE	0
#define DOC_RSA		1
#define DOC_CHAL	2
#define DOC_DEC		3
#define DOC_GOOD        4

#define BUFSIZE		512
#define REGBRANCH	"Software\\Hwy\\winrespond"

HINSTANCE hMainInstance;
HWND hMainWnd;
char g_Passphrase[BUFSIZE];
HWND TabArray[6];

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpszArgs, int nWinMode);
BOOL CALLBACK DialogFunc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam);
static UINT DoChallenge(char *, char *, char *, char *);
static void SaveRegistry(HWND hwnd);

int WINAPI WinMain(HINSTANCE hInstance,
                   HINSTANCE hPrevInstance,
                   PSTR szCmdLine,
                   int iCmdShow)
{
  int nRes;
  hMainInstance=hInstance;
  memset(g_Passphrase, '\0', BUFSIZE);
  nRes=DialogBox(hMainInstance, MAKEINTRESOURCE(IDD_DIALOG_MAIN), 0,
                 DialogFunc);
  return 0;
}

BOOL CALLBACK DialogFunc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
  switch(iMsg)
  {

    case WM_INITDIALOG:
    {
      HKEY hRegKey;
      hMainWnd=hwnd;
      SetClassLong(hwnd, GCL_HICON, (LONG)LoadIcon(hMainInstance,"winrespond"));
      if (RegOpenKey(HKEY_CURRENT_USER, REGBRANCH, &hRegKey) == ERROR_SUCCESS)
      {
        char dbuf[BUFSIZE];
        DWORD sdbuf = BUFSIZE;
        if (RegQueryValue(hRegKey, "keyfile", dbuf, &sdbuf) == ERROR_SUCCESS)
        {
          SetDlgItemText(hwnd, ID_KEY, dbuf);
        }
        RegCloseKey(hRegKey);
      }
      return TRUE;
    }

    case WM_COMMAND:
      /* WM_COMMAND messages are used for any button press */
      switch (LOWORD(wParam))
      {
        case ID_GEN:  /* Generate button */
        {
          char prvkey[BUFSIZE];
          char passphrase[BUFSIZE];
          char challenge[BUFSIZE];
          char response[BUFSIZE];

          UINT uRes = GetDlgItemText(hwnd, ID_KEY, prvkey, BUFSIZE - 1);
          if (uRes == 0)
          {
            SetDlgItemText(hwnd, ID_STATUS,
                           "Please Enter a Filename for the Key");
            return TRUE;
          }

          uRes = GetDlgItemText(hwnd, ID_CHAL, challenge, BUFSIZE - 1);
          if (uRes == 0)
          {
            SetDlgItemText(hwnd, ID_STATUS, "Please Enter a Challenge");
            return TRUE;
          }

          uRes = GetDlgItemText(hwnd, ID_PASS, passphrase, BUFSIZE - 1);
          if (uRes == 0)
          {
            SetDlgItemText(hwnd, ID_STATUS, "Using an empty passphrase");
            g_Passphrase[0] = '\0';
          }
          else
          {
            strcpy(g_Passphrase, passphrase);
          }

          uRes = DoChallenge(prvkey, passphrase, challenge, response);
          switch(uRes)
          {
            case DOC_FILE:
              SetDlgItemText(hwnd, ID_STATUS, "Please Enter a Valid Key File");
              return TRUE;

            case DOC_RSA:
              SetDlgItemText(hwnd, ID_STATUS,
                             "Unable to Read Private Key:  Passphrase?");
              return TRUE;
            case DOC_CHAL:
              SetDlgItemText(hwnd, ID_STATUS, "Bad Challenge");
              return TRUE;
            case DOC_DEC:
              SetDlgItemText(hwnd, ID_STATUS, "Decryption Error");
              return TRUE;
            case DOC_GOOD:
              SetDlgItemText(hwnd, ID_STATUS, "Response Sucessful");
              SetDlgItemText(hwnd, ID_RESP, response);
              return TRUE;
          }
          return TRUE;
        }

        case ID_OK:  /* Exit */
          SaveRegistry(hwnd);
          PostQuitMessage(0);
          return FALSE;

        default:
          return FALSE;
      }
    case WM_CLOSE: /* Catch the X or Close from the system menu */
      SaveRegistry(hwnd);
      PostQuitMessage(0);
      return FALSE;
  }
  /* Any unproccessed message in a dialog box is ignored */
  return FALSE;
}

static void SaveRegistry(HWND hwnd)
{
  HKEY hRegKey;
  char dbuf[BUFSIZE];
  UINT uRes;

  if (RegCreateKey(HKEY_CURRENT_USER, REGBRANCH, &hRegKey) == ERROR_SUCCESS)
  {
    uRes = GetDlgItemText(hwnd, ID_KEY, dbuf, BUFSIZE - 1);
    dbuf[BUFSIZE - 1] = '\0';
    if (uRes != 0)
      RegSetValue(hRegKey, "keyfile", REG_SZ, dbuf, BUFSIZE);
    else
      RegSetValue(hRegKey, "keyfile", REG_SZ, "", BUFSIZE);
    RegCloseKey(hRegKey);
  }
}

/* pass_cb is used by OpenSSL to obtain the passphrase.  On *NIX, it would
** do so by using a getpass().  In Windows, we look use what was obtained
** from the password text entry box.
*/
static int pass_cb(char *buf, int size, int rwflag, void *u)
{
	int len;
        char *tmp = g_Passphrase;
	if (tmp == NULL)
		return 0;
        len = strlen(tmp);
        if (len <= 0) 
		return 0;
        if (len > size)
        	len = size;
        memcpy(buf, tmp, len);
        return len;
}

                                                                                        
static void
binary_to_hex( unsigned char * bin, char * hex, int length )
{
	char * trans = "0123456789ABCDEF";
	int i;

	for( i = 0; i < length; i++ )
	{
		hex[i<<1]     = trans[bin[i] >> 4];
		hex[(i<<1)+1] = trans[bin[i] & 0xf];
	}
	hex[i<<1] = '\0';
}

static int
hex_to_binary(const char *from, char *to, int len)
{
	char a, b=1;
	int p=0;
	const char *ptr = from;
	while (-1)
	{
		a = *ptr++;
		if (!a)
			break;
		b = *ptr++;
		
		/* If this happens, we got bad input. */
		if (!b)
			break;
		if (p >= len)
			break;
		if (!((a >= '0' && a <= '9') || (a >= 'A' && a <= 'F')))
			break;
		if (!((b >= '0' && b <= '9') || (b >= 'A' && b <= 'F')))
			break;
		to[p++] = ((a <= '9') ? (a - '0') : (a - 'A' + 0xA))<<4 |
			((b <= '9') ? (b - '0') : (b - 'A' + 0xA));
	}
	return p;
}

/* DOC_ constants are used to return status back to the Dialog box's
** callback routine (in order to display the proper error message to the
** "status" line.
*/
static UINT DoChallenge(char *prvkey, char *passphrase, char *challenge,
                        char *response)
{
  FILE *kfile;
  RSA *rsa = NULL;
  char ndata[257], ddata[257];

  if (!(kfile = fopen(prvkey, "r")))
    return DOC_FILE;

  SSLeay_add_all_ciphers();
  rsa = PEM_read_RSAPrivateKey(kfile, NULL, pass_cb, NULL);

  if (!rsa)
  {
    fclose(kfile);
    return DOC_RSA;
  }

  if (hex_to_binary(challenge, ndata, 128) != 128)
    return DOC_CHAL;

  if (RSA_private_decrypt(128, (unsigned char*)ndata,
                (unsigned char*)ddata, rsa, RSA_PKCS1_PADDING) == -1)
    return DOC_DEC;
  binary_to_hex((unsigned char *)ddata, ndata, 32);
  strcpy(response, ndata);
  return DOC_GOOD;
}
